#define LOG_NDEBUG 0
#define LOG_TAG "InputDevice"
#define LOG_TITLE "InputEventHub"

#include "fb_event_hub.h"

static struct pollfd ufds[MAX_EVENT_COUNT];
static char device_names[MAX_EVENT_COUNT][MAX_EVENT_NAME];
static int nfds;
static int headsetfd;
static struct RawEvent* mRawEvent;
static Bool mNeedToSendFinishedDeviceScan;

static const char *get_label(const struct label *labels, int value)
{
    while(labels->name && value != labels->value) {
        labels++;
    }
    return labels->name;
}

static int print_input_props(int fd)
{
    uint8_t bits[INPUT_PROP_CNT / 8];
    int i, j;
    int res;
    int count;
    const char *bit_label;

    ALOGD("[%s] input props:\n", LOG_TITLE);
    res = ioctl(fd, EVIOCGPROP(sizeof(bits)), bits);
    if(res < 0) {
        ALOGD("[%s] <not available\n", LOG_TITLE);
        return 1;
    }
    count = 0;
    for(i = 0; i < res; i++) {
        for(j = 0; j < 8; j++) {
            if (bits[i] & 1 << j) {
                bit_label = get_label(input_prop_labels, i * 8 + j);
                if(bit_label)
                    ALOGD("[%s] %s\n", LOG_TITLE, bit_label);
                else
                    ALOGD("[%s] %04x\n", LOG_TITLE, i * 8 + j);
                count++;
            }
        }
    }
    if (!count)
        ALOGD("[%s] <none>\n", LOG_TITLE);
    return 0;
}

static int print_possible_events(int fd, int print_flags)
{
    uint8_t *bits = NULL;
    ssize_t bits_size = 0;
    const char* label;
    int i, j, k;
    int res, res2;
    struct label* bit_labels;
    const char *bit_label;

    ALOGD("[%s] events\n", LOG_TITLE);
    for(i = EV_KEY; i <= EV_MAX; i++) { // skip EV_SYN since we cannot query its available codes
        int count = 0;
        while(1) {
            res = ioctl(fd, EVIOCGBIT(i, bits_size), bits);
            if(res < bits_size)
                break;
            bits_size = res + 16;
            bits = (uint8_t *)realloc(bits, bits_size * 2);
            if(bits == NULL) {
                ALOGD("[%s] failed to allocate buffer of size %d\n", LOG_TITLE, (int)bits_size);
                return 1;
            }
        }
        res2 = 0;
        switch(i) {
            case EV_KEY:
                res2 = ioctl(fd, EVIOCGKEY(res), bits + bits_size);
                label = "KEY";
                bit_labels = key_labels;
                break;
            case EV_REL:
                label = "REL";
                bit_labels = rel_labels;
                break;
            case EV_ABS:
                label = "ABS";
                bit_labels = abs_labels;
                break;
            case EV_MSC:
                label = "MSC";
                bit_labels = msc_labels;
                break;
            case EV_LED:
                res2 = ioctl(fd, EVIOCGLED(res), bits + bits_size);
                label = "LED";
                bit_labels = led_labels;
                break;
            case EV_SND:
                res2 = ioctl(fd, EVIOCGSND(res), bits + bits_size);
                label = "SND";
                bit_labels = snd_labels;
                break;
            case EV_SW:
                res2 = ioctl(fd, EVIOCGSW(bits_size), bits + bits_size);
                label = "SW ";
                bit_labels = sw_labels;
                break;
            case EV_REP:
                label = "REP";
                bit_labels = rep_labels;
                break;
            case EV_FF:
                label = "FF ";
                bit_labels = ff_labels;
                break;
            case EV_PWR:
                label = "PWR";
                bit_labels = NULL;
                break;
            case EV_FF_STATUS:
                label = "FFS";
                bit_labels = ff_status_labels;
                break;
            default:
                res2 = 0;
                label = "???";
                bit_labels = NULL;
        }
        for(j = 0; j < res; j++) {
            for(k = 0; k < 8; k++)
                if(bits[j] & 1 << k) {
                    char down;
                    if(j < res2 && (bits[j + bits_size] & 1 << k))
                        down = '*';
                    else
                        down = ' ';
                    if(count == 0)
                        ALOGD("[%s] %s (%04x):", LOG_TITLE, label, i);
                    else if((count & (print_flags & PRINT_LABELS ? 0x3 : 0x7)) == 0 || i == EV_ABS)
                        ALOGD("[%s] \n", LOG_TITLE);
                    if(bit_labels && (print_flags & PRINT_LABELS)) {
                        bit_label = get_label(bit_labels, j * 8 + k);
                        if(bit_label)
                            ALOGD("[%s] %.20s%c%*s", LOG_TITLE, bit_label, down, 20 - strlen(bit_label), "");
                        else
                            ALOGD("[%s] %04x%c", LOG_TITLE, j * 8 + k, down);
                    } else {
                        ALOGD("[%s] %04x%c", LOG_TITLE, j * 8 + k, down);
                    }
                    if(i == EV_ABS) {
                        struct input_absinfo abs;
                        if(ioctl(fd, EVIOCGABS(j * 8 + k), &abs) == 0) {
                            ALOGD("[%s] value %d, min %d, max %d, fuzz %d, flat %d, resolution %d", LOG_TITLE, abs.value, abs.minimum, abs.maximum, abs.fuzz, abs.flat, abs.resolution);
                        }
                    }
                    count++;
                }
        }
        if(count)
            ALOGD("[%s]\n", LOG_TITLE);
    }
    free(bits);
    return 0;
}

static void print_hid_descriptor(int bus, int vendor, int product)
{
    const char *dirname = "/sys/kernel/debug/hid";
    char prefix[16];
    DIR *dir;
    struct dirent *de;
    char filename[PATH_MAX];
    FILE *file;
    char line[2048];

    snprintf(prefix, sizeof(prefix), "%04X:%04X:%04X.", bus, vendor, product);

    dir = opendir(dirname);
    if(dir == NULL)
        return;
    while((de = readdir(dir))) {
        if (strstr(de->d_name, prefix) == de->d_name) {
            snprintf(filename, sizeof(filename), "%s/%s/rdesc", dirname, de->d_name);

            file = fopen(filename, "r");
            if (file) {
                ALOGD("[%s] HID descriptor: %s\n\n", LOG_TITLE, de->d_name);
                while (fgets(line, sizeof(line), file)) {
                    fputs("    ", stdout);
                    fputs(line, stdout);
                }
                fclose(file);
                puts("");
            }
        }
    }
    closedir(dir);
}

static Bool containsNonZeroByte(const uint8_t* array, uint32_t startIndex, uint32_t endIndex) {
    const uint8_t* end = array + endIndex;
    array += startIndex;
    while (array != end) {
        if (*(array++) != 0) {
            return true;
        }
    }
    return false;
}

static uint32_t getAbsAxisUsage(int32_t axis, uint32_t deviceClasses) {
    // Touch devices get dibs on touch-related axes.
    if (deviceClasses & INPUT_DEVICE_CLASS_TOUCH) {
        switch (axis) {
            case ABS_X:
            case ABS_Y:
            case ABS_PRESSURE:
            case ABS_TOOL_WIDTH:
            case ABS_DISTANCE:
            case ABS_TILT_X:
            case ABS_TILT_Y:
            case ABS_MT_SLOT:
            case ABS_MT_TOUCH_MAJOR:
            case ABS_MT_TOUCH_MINOR:
            case ABS_MT_WIDTH_MAJOR:
            case ABS_MT_WIDTH_MINOR:
            case ABS_MT_ORIENTATION:
            case ABS_MT_POSITION_X:
            case ABS_MT_POSITION_Y:
            case ABS_MT_TOOL_TYPE:
            case ABS_MT_BLOB_ID:
            case ABS_MT_TRACKING_ID:
            case ABS_MT_PRESSURE:
            case ABS_MT_DISTANCE:
                return INPUT_DEVICE_CLASS_TOUCH;
        }
    }

    // Joystick devices get the rest.
    return deviceClasses & INPUT_DEVICE_CLASS_JOYSTICK;
}


static int open_device(const char *device, int print_flags)
{
    int version;
    int fd, i;
    char name[80];
    char location[80];
    char idstr[80];
    uint32_t classes = 0;

    Bool haveKeyboardKeys;
    Bool haveGamepadButtons;

    uint8_t keyBitmask[(KEY_MAX + 1) / 8];
    uint8_t absBitmask[(ABS_MAX + 1) / 8];
    uint8_t relBitmask[(REL_MAX + 1) / 8];
    uint8_t swBitmask[(SW_MAX + 1) / 8];
    uint8_t ledBitmask[(LED_MAX + 1) / 8];
    uint8_t ffBitmask[(FF_MAX + 1) / 8];
    uint8_t propBitmask[(INPUT_PROP_MAX + 1) / 8];

    struct input_id id;

    fd = open(device, O_RDWR);
    if(fd < 0) {
        // get interrupt system call so retry
        if(errno == EINTR) {
            usleep(100*1000);
            fd = open(device, O_RDWR);
        }

        if(fd < 0) {
            if(print_flags & PRINT_DEVICE_ERRORS)
                ALOGD("[%s] could not open %s, %s\n", LOG_TITLE, device, strerror(errno));
            return -1;
        }
    }

    if(ioctl(fd, EVIOCGVERSION, &version)) {
        if(print_flags & PRINT_DEVICE_ERRORS)
            ALOGD("[%s] could not get driver version for %s, %s\n", LOG_TITLE, device, strerror(errno));
        close(fd);
        return -1;
    }
    if(ioctl(fd, EVIOCGID, &id)) {
        if(print_flags & PRINT_DEVICE_ERRORS)
            ALOGD("[%s] could not get driver id for %s, %s\n", LOG_TITLE, device, strerror(errno));
        close(fd);
        return -1;
    }
    name[sizeof(name) - 1] = '\0';
    location[sizeof(location) - 1] = '\0';
    idstr[sizeof(idstr) - 1] = '\0';
    if(ioctl(fd, EVIOCGNAME(sizeof(name) - 1), &name) < 1) {
        ALOGD("[%s]could not get device name for %s, %s\n", LOG_TITLE, device, strerror(errno));
        name[0] = '\0';
    }

#if 0
    if(ioctl(fd, EVIOCGPHYS(sizeof(location) - 1), &location) < 1) {
        ALOGD("[%s] could not get location for %s, %s\n", LOG_TITLE, device, strerror(errno));
        location[0] = '\0';
    }
    if(ioctl(fd, EVIOCGUNIQ(sizeof(idstr) - 1), &idstr) < 1) {
        ALOGD("[%s] could not get idstring for %s, %s\n", LOG_TITLE, device, strerror(errno));
        idstr[0] = '\0';
    }
#endif

    // Figure out the kinds of events the device reports.
    ioctl(fd, EVIOCGBIT(EV_KEY, sizeof(keyBitmask)), keyBitmask);
    ioctl(fd, EVIOCGBIT(EV_ABS, sizeof(absBitmask)), absBitmask);
    ioctl(fd, EVIOCGBIT(EV_REL, sizeof(relBitmask)), relBitmask);
    ioctl(fd, EVIOCGBIT(EV_SW, sizeof(swBitmask)), swBitmask);
    ioctl(fd, EVIOCGBIT(EV_LED, sizeof(ledBitmask)), ledBitmask);
    ioctl(fd, EVIOCGBIT(EV_FF, sizeof(ffBitmask)), ffBitmask);
    ioctl(fd, EVIOCGPROP(sizeof(propBitmask)), propBitmask);

    // See if this is a keyboard.  Ignore everything in the button range except for
    // joystick and gamepad buttons which are handled like keyboards for the most part.
    haveKeyboardKeys = containsNonZeroByte(keyBitmask, 0, sizeof_bit_array(BTN_MISC))
            || containsNonZeroByte(keyBitmask, sizeof_bit_array(KEY_OK),
                    sizeof_bit_array(KEY_MAX + 1));
    haveGamepadButtons = containsNonZeroByte(keyBitmask, sizeof_bit_array(BTN_MISC),
                    sizeof_bit_array(BTN_MOUSE))
            || containsNonZeroByte(keyBitmask, sizeof_bit_array(BTN_JOYSTICK),
                    sizeof_bit_array(BTN_DIGI));
    if (haveKeyboardKeys || haveGamepadButtons) {
        classes |= INPUT_DEVICE_CLASS_KEYBOARD;
    }

    // See if this is a cursor device such as a trackball or mouse.
    if (test_bit(BTN_MOUSE, keyBitmask) && test_bit(REL_X, relBitmask) && test_bit(REL_Y, relBitmask)) {
        classes |= INPUT_DEVICE_CLASS_CURSOR;
    }

    // See if this is a touch pad.
    // Is this a new modern multi-touch driver?
    if (test_bit(ABS_MT_POSITION_X, absBitmask) && test_bit(ABS_MT_POSITION_Y, absBitmask)) {
        // Some joysticks such as the PS3 controller report axes that conflict
        // with the ABS_MT range.  Try to confirm that the device really is a touch screen.
        if (test_bit(BTN_TOUCH, keyBitmask) || !haveGamepadButtons) {
            classes |= INPUT_DEVICE_CLASS_TOUCH | INPUT_DEVICE_CLASS_TOUCH_MT;
        }
    }// Is this an old style single-touch driver?
    else if (test_bit(BTN_TOUCH, keyBitmask) && test_bit(ABS_X, absBitmask) && test_bit(ABS_Y, absBitmask)) {
        classes |= INPUT_DEVICE_CLASS_TOUCH;
    }

    // See if this device is a joystick.
    // Assumes that joysticks always have gamepad buttons in order to distinguish them
    // from other devices such as accelerometers that also have absolute axes.
    if (haveGamepadButtons) {
        uint32_t assumedClasses = classes | INPUT_DEVICE_CLASS_JOYSTICK;
        for (i = 0; i <= ABS_MAX; i++) {
            if (test_bit(i, absBitmask) && (getAbsAxisUsage(i, assumedClasses) & INPUT_DEVICE_CLASS_JOYSTICK)) {
                classes = assumedClasses;
                break;
            }
        }
    }

    // Check whether this device has switches.
    for (i = 0; i <= SW_MAX; i++) {
        if (test_bit(i, swBitmask)) {
            classes |= INPUT_DEVICE_CLASS_SWITCH;
            headsetfd = fd;
            break;
        }
    }

    // Check whether this device supports the vibrator.
    if (test_bit(FF_RUMBLE, ffBitmask)) {
        classes |= INPUT_DEVICE_CLASS_VIBRATOR;
    }

    if(nfds >= MAX_EVENT_COUNT) {
        ALOGD("[%s] out of memory\n", LOG_TITLE);
        close(fd);
        return -1;
    }

    if(print_flags & PRINT_DEVICE)
        ALOGD("[%s] add device %d: %s  class: %d\n", LOG_TITLE, nfds, device, classes);
    if(print_flags & PRINT_DEVICE_INFO)
        ALOGD("[%s] bus: %04x  vendor: %04x  product: %04x  version: %04x\n", LOG_TITLE, id.bustype, id.vendor, id.product, id.version);
    if(print_flags & PRINT_DEVICE_NAME)
        ALOGD("[%s] name: \"%s\"\n", LOG_TITLE, name);
#if 0
    if(print_flags & PRINT_DEVICE_INFO)
        ALOGD("[%s] location: \"%s\"  id: \"%s\"\n", LOG_TITLE, location, idstr);
#endif
    if(print_flags & PRINT_VERSION)
        ALOGD("[%s] version: %d.%d.%d\n", LOG_TITLE, version >> 16, (version >> 8) & 0xff, version & 0xff);

    if(print_flags & PRINT_POSSIBLE_EVENTS) {
        print_possible_events(fd, print_flags);
    }

    if(print_flags & PRINT_INPUT_PROPS) {
        print_input_props(fd);
    }
    if(print_flags & PRINT_HID_DESCRIPTOR) {
        print_hid_descriptor(id.bustype, id.vendor, id.product);
    }

    ufds[nfds].fd = fd;
    ufds[nfds].events = POLLIN;
    strncpy(device_names[nfds], device, (sizeof(char) * MAX_EVENT_NAME));
    nfds++;

    mRawEvent->deviceId = (nfds - 1);
    mRawEvent->type = DEVICE_ADDED;
    mRawEvent->classes = classes;
    ALOGD("[%s] event[%d]:%s  type:0x%08x  classes:0x%04x\n", LOG_TITLE, mRawEvent->deviceId, device_names[mRawEvent->deviceId], mRawEvent->type, mRawEvent->classes);
    mRawEvent += 1;

    mNeedToSendFinishedDeviceScan = true;

    return 0;
}

int close_device(const char *device, int print_flags)
{
    int i;
    for(i = 1; i < nfds; i++) {
        if(strcmp(device_names[i], device) == 0) {
            int count = nfds - i - 1;
            if(print_flags & PRINT_DEVICE)
                ALOGD("[%s] remove device %d: %s\n", LOG_TITLE, i, device);

            memset((void *)&device_names[i], 0x00, (sizeof(char) * MAX_EVENT_NAME));
            memcpy((void *)&device_names[i], (void *)&device_names[i+1], (sizeof(char) * MAX_EVENT_NAME * count));
            memset((void *)&device_names[nfds -1], 0x00, (sizeof(char) * MAX_EVENT_NAME));

            close(ufds[i].fd);
            memset((void *)&ufds[i], 0x00, sizeof(struct pollfd));
            memcpy((void *)&ufds[i], (void *)&ufds[i+1], (sizeof(struct pollfd) * count));
            memset((void *)&ufds[nfds -1], 0x00, sizeof(struct pollfd));
            nfds--;

            mRawEvent->deviceId = i;
            mRawEvent->type = DEVICE_REMOVED;
            mRawEvent += 1;

            mNeedToSendFinishedDeviceScan = true;

            return 0;
        }
    }
    if(print_flags & PRINT_DEVICE_ERRORS) {
        ALOGD("[%s] remote device: %s not found\n", LOG_TITLE, device);
    }

    return -1;
}

static int read_notify(const char *dirname, int nfd, int print_flags)
{
    int res;
    char devname[PATH_MAX];
    char *filename;
    char event_buf[512];
    int event_size;
    int event_pos = 0;
    struct inotify_event *event;

    res = read(nfd, event_buf, sizeof(event_buf));
    if(res < (int)sizeof(*event)) {
        if(errno == EINTR)
            return 0;
        ALOGD("[%s] could not get event, %s\n", LOG_TITLE, strerror(errno));
        return 1;
    }
    //ALOGD("[%s] got %d bytes of event information\n", LOG_TITLE, res);

    strcpy(devname, dirname);
    filename = devname + strlen(devname);
    *filename++ = '/';

    while(res >= (int)sizeof(*event)) {
        event = (struct inotify_event *)(event_buf + event_pos);
        //ALOGD("[%s] %d: %08x \"%s\"\n", LOG_TITLE, event->wd, event->mask, event->len ? event->name : "");
        if(event->len) {
            strcpy(filename, event->name);
            if(event->mask & IN_CREATE) {
                open_device(devname, print_flags);
            }
            else {
                close_device(devname, print_flags);
            }
        }
        event_size = sizeof(*event) + event->len;
        res -= event_size;
        event_pos += event_size;
    }

    return 0;
}

static int scan_dir(const char *dirname, int print_flags)
{
    char devname[PATH_MAX];
    char *filename;
    DIR *dir;
    struct dirent *de;
    dir = opendir(dirname);
    if(dir == NULL)
        return -1;
    strcpy(devname, dirname);
    filename = devname + strlen(devname);
    *filename++ = '/';
    while((de = readdir(dir))) {
        if(de->d_name[0] == '.' &&
           (de->d_name[1] == '\0' ||
            (de->d_name[1] == '.' && de->d_name[2] == '\0')))
            continue;
        strcpy(filename, de->d_name);
        open_device(devname, print_flags);
    }

    closedir(dir);

    return 0;
}

int FB_GetEvent_Init(struct RawEvent* buffer)
{
    int res;
    int print_flags = 0;

    const char *device_path = "/dev/input";

    mRawEvent = buffer;
    mNeedToSendFinishedDeviceScan = false;

    nfds = 1;
    memset((void *)ufds, 0x00, (sizeof(struct pollfd) * MAX_EVENT_COUNT));
    memset((void *)device_names, 0x00, (sizeof(char) * MAX_EVENT_COUNT * MAX_EVENT_NAME));

    ufds[0].fd = inotify_init();
    ufds[0].events = POLLIN;

    print_flags |= PRINT_DEVICE_ERRORS | PRINT_DEVICE | PRINT_DEVICE_NAME | PRINT_DEVICE_INFO | PRINT_POSSIBLE_EVENTS;
    res = inotify_add_watch(ufds[0].fd, device_path, IN_DELETE | IN_CREATE);
    if(res < 0) {
        ALOGD("[%s] could not add watch for %s, %s\n", LOG_TITLE, device_path, strerror(errno));
        return -1;
    }
    res = scan_dir(device_path, print_flags);
    if(res < 0) {
        ALOGD("[%s] scan dir failed for %s\n", LOG_TITLE, device_path);
        return -1;
    }

    if(mNeedToSendFinishedDeviceScan)
    {
        mRawEvent->type = FINISHED_DEVICE_SCAN;
        ALOGD("[%s] finish scan type:0x%08x\n", LOG_TITLE, mRawEvent->type);
        mRawEvent += 1;
    }

    return (mRawEvent - buffer);
}

int FB_GetEvent_Handle(struct RawEvent* buffer)
{
    int i = 0;
    int res = 0;
    int pollres = 0;
    int print_flags = 0;

    struct input_event event;

    const char *device_path = "/dev/input";

    mRawEvent = buffer;
    mNeedToSendFinishedDeviceScan = false;

    print_flags |= PRINT_DEVICE_ERRORS | PRINT_DEVICE | PRINT_DEVICE_NAME;

    do {
        pollres = poll((struct pollfd *)&ufds[0], nfds, -1);

        if(ufds[0].revents & POLLIN) {
            read_notify(device_path, ufds[0].fd, print_flags);
        }

        for(i = 1; i < nfds; i++) {
            if(ufds[i].revents) {
                if(ufds[i].revents & POLLIN) {
                    res = read(ufds[i].fd, &event, sizeof(event));
                    if(res < (int)sizeof(event)) {
                        ALOGD("[%s] could not get event %s\n", LOG_TITLE, device_names[i]);
                        return -1;
                    }

                    mRawEvent->deviceId = i;
                    mRawEvent->type = event.type;
                    mRawEvent->code = event.code;
                    mRawEvent->value = event.value;
                    mRawEvent += 1;

                    //ALOGD("[%s] event[%d]:%s  type:0x%08x  code:0x%04x  value:0x%x\n", LOG_TITLE, i, device_names[i], event.type, event.code, event.value);
                }
            }
        }
    } while (0);

    if(mNeedToSendFinishedDeviceScan)
    {
        mRawEvent->type = FINISHED_DEVICE_SCAN;
        ALOGD("[%s] finish scan type:0x%08x\n", LOG_TITLE, mRawEvent->type);
        mRawEvent += 1;
    }

    return (mRawEvent - buffer);
}

int getAbsoluteAxisInfo(int deviceId, int axis)
{
    struct input_absinfo info;
    if(ioctl(ufds[deviceId].fd, EVIOCGABS(axis), &info)) {
        ALOGD("Error reading absolute controller %d for device id %d fd %d, errno=%d", LOG_TITLE, axis, deviceId, ufds[deviceId].fd, errno);
        return -errno;
    }

    return info.maximum;
}

int FB_GetHeadsetStatus(int32_t deviceId, int32_t sw)
{
    if(sw >= 0 && sw <= SW_MAX) {
        uint8_t swState[sizeof_bit_array(SW_MAX + 1)];
        memset(swState, 0, sizeof(swState));
        if (ioctl(headsetfd, EVIOCGSW(sizeof(swState)), swState) >= 0) {
            return test_bit(sw, swState) ? 1 : 0;
        }
    }
    return -1;
}

int FB_CloseDevice(int device_id)
{
    int print_flag = PRINT_DEVICE | PRINT_DEVICE_ERRORS;

    if(nfds <= 0) {
        return 0;
    }

    if(device_id == 0) {
        int count = nfds - device_id - 1;
        if(print_flag & PRINT_DEVICE)
            ALOGD("[%s] remove device %d: %s\n", LOG_TITLE, device_id, "/dev/input");
        close(ufds[device_id].fd);
        nfds--;
        return 0;
    }
    else {
        return close_device(device_names[device_id], print_flag);
    }
}
