/*
 * libjingle
 * Copyright 2004--2010, Google Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "talk/base/ratetracker.h"
#include "talk/base/time.h"

namespace talk_base {

RateTracker::RateTracker()
    : total_units_(0), units_second_(0),
      last_units_second_time_(static_cast<uint32>(-1)),
      last_units_second_calc_(0) {
}

size_t RateTracker::total_units() const {
  return total_units_;
}

size_t RateTracker::units_second() {
  // Snapshot units / second calculator. Determine how many seconds have
  // elapsed since our last reference point. If over 1 second, establish
  // a new reference point that is an integer number of seconds since the
  // last one, and compute the units over that interval.
  uint32 current_time = Time();
  if (last_units_second_time_ != static_cast<uint32>(-1)) {
    int delta = talk_base::TimeDiff(current_time, last_units_second_time_);
    if (delta >= 1000) {
      int fraction_time = delta % 1000;
      int seconds = delta / 1000;
      int fraction_units =
          static_cast<int>(total_units_ - last_units_second_calc_) *
              fraction_time / delta;
      // Compute "units received during the interval" / "seconds in interval"
      units_second_ =
          (total_units_ - last_units_second_calc_ - fraction_units) / seconds;
      last_units_second_time_ = current_time - fraction_time;
      last_units_second_calc_ = total_units_ - fraction_units;
    }
  }
  if (last_units_second_time_ == static_cast<uint32>(-1)) {
    last_units_second_time_ = current_time;
    last_units_second_calc_ = total_units_;
  }

  return units_second_;
}

void RateTracker::Update(size_t units) {
  total_units_ += units;
}

uint32 RateTracker::Time() const {
  return talk_base::Time();
}

}  // namespace talk_base
