/*
 * =================================================================
 *
 *
 *	Description:  samsung display panel file
 *
 *	Author: jb09.kim
 *	Company:  Samsung Electronics
 *
 * ================================================================
 */
/*
<one line to give the program's name and a brief idea of what it does.>
Copyright (C) 2012, Samsung Electronics. All rights reserved.

*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
*/
#include "ss_dsi_panel_S6E3HF4_AMB526JS01.h"
#include "ss_dsi_mdnie_S6E3HF4_AMB526JS01.h"
#include "../../../mdss/mdss_dsi.h"
#include <linux/sec_param.h>

/* AOD Mode status on AOD Service */
enum {
	AOD_MODE_ALPM_2NIT_ON = MAX_LPM_MODE + 1,
	AOD_MODE_HLPM_2NIT_ON,
	AOD_MODE_ALPM_40NIT_ON,
	AOD_MODE_HLPM_40NIT_ON,
};

enum {
	ALPM_CTRL_2NIT,
	ALPM_CTRL_40NIT,
	HLPM_CTRL_2NIT,
	HLPM_CTRL_40NIT,
	MAX_LPM_CTRL,
};

/*#define OTHER_LINE_PANEL_SUPPORT_DEBUG*/

#define BASIC_DATA_ELEMENT_CNT 5
#define REF_SHIFT_ELEMENT_CNT 10
#define COLOR_SHIFT_ELEMENT_CNT 28
#define ELVSS_LOW_TEMP_ELEMENT_CNT 6
#define ELVSS_DATA_CNT 2
#define VINT_ELEMENT_CNT 11
#define AOR_INTERPOLATION_ELEMENT_CNT 4
#define IRC_ELEMENT_CNT 21
#define HBM_ELVSS_ELEMENT_CNT 9

struct OtherLine_ParseDataList a2_line_data;

/* Register to control brightness level */
#define ALPM_REG		0x53
/* Register to cnotrol ALPM/HLPM mode */
#define ALPM_CTRL_REG	0xBB

static int hero2_fab = S6E3HF4_AMB526JS01_A3_LINE;
static int hero2_a2_revM = 0;

/*cmd_idx(of vdd->bl_level)is  mapping
	to color weakness platform bl level :
	it should be changed according the
	data sheet from display lab.*/

#define LUMINANCE_MAX 74

static int init_ldi_fps(struct mdss_dsi_ctrl_pdata *ctrl);

static void mdss_init_hero2_fab_type(struct samsung_display_driver_data *vdd)
{
	/*
	0xDA register's D7, D6 bit
		a3 01
		a2 00
	*/

	int id0, id1, id2;

	if (get_lcd_attached("GET") == PBA_ID)
		return;

	id0 = (get_lcd_attached("GET") & 0xFF0000) >> 16;
	id1 = (get_lcd_attached("GET") & 0xFF00) >> 8;
	id2 = (get_lcd_attached("GET") & 0xFF);

	LCD_ERR("%s : %2x %2x %2x\n", __func__, id0, id1, id2);

	/*
	 * HERO2
	 * The 0x10, 0x11 use different EL Material
	 */

	if (id1 & 0x10){ /*hero2*/
		if (id0 >> 6 == S6E3HF4_AMB526JS01_A2_LINE){
			vdd->aid_subdivision_enable = true;
			vdd->panel_revision = LATEST_PANLE_REV - 'A'; /* a2's panel version has to same with a3 latest panel version: need to update */

			LCD_INFO("panel_revision = %c %d \n",
					vdd->panel_revision + 'A', vdd->panel_revision);
		}else
			hero2_fab = S6E3HF4_AMB526JS01_A3_LINE;

	LCD_ERR("projcet fab type is %s\n", (hero2_fab == S6E3HF4_AMB526JS01_A2_LINE) ? "hero2_a2" : "hero2_a3");
	}
}

int  mdss_get_hero2_fab_type(void)
{
	return hero2_fab;
}

static void  mdss_set_hero2_fab_type(int type)
{
	LCD_DEBUG("hero2 fab type (%d)\n", type);
	hero2_fab = type;
}

static int mdss_panel_on_pre(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	LCD_INFO("+: ndx=%d \n", ctrl->ndx);
	mdss_panel_attach_set(ctrl, true);
	LCD_INFO("-: ndx=%d \n", ctrl->ndx);

	init_ldi_fps(ctrl);

	return true;
}

static int mdss_panel_on_post(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	return true;
}

static int mdss_panel_revision(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	if (vdd->manufacture_id_dsi[ctrl->ndx] == PBA_ID)
		mdss_panel_attach_set(ctrl, false);
	else
		mdss_panel_attach_set(ctrl, true);

	/*
	 * HERO2
	 */
	vdd->aid_subdivision_enable = true;

	if (mdss_panel_rev_get(ctrl) == 0x00)
		vdd->panel_revision = 'A'; /* REV A */
	else if (mdss_panel_rev_get(ctrl) == 0x01)
		vdd->panel_revision = 'B'; /* REV B */
	else if (mdss_panel_rev_get(ctrl) == 0x02)
		vdd->panel_revision = 'C'; /* REV C */
	else if (mdss_panel_rev_get(ctrl) == 0x03)
		vdd->panel_revision = 'D'; /* REV D E */
	else if (mdss_panel_rev_get(ctrl) == 0x04)
		vdd->panel_revision = 'F'; /* REV F */
	else if (mdss_panel_rev_get(ctrl) == 0x05)
		vdd->panel_revision = 'G'; /* REV G */
	else if (mdss_panel_rev_get(ctrl) == 0x06)
		vdd->panel_revision = 'H'; /* REV H I */
	else if ((mdss_panel_rev_get(ctrl) == 0x07) ||
			(mdss_panel_rev_get(ctrl) == 0x08)) {
		vdd->panel_revision = 'J'; /* REV J K */
	} else
		vdd->panel_revision = 'J';

	/* In the a2 line panel case, panel_revision should be set latest one. */
	if (mdss_get_hero2_fab_type() == S6E3HF4_AMB526JS01_A2_LINE)
		vdd->panel_revision = LATEST_PANLE_REV;

	if (((mdss_panel_id0_get(ctrl) & 0x40) == 0x00) &&
		(mdss_panel_rev_get(ctrl) == 0x08)) {
		vdd->panel_revision = LATEST_PANLE_REV;
		hero2_a2_revM = 1;
		LCD_ERR("hero2_a2_revM (%d)\n", hero2_a2_revM);
	}

	vdd->panel_revision -= 'A';

	LCD_DEBUG("panel_revision = %c %d \n",
					vdd->panel_revision + 'A', vdd->panel_revision);

	return true;
}

static int mdss_manufacture_date_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	unsigned char date[4];
	int year, month, day;
	int hour, min;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (C8h 41,42th) for manufacture date */
	if (vdd->dtsi_data[ctrl->ndx].manufacture_date_rx_cmds[vdd->panel_revision].cmd_cnt) {
		mdss_samsung_panel_data_read(ctrl,
			&vdd->dtsi_data[ctrl->ndx].manufacture_date_rx_cmds[vdd->panel_revision],
			date, PANEL_LEVE1_KEY);

		year = date[0] & 0xf0;
		year >>= 4;
		year += 2011; // 0 = 2011 year
		month = date[0] & 0x0f;
		day = date[1] & 0x1f;
		hour = date[2]& 0x0f;
		min = date[3] & 0x1f;

		vdd->manufacture_date_dsi[ctrl->ndx] = year * 10000 + month * 100 + day;
		vdd->manufacture_time_dsi[ctrl->ndx] = hour * 100 + min;

		LCD_ERR("manufacture_date DSI%d = (%d%04d) - year(%d) month(%d) day(%d) hour(%d) min(%d)\n",
			ctrl->ndx, vdd->manufacture_date_dsi[ctrl->ndx], vdd->manufacture_time_dsi[ctrl->ndx],
			year, month, day, hour, min);

	} else {
		LCD_ERR("DSI%d no manufacture_date_rx_cmds cmds(%d)",  ctrl->ndx,vdd->panel_revision);
		return false;
	}

	return true;
}

static int mdss_ddi_id_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char ddi_id[5];
	int loop;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (D6h 1~5th) for ddi id */
	if (vdd->dtsi_data[ctrl->ndx].ddi_id_rx_cmds[vdd->panel_revision].cmd_cnt) {
		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].ddi_id_rx_cmds[vdd->panel_revision]),
			ddi_id, PANEL_LEVE1_KEY);

		for(loop = 0; loop < 5; loop++)
			vdd->ddi_id_dsi[ctrl->ndx][loop] = ddi_id[loop];

		LCD_INFO("DSI%d : %02x %02x %02x %02x %02x\n", ctrl->ndx,
			vdd->ddi_id_dsi[ctrl->ndx][0], vdd->ddi_id_dsi[ctrl->ndx][1],
			vdd->ddi_id_dsi[ctrl->ndx][2], vdd->ddi_id_dsi[ctrl->ndx][3],
			vdd->ddi_id_dsi[ctrl->ndx][4]);
	} else {
		LCD_ERR("DSI%d no ddi_id_rx_cmds cmds", ctrl->ndx);
		return false;
	}

	return true;
}

static int get_hbm_candela_value(int level)
{
	if (level == 13)
		return 443;
	else if (level == 6)
		return 465;
	else if (level == 7)
		return 488;
	else if (level == 8)
		return 510;
	else if (level == 9)
		return 533;
	else if (level == 10)
		return 555;
	else if (level == 11)
		return 578;
	else if (level == 12)
		return 600;
	else
		return 600;
}

static struct dsi_panel_cmds *mdss_hbm_gamma(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi[ctrl->ndx]->generate_hbm_gamma)) {
		LCD_ERR("generate_hbm_gamma is NULL error");
		return NULL;
	} else {
		if (mdss_panel_id1_get(ctrl) & 0x10) { /* hero 2 */
			vdd->smart_dimming_dsi[ctrl->ndx]->generate_hbm_gamma(
				vdd->smart_dimming_dsi[ctrl->ndx],
				vdd->auto_brightness,
				&vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1]);

			*level_key = PANEL_LEVE1_KEY;

			return &vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision];
		} else {
			if (vdd->auto_brightness > 6)
				return NULL;
			return &vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision];
		}
	}
}

static struct dsi_panel_cmds *mdss_hbm_etc(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	char elvss_3th_val, elvss_23th_val;
	char elvss_443_offset, elvss_465_offset, elvss_488_offset, elvss_510_offset, elvss_533_offset;
	char elvss_555_offset, elvss_578_offset, elvss_600_offset, elvss_23th_offset;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	elvss_3th_val = elvss_23th_val = 0;

	/* OTP value - B5 23th */
	elvss_23th_val = vdd->display_status_dsi[ctrl->ndx].elvss_value1;

	LCD_DEBUG("OTP val %x\n", elvss_23th_val);

	if (mdss_get_hero2_fab_type() == S6E3HF4_AMB526JS01_A2_LINE){

		/* ELVSS 0xB5 3th*/
			elvss_443_offset = a2_line_data.hbm_elvss[0];
			elvss_465_offset = a2_line_data.hbm_elvss[1];
			elvss_488_offset = a2_line_data.hbm_elvss[2];
			elvss_510_offset = a2_line_data.hbm_elvss[3];
			elvss_533_offset = a2_line_data.hbm_elvss[4];
			elvss_555_offset = a2_line_data.hbm_elvss[5];
			elvss_578_offset = a2_line_data.hbm_elvss[6];
			elvss_600_offset = a2_line_data.hbm_elvss[7];
		/* ELVSS 0xB5 23th*/
			elvss_23th_offset = a2_line_data.elvss_offset[2][4];
	}else{
		/* ELVSS 0xB5 3th*/
			elvss_443_offset = 0x09;
			elvss_465_offset = 0x07;
			elvss_488_offset = 0x06;
			elvss_510_offset = 0x05;
			elvss_533_offset = 0x04;
			elvss_555_offset = 0x02;
			elvss_578_offset = 0x01;
			elvss_600_offset = 0x00;
		/* ELVSS 0xB5 23th*/
			elvss_23th_offset = -0x04;
	}

	/* ELVSS 0xB5 3th*/
	if (vdd->auto_brightness == HBM_MODE) /* 465CD */
		elvss_3th_val = elvss_465_offset;
	else if (vdd->auto_brightness == HBM_MODE + 1) /* 488CD */
		elvss_3th_val = elvss_488_offset;
	else if (vdd->auto_brightness == HBM_MODE + 2) /* 510CD */
		elvss_3th_val = elvss_510_offset;
	else if (vdd->auto_brightness == HBM_MODE + 3) /* 533CD */
		elvss_3th_val = elvss_533_offset;
	else if (vdd->auto_brightness == HBM_MODE + 4) /* 555CD */
		elvss_3th_val= elvss_555_offset;
	else if (vdd->auto_brightness == HBM_MODE + 5) /* 578CD */
		elvss_3th_val = elvss_578_offset;
	else if (vdd->auto_brightness == HBM_MODE + 6)/* 600CD */
		elvss_3th_val = elvss_600_offset;
	else if (vdd->auto_brightness == HBM_MODE + 7) /* 443CD */
		elvss_3th_val = elvss_443_offset;

	/* ELVSS 0xB5 23th*/
	if (vdd->temperature <= vdd->elvss_interpolation_temperature)
		elvss_23th_val += elvss_23th_offset;

	/* 0xB5 1th temperature */
	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[1].payload[1] =
			vdd->temperature > 0 ? vdd->temperature : 0x80|(-1*vdd->temperature);

	/* ELVSS 0xB5 3th, 23th */
	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[1].payload[3] = elvss_3th_val;
	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[3].payload[1] = elvss_23th_val ;

	/* ACL 15% in LDU/CCB mode */
	if (vdd->color_weakness_mode || vdd->ldu_correction_state)
		vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[5].payload[4] = 0x12; /* 0x12 : 15% */
	else
		vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[5].payload[4] = 0x0A; /* 0x0A : 8% */

	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[6].payload[1] = 0x02; /* ALC ON */

	*level_key = PANEL_LEVE1_KEY;

	LCD_DEBUG("0xB5 3th: 0x%x 0xB5 23th(elvss val) 0x%x\n", elvss_3th_val, elvss_23th_val);

	return &vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision];
}

static int mdss_elvss_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char elvss_b5_23rd;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (B5h 23th) for elvss*/
	mdss_samsung_panel_data_read(ctrl,
		&(vdd->dtsi_data[ctrl->ndx].elvss_rx_cmds[vdd->panel_revision]),
		&elvss_b5_23rd, PANEL_LEVE1_KEY);
	vdd->display_status_dsi[ctrl->ndx].elvss_value1 = elvss_b5_23rd; /* OTP value */

	return true;
}

static int mdss_hbm_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char hbm_buffer1[15];
	char hbm_buffer2[20];

	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (C8h 51~65th) for hbm gamma */
	mdss_samsung_panel_data_read(ctrl,
		&(vdd->dtsi_data[ctrl->ndx].hbm_rx_cmds[vdd->panel_revision]),
		hbm_buffer1, PANEL_LEVE1_KEY);

	/* Read mtp (C8h 69~88th) for hbm gamma */
	mdss_samsung_panel_data_read(ctrl,
		&(vdd->dtsi_data[ctrl->ndx].hbm2_rx_cmds[vdd->panel_revision]),
		hbm_buffer2, PANEL_LEVE1_KEY);

	memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1], hbm_buffer1, 15);
	memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[16], hbm_buffer2, 20);

	return true;
}

#define COORDINATE_DATA_SIZE 6
#define MDNIE_SCR_WR_ADDR 	0x32

#define F1(x,y) ((y)-((43*(x))/40)+45)
#define F2(x,y) ((y)-((310*(x))/297)-3)
#define F3(x,y) ((y)+((367*(x))/84)-16305)
#define F4(x,y) ((y)+((333*(x))/107)-12396)

static char coordinate_data_1[][COORDINATE_DATA_SIZE] = {
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00}, /* dummy */
	{0xff, 0x00, 0xfb, 0x00, 0xfb, 0x00}, /* Tune_1 */
	{0xff, 0x00, 0xfd, 0x00, 0xff, 0x00}, /* Tune_2 */
	{0xfb, 0x00, 0xfa, 0x00, 0xff, 0x00}, /* Tune_3 */
	{0xff, 0x00, 0xfe, 0x00, 0xfc, 0x00}, /* Tune_4 */
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00}, /* Tune_5 */
	{0xfb, 0x00, 0xfc, 0x00, 0xff, 0x00}, /* Tune_6 */
	{0xfd, 0x00, 0xff, 0x00, 0xfa, 0x00}, /* Tune_7 */
	{0xfd, 0x00, 0xff, 0x00, 0xfd, 0x00}, /* Tune_8 */
	{0xfb, 0x00, 0xff, 0x00, 0xff, 0x00}, /* Tune_9 */
};

static char coordinate_data_2[][COORDINATE_DATA_SIZE] = {
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00}, /* dummy */
	{0xff, 0x00, 0xf7, 0x00, 0xee, 0x00}, /* Tune_1 */
	{0xff, 0x00, 0xf8, 0x00, 0xf1, 0x00}, /* Tune_2 */
	{0xff, 0x00, 0xf9, 0x00, 0xf5, 0x00}, /* Tune_3 */
	{0xff, 0x00, 0xf9, 0x00, 0xee, 0x00}, /* Tune_4 */
	{0xff, 0x00, 0xfa, 0x00, 0xf1, 0x00}, /* Tune_5 */
	{0xff, 0x00, 0xfb, 0x00, 0xf5, 0x00}, /* Tune_6 */
	{0xff, 0x00, 0xfc, 0x00, 0xee, 0x00}, /* Tune_7 */
	{0xff, 0x00, 0xfc, 0x00, 0xf1, 0x00}, /* Tune_8 */
	{0xff, 0x00, 0xfd, 0x00, 0xf4, 0x00}, /* Tune_9 */
};

static char (*coordinate_data[MAX_MODE])[COORDINATE_DATA_SIZE] = {
	coordinate_data_1,
	coordinate_data_2,
	coordinate_data_2,
	coordinate_data_1,
	coordinate_data_1,
	coordinate_data_1
};

static int mdnie_coordinate_index(int x, int y)
{
	int tune_number = 0;

	if (F1(x,y) > 0) {
		if (F3(x,y) > 0) {
			tune_number = 3;
		} else {
			if (F4(x,y) < 0)
				tune_number = 1;
			else
				tune_number = 2;
		}
	} else {
		if (F2(x,y) < 0) {
			if (F3(x,y) > 0) {
				tune_number = 9;
			} else {
				if (F4(x,y) < 0)
					tune_number = 7;
				else
					tune_number = 8;
			}
		} else {
			if (F3(x,y) > 0)
				tune_number = 6;
			else {
				if (F4(x,y) < 0)
					tune_number = 4;
				else
					tune_number = 5;
			}
		}
	}

	return tune_number;
}

static int mdss_mdnie_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char x_y_location[4];
	int mdnie_tune_index = 0;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (D6h 1~5th) for ddi id */
	if (vdd->dtsi_data[ctrl->ndx].mdnie_read_rx_cmds[vdd->panel_revision].cmd_cnt) {
		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].mdnie_read_rx_cmds[vdd->panel_revision]),
			x_y_location, PANEL_LEVE1_KEY);

		vdd->mdnie_x[ctrl->ndx] = x_y_location[0] << 8 | x_y_location[1];	/* X */
		vdd->mdnie_y[ctrl->ndx] = x_y_location[2] << 8 | x_y_location[3];	/* Y */

		mdnie_tune_index = mdnie_coordinate_index(vdd->mdnie_x[ctrl->ndx], vdd->mdnie_y[ctrl->ndx]);
		coordinate_tunning_multi(ctrl->ndx, coordinate_data, mdnie_tune_index,
			MDNIE_SCR_WR_ADDR, COORDINATE_DATA_SIZE);

		LCD_INFO("DSI%d : X-%d Y-%d \n", ctrl->ndx,
			vdd->mdnie_x[ctrl->ndx], vdd->mdnie_y[ctrl->ndx]);
	} else {
		LCD_ERR("DSI%d no mdnie_read_rx_cmds cmds", ctrl->ndx);
		return false;
	}

	return true;
}

static int mdss_samart_dimming_init(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	vdd->smart_dimming_dsi[ctrl->ndx] = vdd->panel_func.samsung_smart_get_conf();

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi[ctrl->ndx])) {
		LCD_ERR("DSI%d smart_dimming_dsi is null", ctrl->ndx);
		return false;
	} else {
		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].smart_dimming_mtp_rx_cmds[vdd->panel_revision]),
			vdd->smart_dimming_dsi[ctrl->ndx]->mtp_buffer, PANEL_LEVE1_KEY);

		/* Initialize smart dimming related things here */
		/* lux_tab setting for 350cd */
		vdd->smart_dimming_dsi[ctrl->ndx]->lux_tab = vdd->dtsi_data[ctrl->ndx].candela_map_table[vdd->panel_revision].lux_tab;
		vdd->smart_dimming_dsi[ctrl->ndx]->lux_tabsize = vdd->dtsi_data[ctrl->ndx].candela_map_table[vdd->panel_revision].lux_tab_size;
		vdd->smart_dimming_dsi[ctrl->ndx]->man_id = vdd->manufacture_id_dsi[ctrl->ndx];

		/* hero2 */
		if (mdss_panel_id1_get(ctrl) & 0x10) {
			/* copy hbm gamma payload for hbm interpolation calc */
			vdd->smart_dimming_dsi[ctrl->ndx]->hbm_payload = &vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1];
		}

		/* Just a safety check to ensure smart dimming data is initialised well */
		vdd->smart_dimming_dsi[ctrl->ndx]->init(vdd->smart_dimming_dsi[ctrl->ndx]);

		vdd->temperature = 20; // default temperature

		vdd->smart_dimming_loaded_dsi[ctrl->ndx] = true;
	}

	LCD_INFO("DSI%d : --\n", ctrl->ndx);

	return true;
}

static int mdss_cell_id_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char cell_id_buffer[MAX_CELL_ID] = {0,};
	int loop;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read Panel Unique Cell ID (C8h 41~51th) */
	if (vdd->dtsi_data[ctrl->ndx].cell_id_rx_cmds[vdd->panel_revision].cmd_cnt) {
		memset(cell_id_buffer, 0x00, MAX_CELL_ID);

		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].cell_id_rx_cmds[vdd->panel_revision]),
			cell_id_buffer, PANEL_LEVE1_KEY);

		for(loop = 0; loop < MAX_CELL_ID; loop++)
			vdd->cell_id_dsi[ctrl->ndx][loop] = cell_id_buffer[loop];

		LCD_INFO("DSI%d: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			ctrl->ndx, vdd->cell_id_dsi[ctrl->ndx][0],
			vdd->cell_id_dsi[ctrl->ndx][1],	vdd->cell_id_dsi[ctrl->ndx][2],
			vdd->cell_id_dsi[ctrl->ndx][3],	vdd->cell_id_dsi[ctrl->ndx][4],
			vdd->cell_id_dsi[ctrl->ndx][5],	vdd->cell_id_dsi[ctrl->ndx][6],
			vdd->cell_id_dsi[ctrl->ndx][7],	vdd->cell_id_dsi[ctrl->ndx][8],
			vdd->cell_id_dsi[ctrl->ndx][9],	vdd->cell_id_dsi[ctrl->ndx][10]);

	} else {
		LCD_ERR("DSI%d no cell_id_rx_cmds cmd\n", ctrl->ndx);
		return false;
	}

	return true;
}

static struct dsi_panel_cmds aid_cmd;
static struct dsi_panel_cmds *mdss_aid(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (vdd->aid_subdivision_enable) {
		aid_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].aid_subdivision_tx_cmds[vdd->panel_revision].cmds[vdd->bl_level]);
		LCD_ERR("level(%d), aid(%x %x)\n", vdd->bl_level, aid_cmd.cmds->payload[1], aid_cmd.cmds->payload[2]);
	} else {
		cd_index = get_cmd_index(vdd, ctrl->ndx);

		if (!vdd->dtsi_data[ctrl->ndx].aid_map_table[vdd->panel_revision].size ||
			cd_index > vdd->dtsi_data[ctrl->ndx].aid_map_table[vdd->panel_revision].size)
			goto end;

		cmd_idx = vdd->dtsi_data[ctrl->ndx].aid_map_table[vdd->panel_revision].cmd_idx[cd_index];

		aid_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].aid_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	}

	aid_cmd.cmd_cnt = 1;
	*level_key = PANEL_LEVE1_KEY;

	return &aid_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static struct dsi_panel_cmds * mdss_acl_on(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	if (vdd->gradual_acl_val)
		vdd->dtsi_data[ctrl->ndx].acl_on_tx_cmds[vdd->panel_revision].cmds[0].payload[4] = vdd->gradual_acl_val;

	return &(vdd->dtsi_data[ctrl->ndx].acl_on_tx_cmds[vdd->panel_revision]);
}

static struct dsi_panel_cmds * mdss_acl_off(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	return &(vdd->dtsi_data[ctrl->ndx].acl_off_tx_cmds[vdd->panel_revision]);
}

#if 0
static struct dsi_panel_cmds acl_percent_cmd;
static struct dsi_panel_cmds * mdss_acl_precent(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	cd_index = get_cmd_index(vdd, ctrl->ndx);

	if (!vdd->dtsi_data[ctrl->ndx].acl_map_table[vdd->panel_revision].size ||
		cd_index > vdd->dtsi_data[ctrl->ndx].acl_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].acl_map_table[vdd->panel_revision].cmd_idx[cd_index];

	acl_percent_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].acl_percent_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	acl_percent_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &acl_percent_cmd;

end :
	LCD_ERR("error");
	return NULL;

}
#endif

static struct dsi_panel_cmds elvss_cmd;
static struct dsi_panel_cmds * mdss_elvss(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	cd_index = get_cmd_index(vdd, ctrl->ndx);

	if (!vdd->dtsi_data[ctrl->ndx].smart_acl_elvss_map_table[vdd->panel_revision].size ||
		cd_index > vdd->dtsi_data[ctrl->ndx].smart_acl_elvss_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].smart_acl_elvss_map_table[vdd->panel_revision].cmd_idx[cd_index];

	elvss_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].elvss_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);

	elvss_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &elvss_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static struct dsi_panel_cmds * mdss_elvss_temperature1(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	char elvss_23th_val;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	/* OTP value - B5 23th */
	elvss_23th_val = vdd->display_status_dsi[ctrl->ndx].elvss_value1;

	LCD_DEBUG("OTP val %x\n", elvss_23th_val);

	if (vdd->temperature > 0)
		elvss_23th_val += 0x00;
	else if (vdd->temperature > vdd->elvss_interpolation_temperature) {
		/* B5 23th */
		if (mdss_get_hero2_fab_type() == S6E3HF4_AMB526JS01_A2_LINE){

			if (vdd->candela_level == 5)
				elvss_23th_val += a2_line_data.elvss_offset[1][3];
			else if (vdd->candela_level == 4)
				elvss_23th_val += a2_line_data.elvss_offset[1][2];
			else if (vdd->candela_level == 3)
				elvss_23th_val += a2_line_data.elvss_offset[1][1];
			else if (vdd->candela_level == 2)
				elvss_23th_val += a2_line_data.elvss_offset[1][0];
			else
				elvss_23th_val += 0x00;
		}else {
			if (vdd->candela_level == 5)
				elvss_23th_val += 0x01;
			else if (vdd->candela_level == 4)
				elvss_23th_val += 0x02;
			else if (vdd->candela_level == 3)
				elvss_23th_val += 0x03;
			else if (vdd->candela_level == 2)
				elvss_23th_val += 0x04;
			else
				elvss_23th_val += 0x00;
		}

	} else {
			/* B5 23th */

		if (mdss_get_hero2_fab_type() == S6E3HF4_AMB526JS01_A2_LINE){
			if (vdd->candela_level >= 6)
				elvss_23th_val += a2_line_data.elvss_offset[2][4];
			else if (vdd->candela_level == 5)
				elvss_23th_val += a2_line_data.elvss_offset[2][3];
			else if (vdd->candela_level == 4)
				elvss_23th_val += a2_line_data.elvss_offset[2][2];
			else if (vdd->candela_level == 3)
				elvss_23th_val += a2_line_data.elvss_offset[2][1];
			else
				elvss_23th_val += a2_line_data.elvss_offset[2][0];
		}else {
			if (vdd->candela_level >= 6)
				elvss_23th_val -= 0x04;
			else if (vdd->candela_level == 5)
				elvss_23th_val -= 0x03;
			else if (vdd->candela_level == 4)
				elvss_23th_val -= 0x02;
			else if (vdd->candela_level == 3)
				elvss_23th_val -= 0x01;
			else
				elvss_23th_val += 0x00;
		}
	}

	/* 0xB5 1th TSET */
	vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[0].payload[1] =
		vdd->temperature > 0 ? vdd->temperature : 0x80|(-1*vdd->temperature);

	/* 0xB5 23th elvss_offset */
	vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[2].payload[1] = elvss_23th_val;

	LCD_DEBUG("acl : %d, interpolation_temp : %d temp : %d, cd : %d, B5 1st :0x%x, B5 23th :0x%x\n",
		vdd->acl_status, vdd->elvss_interpolation_temperature, vdd->temperature, vdd->candela_level,
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[2].payload[1]);

	*level_key = PANEL_LEVE1_KEY;

	return &(vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision]);
}

static struct dsi_panel_cmds vint_cmd;
static struct dsi_panel_cmds *mdss_vint(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	cd_index = get_cmd_index(vdd, ctrl->ndx);

	if (!vdd->dtsi_data[ctrl->ndx].vint_map_table[vdd->panel_revision].size ||
		cd_index > vdd->dtsi_data[ctrl->ndx].vint_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].vint_map_table[vdd->panel_revision].cmd_idx[cd_index];

	if (vdd->temperature > vdd->elvss_interpolation_temperature )
		vint_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].vint_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	else
		vint_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].vint_tx_cmds[vdd->panel_revision].cmds[0]);
	vint_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &vint_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static char irc_dimming_para [256][20] = {
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4B, 0x47, 0x42, 0x19, 0x19, 0x1B, 0x0F, 0x11, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4B, 0x47, 0x42, 0x19, 0x19, 0x1B, 0x0F, 0x11, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4A, 0x46, 0x42, 0x19, 0x19, 0x1A, 0x0F, 0x11, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4A, 0x46, 0x41, 0x19, 0x19, 0x1B, 0x0F, 0x11, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4A, 0x46, 0x41, 0x19, 0x19, 0x1B, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4A, 0x46, 0x41, 0x18, 0x18, 0x1A, 0x0F, 0x11, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x49, 0x45, 0x40, 0x19, 0x19, 0x1B, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x49, 0x45, 0x40, 0x18, 0x18, 0x1A, 0x0F, 0x11, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x48, 0x44, 0x40, 0x18, 0x19, 0x1A, 0x0F, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x48, 0x44, 0x3F, 0x18, 0x18, 0x1A, 0x0E, 0x10, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x48, 0x44, 0x3F, 0x17, 0x18, 0x1A, 0x0F, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x47, 0x43, 0x3F, 0x18, 0x18, 0x19, 0x0E, 0x10, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x47, 0x43, 0x3E, 0x18, 0x18, 0x1A, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x47, 0x43, 0x3E, 0x17, 0x17, 0x19, 0x0E, 0x10, 0x0F},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x46, 0x42, 0x3E, 0x18, 0x18, 0x19, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x46, 0x42, 0x3D, 0x17, 0x17, 0x1A, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x45, 0x42, 0x3D, 0x18, 0x17, 0x19, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x45, 0x41, 0x3D, 0x17, 0x17, 0x19, 0x0E, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x45, 0x41, 0x3D, 0x17, 0x17, 0x18, 0x0D, 0x10, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x45, 0x41, 0x3C, 0x16, 0x17, 0x19, 0x0E, 0x0F, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x44, 0x41, 0x3C, 0x17, 0x16, 0x19, 0x0E, 0x10, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x44, 0x40, 0x3C, 0x16, 0x17, 0x18, 0x0E, 0x0F, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x44, 0x40, 0x3B, 0x16, 0x16, 0x19, 0x0E, 0x10, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x43, 0x40, 0x3B, 0x17, 0x16, 0x18, 0x0D, 0x0F, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x43, 0x3F, 0x3B, 0x16, 0x16, 0x18, 0x0D, 0x10, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x42, 0x3F, 0x3A, 0x17, 0x16, 0x18, 0x0D, 0x0F, 0x0E},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x42, 0x3F, 0x3A, 0x16, 0x16, 0x18, 0x0D, 0x0F, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x42, 0x3E, 0x3A, 0x15, 0x16, 0x17, 0x0D, 0x0F, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x41, 0x3E, 0x39, 0x16, 0x15, 0x18, 0x0D, 0x0F, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x41, 0x3D, 0x39, 0x15, 0x16, 0x17, 0x0D, 0x0F, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x40, 0x3D, 0x39, 0x16, 0x16, 0x17, 0x0D, 0x0E, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x40, 0x3D, 0x38, 0x15, 0x15, 0x17, 0x0D, 0x0F, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x40, 0x3C, 0x38, 0x15, 0x16, 0x17, 0x0D, 0x0E, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3F, 0x3C, 0x38, 0x16, 0x15, 0x17, 0x0C, 0x0F, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3F, 0x3C, 0x38, 0x15, 0x15, 0x16, 0x0D, 0x0E, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3F, 0x3C, 0x37, 0x15, 0x14, 0x17, 0x0C, 0x0F, 0x0D},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3F, 0x3B, 0x37, 0x14, 0x15, 0x17, 0x0D, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3E, 0x3B, 0x37, 0x15, 0x15, 0x16, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3E, 0x3A, 0x36, 0x14, 0x15, 0x17, 0x0D, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3D, 0x3A, 0x36, 0x15, 0x15, 0x16, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3D, 0x3A, 0x36, 0x14, 0x14, 0x16, 0x0D, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3D, 0x3A, 0x36, 0x14, 0x14, 0x16, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3D, 0x39, 0x35, 0x14, 0x14, 0x16, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3C, 0x39, 0x35, 0x14, 0x14, 0x16, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3C, 0x39, 0x35, 0x14, 0x14, 0x15, 0x0C, 0x0D, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3B, 0x38, 0x34, 0x14, 0x14, 0x16, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3B, 0x38, 0x34, 0x14, 0x14, 0x15, 0x0C, 0x0D, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3B, 0x38, 0x34, 0x13, 0x13, 0x15, 0x0C, 0x0E, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3A, 0x37, 0x33, 0x14, 0x14, 0x15, 0x0C, 0x0D, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3A, 0x37, 0x33, 0x13, 0x13, 0x15, 0x0C, 0x0D, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3A, 0x37, 0x33, 0x13, 0x13, 0x15, 0x0C, 0x0D, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3A, 0x36, 0x33, 0x13, 0x14, 0x14, 0x0B, 0x0D, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x39, 0x36, 0x32, 0x13, 0x13, 0x15, 0x0C, 0x0D, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x39, 0x36, 0x32, 0x13, 0x13, 0x14, 0x0B, 0x0D, 0x0C},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x38, 0x35, 0x32, 0x13, 0x13, 0x14, 0x0C, 0x0D, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x38, 0x35, 0x31, 0x13, 0x13, 0x15, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x38, 0x35, 0x31, 0x12, 0x12, 0x14, 0x0B, 0x0D, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x37, 0x34, 0x31, 0x13, 0x13, 0x14, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x37, 0x34, 0x30, 0x12, 0x12, 0x14, 0x0B, 0x0D, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x37, 0x34, 0x30, 0x12, 0x12, 0x14, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x36, 0x33, 0x30, 0x12, 0x12, 0x13, 0x0B, 0x0D, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x36, 0x33, 0x2F, 0x12, 0x12, 0x14, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x36, 0x33, 0x2F, 0x11, 0x12, 0x13, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x35, 0x32, 0x2F, 0x12, 0x12, 0x13, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x35, 0x32, 0x2F, 0x11, 0x12, 0x13, 0x0B, 0x0C, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x35, 0x32, 0x2E, 0x11, 0x11, 0x13, 0x0B, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x34, 0x32, 0x2E, 0x12, 0x11, 0x13, 0x0A, 0x0C, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x34, 0x31, 0x2E, 0x11, 0x12, 0x12, 0x0B, 0x0B, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x34, 0x31, 0x2D, 0x11, 0x11, 0x13, 0x0A, 0x0C, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x33, 0x31, 0x2D, 0x11, 0x11, 0x13, 0x0B, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x33, 0x30, 0x2D, 0x11, 0x11, 0x12, 0x0A, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x33, 0x30, 0x2D, 0x11, 0x11, 0x12, 0x0A, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x32, 0x30, 0x2C, 0x11, 0x10, 0x12, 0x0A, 0x0C, 0x0B},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x32, 0x2F, 0x2C, 0x11, 0x11, 0x12, 0x0A, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x32, 0x2F, 0x2C, 0x10, 0x11, 0x12, 0x0A, 0x0B, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x31, 0x2F, 0x2B, 0x11, 0x10, 0x12, 0x0A, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x31, 0x2E, 0x2B, 0x10, 0x11, 0x12, 0x0A, 0x0B, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x31, 0x2E, 0x2B, 0x10, 0x10, 0x11, 0x0A, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x30, 0x2E, 0x2B, 0x11, 0x10, 0x11, 0x9, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x30, 0x2D, 0x2A, 0x10, 0x10, 0x12, 0x0A, 0x0B, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x30, 0x2D, 0x2A, 0x10, 0x10, 0x11, 0x9, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2F, 0x2D, 0x2A, 0x10, 0x10, 0x11, 0x0A, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2F, 0x2C, 0x29, 0x10, 0x10, 0x11, 0x9, 0x0B, 0x0A},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2F, 0x2C, 0x29, 0x0F, 0x10, 0x11, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2E, 0x2C, 0x29, 0x10, 0x0F, 0x11, 0x9, 0x0B, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2E, 0x2C, 0x29, 0x0F, 0x0F, 0x10, 0x0A, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2E, 0x2B, 0x28, 0x0F, 0x10, 0x11, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2D, 0x2B, 0x28, 0x0F, 0x0F, 0x10, 0x0A, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2D, 0x2B, 0x28, 0x0F, 0x0F, 0x10, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2D, 0x2A, 0x27, 0x0F, 0x0F, 0x11, 0x9, 0x0B, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2C, 0x2A, 0x27, 0x0F, 0x0F, 0x10, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2C, 0x2A, 0x27, 0x0F, 0x0E, 0x10, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2C, 0x29, 0x27, 0x0E, 0x0F, 0x0F, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2B, 0x29, 0x26, 0x0F, 0x0F, 0x10, 0x9, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2B, 0x29, 0x26, 0x0F, 0x0E, 0x10, 0x8, 0x0A, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2B, 0x29, 0x26, 0x0E, 0x0E, 0x0F, 0x9, 0x0A, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2B, 0x28, 0x25, 0x0E, 0x0E, 0x10, 0x8, 0x0A, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2A, 0x28, 0x25, 0x0E, 0x0E, 0x0F, 0x9, 0x9, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2A, 0x28, 0x25, 0x0E, 0x0D, 0x0F, 0x8, 0x0A, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x29, 0x27, 0x24, 0x0E, 0x0E, 0x0F, 0x9, 0x9, 0x9},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x29, 0x27, 0x24, 0x0E, 0x0E, 0x0F, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x29, 0x27, 0x24, 0x0E, 0x0D, 0x0F, 0x8, 0x0A, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x29, 0x26, 0x24, 0x0D, 0x0E, 0x0E, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x28, 0x26, 0x23, 0x0E, 0x0D, 0x0F, 0x8, 0x0A, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x28, 0x26, 0x23, 0x0D, 0x0D, 0x0F, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x28, 0x26, 0x23, 0x0D, 0x0D, 0x0E, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x27, 0x25, 0x23, 0x0D, 0x0D, 0x0E, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x27, 0x25, 0x22, 0x0D, 0x0D, 0x0E, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x27, 0x25, 0x22, 0x0C, 0x0C, 0x0E, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x26, 0x24, 0x22, 0x0D, 0x0D, 0x0D, 0x8, 0x9, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x26, 0x24, 0x21, 0x0D, 0x0D, 0x0E, 0x7, 0x8, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x26, 0x24, 0x21, 0x0C, 0x0C, 0x0E, 0x8, 0x9, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x25, 0x23, 0x21, 0x0D, 0x0D, 0x0D, 0x7, 0x8, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x25, 0x23, 0x21, 0x0C, 0x0C, 0x0D, 0x8, 0x9, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x25, 0x23, 0x20, 0x0C, 0x0C, 0x0D, 0x7, 0x8, 0x8},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x24, 0x22, 0x20, 0x0D, 0x0D, 0x0D, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x24, 0x22, 0x20, 0x0C, 0x0C, 0x0D, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x24, 0x22, 0x1F, 0x0C, 0x0C, 0x0D, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x23, 0x21, 0x1F, 0x0C, 0x0C, 0x0D, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x23, 0x21, 0x1F, 0x0C, 0x0C, 0x0D, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x23, 0x21, 0x1F, 0x0B, 0x0C, 0x0C, 0x7, 0x7, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x22, 0x21, 0x1E, 0x0C, 0x0B, 0x0D, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x22, 0x20, 0x1E, 0x0C, 0x0C, 0x0D, 0x7, 0x8, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x22, 0x20, 0x1E, 0x0B, 0x0B, 0x0C, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x22, 0x20, 0x1E, 0x0B, 0x0B, 0x0C, 0x6, 0x8, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x21, 0x1F, 0x1D, 0x0B, 0x0C, 0x0C, 0x7, 0x7, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x21, 0x1F, 0x1D, 0x0B, 0x0B, 0x0C, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x21, 0x1F, 0x1D, 0x0B, 0x0B, 0x0C, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x20, 0x1F, 0x1C, 0x0B, 0x0A, 0x0C, 0x7, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x20, 0x1E, 0x1C, 0x0B, 0x0B, 0x0C, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x20, 0x1E, 0x1C, 0x0A, 0x0B, 0x0B, 0x7, 0x7, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x20, 0x1E, 0x1C, 0x0A, 0x0A, 0x0B, 0x6, 0x8, 0x7},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1F, 0x1E, 0x1C, 0x0B, 0x0A, 0x0B, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1F, 0x1D, 0x1B, 0x0A, 0x0B, 0x0C, 0x7, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1F, 0x1D, 0x1B, 0x0A, 0x0A, 0x0B, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1E, 0x1D, 0x1B, 0x0A, 0x0A, 0x0B, 0x7, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1E, 0x1C, 0x1A, 0x0A, 0x0A, 0x0B, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1E, 0x1C, 0x1A, 0x0A, 0x0A, 0x0B, 0x5, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1D, 0x1C, 0x1A, 0x0A, 0x9, 0x0A, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1D, 0x1B, 0x19, 0x0A, 0x0A, 0x0B, 0x5, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1D, 0x1B, 0x19, 0x9, 0x0A, 0x0A, 0x6, 0x6, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1C, 0x1B, 0x19, 0x0A, 0x9, 0x0A, 0x6, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1C, 0x1B, 0x19, 0x9, 0x9, 0x0A, 0x6, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1C, 0x1A, 0x18, 0x9, 0x9, 0x0A, 0x5, 0x7, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1C, 0x1A, 0x18, 0x9, 0x9, 0x0A, 0x5, 0x6, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1B, 0x1A, 0x18, 0x9, 0x9, 0x0A, 0x6, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1B, 0x1A, 0x18, 0x9, 0x9, 0x9, 0x5, 0x6, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1B, 0x19, 0x17, 0x8, 0x9, 0x0A, 0x6, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1A, 0x19, 0x17, 0x9, 0x9, 0x0A, 0x6, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1A, 0x19, 0x17, 0x9, 0x8, 0x9, 0x5, 0x6, 0x6},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1A, 0x19, 0x17, 0x9, 0x8, 0x9, 0x5, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x1A, 0x18, 0x16, 0x8, 0x9, 0x0A, 0x5, 0x5, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x19, 0x18, 0x16, 0x9, 0x8, 0x9, 0x5, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x19, 0x17, 0x16, 0x8, 0x9, 0x9, 0x5, 0x5, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x19, 0x17, 0x16, 0x8, 0x9, 0x9, 0x5, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x18, 0x17, 0x15, 0x8, 0x8, 0x9, 0x5, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x18, 0x17, 0x15, 0x8, 0x8, 0x9, 0x5, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x18, 0x16, 0x15, 0x7, 0x8, 0x8, 0x5, 0x6, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x17, 0x16, 0x15, 0x8, 0x8, 0x8, 0x5, 0x5, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x17, 0x16, 0x14, 0x8, 0x7, 0x9, 0x4, 0x6, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x17, 0x16, 0x14, 0x7, 0x7, 0x8, 0x5, 0x5, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x17, 0x15, 0x14, 0x7, 0x8, 0x8, 0x5, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x16, 0x15, 0x14, 0x8, 0x8, 0x8, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x16, 0x15, 0x13, 0x7, 0x7, 0x8, 0x5, 0x5, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x16, 0x15, 0x13, 0x7, 0x7, 0x8, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x15, 0x14, 0x13, 0x8, 0x7, 0x8, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x15, 0x14, 0x13, 0x7, 0x7, 0x7, 0x5, 0x5, 0x5},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x15, 0x14, 0x12, 0x7, 0x7, 0x8, 0x4, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x15, 0x14, 0x12, 0x7, 0x7, 0x8, 0x4, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x14, 0x13, 0x12, 0x7, 0x7, 0x7, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x14, 0x13, 0x12, 0x7, 0x7, 0x7, 0x4, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x14, 0x13, 0x11, 0x6, 0x6, 0x8, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x14, 0x13, 0x11, 0x6, 0x6, 0x7, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x13, 0x12, 0x11, 0x7, 0x7, 0x7, 0x4, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x13, 0x12, 0x11, 0x6, 0x6, 0x7, 0x4, 0x5, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x13, 0x12, 0x11, 0x6, 0x6, 0x6, 0x4, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x12, 0x11, 0x10, 0x7, 0x7, 0x7, 0x3, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x12, 0x11, 0x10, 0x6, 0x6, 0x7, 0x4, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x12, 0x11, 0x10, 0x6, 0x6, 0x6, 0x3, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x12, 0x11, 0x0F, 0x5, 0x5, 0x7, 0x4, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x11, 0x10, 0x0F, 0x6, 0x6, 0x6, 0x3, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x11, 0x10, 0x0F, 0x6, 0x6, 0x6, 0x3, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x11, 0x10, 0x0F, 0x5, 0x5, 0x6, 0x3, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x10, 0x10, 0x0E, 0x6, 0x5, 0x6, 0x3, 0x4, 0x4},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x10, 0x0F, 0x0E, 0x5, 0x6, 0x6, 0x4, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x10, 0x0F, 0x0E, 0x5, 0x5, 0x6, 0x3, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x10, 0x0F, 0x0E, 0x5, 0x5, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0F, 0x0F, 0x0E, 0x5, 0x5, 0x5, 0x4, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0F, 0x0E, 0x0D, 0x5, 0x5, 0x6, 0x3, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0F, 0x0E, 0x0D, 0x5, 0x5, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0F, 0x0E, 0x0D, 0x5, 0x5, 0x5, 0x2, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0E, 0x0E, 0x0D, 0x5, 0x4, 0x5, 0x3, 0x4, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0E, 0x0D, 0x0C, 0x5, 0x5, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0E, 0x0D, 0x0C, 0x4, 0x5, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0E, 0x0D, 0x0C, 0x4, 0x4, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0D, 0x0D, 0x0C, 0x5, 0x4, 0x5, 0x3, 0x3, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0D, 0x0C, 0x0B, 0x4, 0x5, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0D, 0x0C, 0x0B, 0x4, 0x4, 0x5, 0x3, 0x3, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0D, 0x0C, 0x0B, 0x4, 0x4, 0x5, 0x2, 0x3, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0C, 0x0B, 0x0B, 0x4, 0x5, 0x4, 0x3, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0C, 0x0B, 0x0B, 0x4, 0x4, 0x4, 0x2, 0x3, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0C, 0x0B, 0x0A, 0x3, 0x4, 0x4, 0x3, 0x2, 0x3},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0B, 0x0B, 0x0A, 0x4, 0x4, 0x4, 0x3, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0B, 0x0B, 0x0A, 0x4, 0x3, 0x4, 0x2, 0x3, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0B, 0x0A, 0x0A, 0x4, 0x4, 0x4, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0B, 0x0A, 0x9, 0x3, 0x4, 0x4, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0B, 0x0A, 0x9, 0x3, 0x3, 0x4, 0x2, 0x3, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0A, 0x0A, 0x9, 0x4, 0x3, 0x4, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0A, 0x9, 0x9, 0x3, 0x4, 0x3, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0A, 0x9, 0x9, 0x3, 0x4, 0x3, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x9, 0x9, 0x8, 0x4, 0x3, 0x4, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x9, 0x9, 0x8, 0x3, 0x3, 0x4, 0x2, 0x2, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x9, 0x8, 0x8, 0x3, 0x3, 0x3, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x9, 0x8, 0x8, 0x3, 0x3, 0x3, 0x1, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x8, 0x8, 0x7, 0x3, 0x3, 0x3, 0x2, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x8, 0x8, 0x7, 0x3, 0x3, 0x3, 0x2, 0x1, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x8, 0x8, 0x7, 0x3, 0x2, 0x3, 0x1, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x8, 0x7, 0x7, 0x2, 0x3, 0x3, 0x2, 0x2, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x8, 0x7, 0x7, 0x2, 0x3, 0x3, 0x2, 0x2, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x7, 0x7, 0x6, 0x3, 0x2, 0x3, 0x1, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x7, 0x7, 0x6, 0x3, 0x2, 0x3, 0x1, 0x2, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x7, 0x7, 0x6, 0x2, 0x2, 0x3, 0x2, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x7, 0x6, 0x6, 0x2, 0x2, 0x2, 0x1, 0x2, 0x2},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x6, 0x6, 0x6, 0x3, 0x2, 0x2, 0x1, 0x2, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x6, 0x6, 0x5, 0x2, 0x2, 0x3, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x6, 0x6, 0x5, 0x2, 0x2, 0x2, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x6, 0x5, 0x5, 0x2, 0x2, 0x2, 0x1, 0x2, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x6, 0x5, 0x5, 0x1, 0x2, 0x2, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x5, 0x5, 0x5, 0x2, 0x2, 0x2, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x5, 0x5, 0x5, 0x2, 0x2, 0x1, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x5, 0x5, 0x4, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x5, 0x4, 0x4, 0x1, 0x2, 0x2, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4, 0x4, 0x4, 0x2, 0x2, 0x2, 0x1, 0x1, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4, 0x4, 0x4, 0x2, 0x1, 0x1, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4, 0x4, 0x3, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4, 0x4, 0x3, 0x1, 0x1, 0x2, 0x1, 0x1, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4, 0x3, 0x3, 0x1, 0x2, 0x1, 0x0, 0x0, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3, 0x3, 0x3, 0x2, 0x1, 0x1, 0x0, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3, 0x3, 0x3, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3, 0x3, 0x3, 0x1, 0x1, 0x1, 0x1, 0x1, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3, 0x3, 0x3, 0x1, 0x1, 0x1, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3, 0x3, 0x2, 0x1, 0x0, 0x1, 0x0, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x3, 0x2, 0x2, 0x0, 0x1, 0x1, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2, 0x2, 0x2, 0x1, 0x1, 0x1, 0x1, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2, 0x2, 0x2, 0x1, 0x1, 0x1, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2, 0x2, 0x2, 0x1, 0x1, 0x0, 0x0, 0x0, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x2, 0x2, 0x2, 0x0, 0x0, 0x0, 0x1, 0x1, 0x1},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	{0x1, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
};

static char irc_dimming_para_revJ [256][20] = {
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4D, 0x4B, 0x45, 0x21, 0x20, 0x21, 0x18, 0x14, 0x15},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4D, 0x4B, 0x45, 0x21, 0x20, 0x21, 0x18, 0x14, 0x15},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4C, 0x4A, 0x45, 0x21, 0x20, 0x20, 0x18, 0x14, 0x15},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4C, 0x4A, 0x44, 0x21, 0x20, 0x21, 0x17, 0x13, 0x15},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4C, 0x4A, 0x44, 0x20, 0x1F, 0x21, 0x18, 0x14, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4C, 0x4A, 0x44, 0x20, 0x1F, 0x20, 0x17, 0x14, 0x15},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4B, 0x49, 0x43, 0x20, 0x1F, 0x21, 0x18, 0x14, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4B, 0x49, 0x43, 0x20, 0x1F, 0x20, 0x17, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4A, 0x48, 0x43, 0x20, 0x1F, 0x1F, 0x17, 0x13, 0x15},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4A, 0x48, 0x42, 0x20, 0x1F, 0x20, 0x17, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4A, 0x48, 0x42, 0x1F, 0x1E, 0x1F, 0x17, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x49, 0x47, 0x42, 0x20, 0x1F, 0x1F, 0x16, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x49, 0x47, 0x41, 0x1F, 0x1E, 0x1F, 0x17, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x48, 0x47, 0x41, 0x1F, 0x1E, 0x1F, 0x17, 0x12, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x48, 0x46, 0x41, 0x1F, 0x1E, 0x1E, 0x16, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x48, 0x46, 0x40, 0x1E, 0x1E, 0x1F, 0x17, 0x12, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x47, 0x45, 0x40, 0x1F, 0x1E, 0x1E, 0x16, 0x13, 0x14},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x47, 0x45, 0x40, 0x1E, 0x1E, 0x1E, 0x16, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x47, 0x45, 0x3F, 0x1E, 0x1D, 0x1F, 0x16, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x46, 0x45, 0x3F, 0x1F, 0x1D, 0x1E, 0x16, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x46, 0x44, 0x3F, 0x1E, 0x1D, 0x1E, 0x16, 0x13, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x46, 0x44, 0x3E, 0x1E, 0x1D, 0x1E, 0x15, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x45, 0x44, 0x3E, 0x1E, 0x1C, 0x1E, 0x16, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x45, 0x43, 0x3E, 0x1D, 0x1D, 0x1D, 0x16, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x45, 0x43, 0x3D, 0x1D, 0x1C, 0x1E, 0x15, 0x12, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x44, 0x42, 0x3D, 0x1D, 0x1D, 0x1D, 0x16, 0x11, 0x13},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x44, 0x42, 0x3D, 0x1D, 0x1C, 0x1D, 0x15, 0x12, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x43, 0x42, 0x3C, 0x1D, 0x1B, 0x1D, 0x15, 0x12, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x43, 0x41, 0x3C, 0x1D, 0x1C, 0x1D, 0x14, 0x11, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x43, 0x41, 0x3C, 0x1C, 0x1B, 0x1C, 0x15, 0x12, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x42, 0x40, 0x3B, 0x1D, 0x1C, 0x1D, 0x14, 0x11, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x42, 0x40, 0x3B, 0x1C, 0x1B, 0x1C, 0x15, 0x12, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x41, 0x40, 0x3B, 0x1D, 0x1B, 0x1C, 0x14, 0x11, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x41, 0x3F, 0x3A, 0x1C, 0x1B, 0x1C, 0x14, 0x11, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x41, 0x3F, 0x3A, 0x1C, 0x1B, 0x1C, 0x14, 0x11, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x41, 0x3F, 0x3A, 0x1B, 0x1B, 0x1B, 0x14, 0x10, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x40, 0x3F, 0x3A, 0x1C, 0x1A, 0x1B, 0x14, 0x11, 0x12},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x40, 0x3E, 0x39, 0x1B, 0x1B, 0x1C, 0x14, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3F, 0x3E, 0x39, 0x1C, 0x1A, 0x1B, 0x13, 0x11, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3F, 0x3D, 0x39, 0x1B, 0x1B, 0x1B, 0x14, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3F, 0x3D, 0x38, 0x1B, 0x1A, 0x1B, 0x13, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3F, 0x3D, 0x38, 0x1A, 0x1A, 0x1B, 0x14, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3E, 0x3D, 0x38, 0x1B, 0x19, 0x1A, 0x13, 0x11, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3E, 0x3C, 0x37, 0x1A, 0x1A, 0x1B, 0x14, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3D, 0x3C, 0x37, 0x1B, 0x19, 0x1A, 0x13, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3D, 0x3B, 0x37, 0x1A, 0x1A, 0x1A, 0x13, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3D, 0x3B, 0x36, 0x1A, 0x19, 0x1A, 0x13, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3C, 0x3B, 0x36, 0x1A, 0x19, 0x1A, 0x13, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3C, 0x3A, 0x36, 0x1A, 0x19, 0x19, 0x12, 0x10, 0x11},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3C, 0x3A, 0x35, 0x19, 0x19, 0x1A, 0x13, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3B, 0x3A, 0x35, 0x1A, 0x19, 0x1A, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3B, 0x3A, 0x35, 0x19, 0x18, 0x19, 0x13, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3B, 0x39, 0x35, 0x19, 0x19, 0x19, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3A, 0x39, 0x34, 0x19, 0x18, 0x19, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3A, 0x38, 0x34, 0x19, 0x19, 0x19, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3A, 0x38, 0x34, 0x18, 0x18, 0x18, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x39, 0x38, 0x33, 0x19, 0x17, 0x19, 0x12, 0x0F, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x39, 0x37, 0x33, 0x18, 0x18, 0x18, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x38, 0x37, 0x33, 0x19, 0x17, 0x18, 0x11, 0x0F, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x38, 0x37, 0x32, 0x18, 0x17, 0x18, 0x12, 0x0F, 0x10},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x38, 0x36, 0x32, 0x18, 0x17, 0x18, 0x11, 0x0F, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x37, 0x36, 0x32, 0x18, 0x17, 0x17, 0x11, 0x0E, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x37, 0x36, 0x31, 0x18, 0x16, 0x18, 0x11, 0x0F, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x37, 0x35, 0x31, 0x17, 0x17, 0x17, 0x11, 0x0E, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x36, 0x35, 0x31, 0x18, 0x16, 0x17, 0x10, 0x0F, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x36, 0x35, 0x30, 0x17, 0x16, 0x18, 0x11, 0x0E, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x36, 0x34, 0x30, 0x17, 0x17, 0x17, 0x10, 0x0E, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x35, 0x34, 0x30, 0x17, 0x16, 0x17, 0x11, 0x0E, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x35, 0x34, 0x2F, 0x17, 0x16, 0x17, 0x10, 0x0D, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x35, 0x33, 0x2F, 0x16, 0x16, 0x17, 0x11, 0x0E, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x34, 0x33, 0x2F, 0x17, 0x16, 0x16, 0x10, 0x0D, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x34, 0x33, 0x2F, 0x16, 0x15, 0x16, 0x11, 0x0E, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x34, 0x32, 0x2E, 0x16, 0x16, 0x16, 0x10, 0x0D, 0x0F},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x33, 0x32, 0x2E, 0x16, 0x15, 0x16, 0x10, 0x0E, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x33, 0x32, 0x2E, 0x16, 0x15, 0x16, 0x10, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x33, 0x31, 0x2D, 0x15, 0x15, 0x16, 0x10, 0x0D, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x32, 0x31, 0x2D, 0x16, 0x15, 0x16, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x32, 0x31, 0x2D, 0x16, 0x15, 0x15, 0x0F, 0x0D, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x32, 0x30, 0x2D, 0x15, 0x15, 0x15, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x31, 0x30, 0x2C, 0x15, 0x15, 0x15, 0x10, 0x0C, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x31, 0x30, 0x2C, 0x15, 0x14, 0x15, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x31, 0x2F, 0x2C, 0x14, 0x15, 0x14, 0x10, 0x0C, 0x0E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x30, 0x2F, 0x2B, 0x15, 0x14, 0x15, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x30, 0x2F, 0x2B, 0x14, 0x13, 0x14, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x30, 0x2E, 0x2B, 0x14, 0x14, 0x14, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2F, 0x2E, 0x2A, 0x15, 0x14, 0x15, 0x0E, 0x0C, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2F, 0x2E, 0x2A, 0x14, 0x13, 0x14, 0x0F, 0x0C, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2F, 0x2D, 0x2A, 0x14, 0x14, 0x14, 0x0E, 0x0C, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2E, 0x2D, 0x2A, 0x14, 0x13, 0x13, 0x0F, 0x0D, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2E, 0x2D, 0x29, 0x14, 0x13, 0x14, 0x0E, 0x0C, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2E, 0x2C, 0x29, 0x13, 0x13, 0x13, 0x0E, 0x0C, 0x0D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2D, 0x2C, 0x29, 0x14, 0x13, 0x13, 0x0E, 0x0C, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2D, 0x2C, 0x28, 0x13, 0x12, 0x14, 0x0E, 0x0C, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2D, 0x2B, 0x28, 0x13, 0x13, 0x13, 0x0E, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2C, 0x2B, 0x28, 0x13, 0x13, 0x13, 0x0E, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2C, 0x2B, 0x27, 0x13, 0x12, 0x13, 0x0E, 0x0C, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2C, 0x2B, 0x27, 0x12, 0x12, 0x13, 0x0E, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2B, 0x2A, 0x27, 0x13, 0x12, 0x12, 0x0D, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2B, 0x2A, 0x26, 0x12, 0x12, 0x13, 0x0E, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2B, 0x29, 0x26, 0x12, 0x12, 0x12, 0x0D, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2A, 0x29, 0x26, 0x13, 0x12, 0x12, 0x0D, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2A, 0x29, 0x26, 0x12, 0x11, 0x12, 0x0D, 0x0B, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2A, 0x29, 0x25, 0x11, 0x11, 0x12, 0x0D, 0x0B, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x29, 0x28, 0x25, 0x12, 0x11, 0x12, 0x0D, 0x0B, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x29, 0x28, 0x25, 0x12, 0x11, 0x11, 0x0C, 0x0B, 0x0C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x29, 0x28, 0x24, 0x11, 0x11, 0x12, 0x0D, 0x0A, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x28, 0x27, 0x24, 0x12, 0x11, 0x11, 0x0C, 0x0B, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x28, 0x27, 0x24, 0x11, 0x11, 0x11, 0x0D, 0x0A, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x28, 0x27, 0x23, 0x11, 0x10, 0x11, 0x0C, 0x0A, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x27, 0x26, 0x23, 0x11, 0x11, 0x11, 0x0C, 0x0A, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x27, 0x26, 0x23, 0x11, 0x10, 0x11, 0x0C, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x27, 0x26, 0x23, 0x10, 0x10, 0x10, 0x0C, 0x0A, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x26, 0x25, 0x22, 0x11, 0x10, 0x11, 0x0C, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x26, 0x25, 0x22, 0x10, 0x10, 0x10, 0x0C, 0x0A, 0x0B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x26, 0x25, 0x22, 0x10, 0x0F, 0x10, 0x0B, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x25, 0x24, 0x22, 0x10, 0x10, 0x10, 0x0C, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x25, 0x24, 0x21, 0x10, 0x0F, 0x10, 0x0B, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x25, 0x24, 0x21, 0x0F, 0x0F, 0x10, 0x0C, 0x09, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x24, 0x23, 0x21, 0x10, 0x0F, 0x0F, 0x0B, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x24, 0x23, 0x20, 0x10, 0x0F, 0x10, 0x0B, 0x0A, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x24, 0x23, 0x20, 0x0F, 0x0F, 0x0F, 0x0B, 0x09, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x23, 0x22, 0x20, 0x10, 0x0F, 0x0F, 0x0B, 0x09, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x23, 0x22, 0x20, 0x0F, 0x0F, 0x0F, 0x0B, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x23, 0x22, 0x1F, 0x0F, 0x0E, 0x0F, 0x0B, 0x09, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x22, 0x22, 0x1F, 0x0F, 0x0E, 0x0F, 0x0B, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x22, 0x21, 0x1F, 0x0F, 0x0E, 0x0E, 0x0A, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x22, 0x21, 0x1E, 0x0E, 0x0E, 0x0F, 0x0B, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x22, 0x21, 0x1E, 0x0E, 0x0E, 0x0E, 0x0A, 0x08, 0x0A},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x21, 0x20, 0x1E, 0x0E, 0x0E, 0x0E, 0x0B, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x21, 0x20, 0x1E, 0x0E, 0x0E, 0x0E, 0x0A, 0x08, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x21, 0x20, 0x1D, 0x0E, 0x0D, 0x0E, 0x0A, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x20, 0x20, 0x1D, 0x0E, 0x0D, 0x0E, 0x0A, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x20, 0x1F, 0x1D, 0x0E, 0x0E, 0x0E, 0x0A, 0x08, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x20, 0x1F, 0x1D, 0x0E, 0x0D, 0x0D, 0x0A, 0x09, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x20, 0x1F, 0x1C, 0x0D, 0x0D, 0x0E, 0x0A, 0x08, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1F, 0x1E, 0x1C, 0x0E, 0x0D, 0x0D, 0x09, 0x08, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1F, 0x1E, 0x1C, 0x0D, 0x0D, 0x0D, 0x0A, 0x08, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1E, 0x1E, 0x1B, 0x0D, 0x0C, 0x0D, 0x0A, 0x08, 0x09},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1E, 0x1D, 0x1B, 0x0D, 0x0D, 0x0D, 0x09, 0x08, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1E, 0x1D, 0x1B, 0x0C, 0x0C, 0x0C, 0x0A, 0x08, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1D, 0x1D, 0x1A, 0x0D, 0x0C, 0x0D, 0x09, 0x07, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1D, 0x1C, 0x1A, 0x0D, 0x0D, 0x0D, 0x09, 0x07, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1D, 0x1C, 0x1A, 0x0C, 0x0C, 0x0C, 0x09, 0x07, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1C, 0x19, 0x0D, 0x0B, 0x0D, 0x08, 0x08, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1C, 0x19, 0x0C, 0x0B, 0x0C, 0x09, 0x08, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1B, 0x19, 0x0C, 0x0C, 0x0C, 0x08, 0x07, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1B, 0x19, 0x0C, 0x0B, 0x0C, 0x08, 0x08, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1B, 0x1B, 0x18, 0x0C, 0x0B, 0x0C, 0x09, 0x07, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1B, 0x1A, 0x18, 0x0C, 0x0C, 0x0C, 0x08, 0x07, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1B, 0x1A, 0x18, 0x0B, 0x0B, 0x0B, 0x09, 0x07, 0x08},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1B, 0x1A, 0x18, 0x0B, 0x0B, 0x0B, 0x08, 0x07, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1A, 0x1A, 0x17, 0x0B, 0x0A, 0x0C, 0x09, 0x07, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1A, 0x19, 0x17, 0x0B, 0x0B, 0x0B, 0x08, 0x07, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x19, 0x19, 0x17, 0x0B, 0x0A, 0x0B, 0x08, 0x07, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x19, 0x19, 0x17, 0x0B, 0x0A, 0x0B, 0x08, 0x07, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x19, 0x18, 0x16, 0x0B, 0x0B, 0x0B, 0x07, 0x06, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x19, 0x18, 0x16, 0x0A, 0x0A, 0x0B, 0x08, 0x07, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x18, 0x18, 0x16, 0x0B, 0x0A, 0x0A, 0x07, 0x06, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x18, 0x17, 0x16, 0x0A, 0x0A, 0x0A, 0x08, 0x07, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x18, 0x17, 0x15, 0x0A, 0x0A, 0x0A, 0x07, 0x06, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x17, 0x15, 0x0B, 0x0A, 0x0A, 0x07, 0x06, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x17, 0x15, 0x0A, 0x09, 0x0A, 0x07, 0x06, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x16, 0x15, 0x0A, 0x0A, 0x09, 0x07, 0x06, 0x07},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x16, 0x14, 0x09, 0x09, 0x0A, 0x07, 0x06, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x16, 0x16, 0x14, 0x0A, 0x09, 0x0A, 0x07, 0x06, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x16, 0x15, 0x14, 0x09, 0x0A, 0x09, 0x07, 0x05, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x16, 0x15, 0x14, 0x09, 0x09, 0x09, 0x07, 0x06, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x15, 0x15, 0x13, 0x0A, 0x09, 0x09, 0x06, 0x05, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x15, 0x15, 0x13, 0x09, 0x09, 0x09, 0x07, 0x05, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x15, 0x14, 0x13, 0x09, 0x09, 0x09, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x15, 0x14, 0x13, 0x09, 0x09, 0x08, 0x06, 0x05, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x14, 0x12, 0x09, 0x08, 0x09, 0x06, 0x06, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x14, 0x12, 0x09, 0x08, 0x09, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x13, 0x12, 0x08, 0x09, 0x08, 0x06, 0x05, 0x06},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x13, 0x12, 0x08, 0x08, 0x08, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x13, 0x13, 0x11, 0x09, 0x08, 0x09, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x13, 0x12, 0x11, 0x08, 0x08, 0x08, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x13, 0x12, 0x11, 0x08, 0x08, 0x08, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x12, 0x12, 0x10, 0x08, 0x07, 0x08, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x12, 0x12, 0x10, 0x08, 0x07, 0x08, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x12, 0x11, 0x10, 0x07, 0x07, 0x07, 0x06, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x11, 0x10, 0x08, 0x07, 0x07, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x11, 0x0F, 0x07, 0x07, 0x08, 0x06, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x10, 0x0F, 0x07, 0x07, 0x07, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x10, 0x0F, 0x07, 0x07, 0x07, 0x05, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x10, 0x0F, 0x07, 0x07, 0x07, 0x05, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x10, 0x0E, 0x07, 0x06, 0x07, 0x05, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x0F, 0x0E, 0x07, 0x07, 0x07, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x0F, 0x0E, 0x06, 0x07, 0x07, 0x05, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0F, 0x0F, 0x0E, 0x07, 0x06, 0x06, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0F, 0x0F, 0x0D, 0x06, 0x06, 0x07, 0x05, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0F, 0x0E, 0x0D, 0x06, 0x06, 0x06, 0x05, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0E, 0x0D, 0x07, 0x06, 0x06, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0E, 0x0D, 0x06, 0x06, 0x06, 0x05, 0x03, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0E, 0x0C, 0x06, 0x05, 0x06, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0D, 0x0C, 0x06, 0x06, 0x06, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0D, 0x0D, 0x0C, 0x06, 0x06, 0x06, 0x04, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0D, 0x0D, 0x0C, 0x06, 0x05, 0x05, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0D, 0x0D, 0x0C, 0x05, 0x05, 0x05, 0x04, 0x03, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0C, 0x0B, 0x06, 0x05, 0x06, 0x04, 0x04, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0C, 0x0B, 0x06, 0x05, 0x05, 0x03, 0x03, 0x04},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0C, 0x0B, 0x05, 0x05, 0x05, 0x04, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0B, 0x0B, 0x05, 0x05, 0x05, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0B, 0x0A, 0x05, 0x05, 0x05, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0B, 0x0B, 0x0A, 0x05, 0x05, 0x05, 0x03, 0x02, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0B, 0x0B, 0x0A, 0x05, 0x04, 0x05, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0B, 0x0B, 0x0A, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x0A, 0x09, 0x05, 0x05, 0x05, 0x03, 0x02, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x0A, 0x09, 0x05, 0x04, 0x05, 0x03, 0x03, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x0A, 0x09, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x09, 0x09, 0x04, 0x05, 0x04, 0x03, 0x02, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x09, 0x09, 0x04, 0x04, 0x04, 0x03, 0x03, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x09, 0x09, 0x08, 0x04, 0x04, 0x04, 0x03, 0x02, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x09, 0x09, 0x08, 0x04, 0x03, 0x04, 0x03, 0x03, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x09, 0x08, 0x08, 0x03, 0x04, 0x03, 0x03, 0x02, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x08, 0x04, 0x04, 0x03, 0x03, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x07, 0x04, 0x03, 0x04, 0x02, 0x03, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x07, 0x04, 0x03, 0x04, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x07, 0x03, 0x03, 0x03, 0x03, 0x02, 0x03},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x07, 0x07, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x07, 0x07, 0x03, 0x03, 0x03, 0x03, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x07, 0x06, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x07, 0x06, 0x03, 0x02, 0x03, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x06, 0x06, 0x02, 0x03, 0x03, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x06, 0x06, 0x03, 0x03, 0x02, 0x02, 0x01, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x06, 0x05, 0x03, 0x02, 0x03, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x06, 0x05, 0x02, 0x02, 0x03, 0x02, 0x02, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x06, 0x05, 0x02, 0x02, 0x03, 0x02, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x05, 0x05, 0x02, 0x03, 0x02, 0x02, 0x01, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x05, 0x05, 0x03, 0x02, 0x02, 0x01, 0x02, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x05, 0x04, 0x02, 0x02, 0x03, 0x02, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x05, 0x04, 0x02, 0x02, 0x02, 0x01, 0x01, 0x02},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x04, 0x04, 0x02, 0x02, 0x02, 0x01, 0x02, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x04, 0x02, 0x02, 0x02, 0x02, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x04, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x03, 0x02, 0x01, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x03, 0x01, 0x01, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x03, 0x02, 0x02, 0x01, 0x01, 0x00, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x01, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x02, 0x01, 0x01, 0x01, 0x00, 0x00, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x02, 0x02, 0x02, 0x01, 0x01, 0x00, 0x00, 0x00, 0x01},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
};

static char irc_dimming_para_revM [256][20] = {
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x6B, 0x66, 0x5B, 0x23, 0x22, 0x26, 0x10, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x6B, 0x66, 0x5B, 0x23, 0x22, 0x26, 0x10, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x6A, 0x65, 0x5A, 0x23, 0x22, 0x26, 0x10, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x6A, 0x65, 0x5A, 0x22, 0x21, 0x25, 0x10, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x69, 0x65, 0x5A, 0x23, 0x21, 0x25, 0x10, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x69, 0x64, 0x59, 0x22, 0x21, 0x26, 0x10, 0x12, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x68, 0x64, 0x59, 0x23, 0x21, 0x25, 0x0F, 0x11, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x68, 0x63, 0x58, 0x22, 0x21, 0x25, 0x0F, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x67, 0x62, 0x58, 0x22, 0x21, 0x24, 0x0F, 0x12, 0x12},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x67, 0x62, 0x57, 0x21, 0x20, 0x25, 0x10, 0x12, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x66, 0x61, 0x57, 0x22, 0x21, 0x24, 0x0F, 0x11, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x66, 0x61, 0x56, 0x21, 0x20, 0x25, 0x0F, 0x11, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x65, 0x60, 0x56, 0x21, 0x21, 0x24, 0x0F, 0x11, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x65, 0x60, 0x56, 0x21, 0x20, 0x23, 0x0F, 0x11, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x64, 0x5F, 0x55, 0x21, 0x20, 0x24, 0x0F, 0x11, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x64, 0x5F, 0x55, 0x20, 0x20, 0x23, 0x0F, 0x10, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x63, 0x5E, 0x54, 0x21, 0x20, 0x23, 0x0E, 0x11, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x63, 0x5E, 0x54, 0x20, 0x1F, 0x23, 0x0F, 0x11, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x62, 0x5E, 0x53, 0x20, 0x1F, 0x23, 0x0F, 0x10, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x62, 0x5D, 0x53, 0x20, 0x1F, 0x23, 0x0E, 0x11, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x61, 0x5D, 0x53, 0x20, 0x1F, 0x22, 0x0F, 0x10, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x61, 0x5C, 0x52, 0x1F, 0x1F, 0x23, 0x0F, 0x10, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x60, 0x5C, 0x52, 0x20, 0x1E, 0x22, 0x0E, 0x11, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x60, 0x5B, 0x51, 0x1F, 0x1F, 0x22, 0x0E, 0x10, 0x11},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5F, 0x5B, 0x51, 0x1F, 0x1E, 0x22, 0x0F, 0x10, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5F, 0x5A, 0x51, 0x1F, 0x1E, 0x21, 0x0E, 0x10, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5E, 0x5A, 0x50, 0x1F, 0x1E, 0x22, 0x0E, 0x10, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5D, 0x59, 0x50, 0x1F, 0x1E, 0x21, 0x0E, 0x10, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5D, 0x59, 0x4F, 0x1E, 0x1D, 0x21, 0x0E, 0x10, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5C, 0x58, 0x4F, 0x1F, 0x1E, 0x20, 0x0E, 0x0F, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5C, 0x58, 0x4E, 0x1E, 0x1D, 0x21, 0x0E, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5B, 0x57, 0x4E, 0x1E, 0x1D, 0x20, 0x0E, 0x10, 0x10},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5B, 0x57, 0x4D, 0x1E, 0x1D, 0x21, 0x0D, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5A, 0x56, 0x4D, 0x1E, 0x1D, 0x20, 0x0E, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5A, 0x56, 0x4D, 0x1E, 0x1D, 0x20, 0x0D, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5A, 0x55, 0x4C, 0x1D, 0x1D, 0x20, 0x0D, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x59, 0x55, 0x4C, 0x1D, 0x1C, 0x20, 0x0E, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x59, 0x55, 0x4B, 0x1D, 0x1C, 0x20, 0x0D, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x58, 0x54, 0x4B, 0x1D, 0x1C, 0x1F, 0x0D, 0x0F, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x58, 0x54, 0x4B, 0x1C, 0x1B, 0x1F, 0x0D, 0x0F, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x57, 0x53, 0x4A, 0x1D, 0x1C, 0x1F, 0x0D, 0x0E, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x57, 0x53, 0x4A, 0x1C, 0x1B, 0x1F, 0x0D, 0x0F, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x56, 0x52, 0x49, 0x1D, 0x1C, 0x1F, 0x0D, 0x0E, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x56, 0x52, 0x49, 0x1C, 0x1B, 0x1F, 0x0D, 0x0F, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x55, 0x51, 0x49, 0x1C, 0x1B, 0x1E, 0x0D, 0x0F, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x55, 0x51, 0x48, 0x1C, 0x1B, 0x1E, 0x0C, 0x0E, 0x0F},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x54, 0x50, 0x48, 0x1C, 0x1B, 0x1E, 0x0D, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x54, 0x50, 0x47, 0x1B, 0x1B, 0x1E, 0x0D, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x53, 0x4F, 0x47, 0x1C, 0x1B, 0x1D, 0x0C, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x53, 0x4F, 0x46, 0x1B, 0x1A, 0x1E, 0x0C, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x53, 0x4F, 0x46, 0x1B, 0x1A, 0x1E, 0x0C, 0x0E, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x52, 0x4E, 0x46, 0x1B, 0x1A, 0x1D, 0x0C, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x52, 0x4E, 0x45, 0x1A, 0x1A, 0x1D, 0x0C, 0x0D, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x51, 0x4D, 0x45, 0x1B, 0x1A, 0x1D, 0x0C, 0x0E, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x51, 0x4D, 0x44, 0x1A, 0x19, 0x1D, 0x0C, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x50, 0x4C, 0x44, 0x1A, 0x1A, 0x1C, 0x0C, 0x0D, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4F, 0x4C, 0x44, 0x1A, 0x19, 0x1C, 0x0C, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4F, 0x4B, 0x43, 0x1A, 0x19, 0x1C, 0x0C, 0x0E, 0x0E},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4E, 0x4B, 0x43, 0x1A, 0x19, 0x1C, 0x0C, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4E, 0x4A, 0x42, 0x19, 0x19, 0x1C, 0x0C, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4D, 0x4A, 0x42, 0x1A, 0x18, 0x1B, 0x0B, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4D, 0x49, 0x41, 0x19, 0x19, 0x1C, 0x0C, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4C, 0x49, 0x41, 0x19, 0x18, 0x1B, 0x0C, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4C, 0x48, 0x41, 0x19, 0x18, 0x1B, 0x0B, 0x0D, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4B, 0x48, 0x40, 0x19, 0x18, 0x1B, 0x0B, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4B, 0x48, 0x40, 0x19, 0x18, 0x1B, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4B, 0x47, 0x3F, 0x18, 0x18, 0x1B, 0x0B, 0x0C, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4A, 0x47, 0x3F, 0x18, 0x17, 0x1A, 0x0B, 0x0D, 0x0D},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x4A, 0x46, 0x3F, 0x18, 0x18, 0x1A, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x49, 0x46, 0x3E, 0x18, 0x17, 0x1A, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x49, 0x45, 0x3E, 0x18, 0x18, 0x1A, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x48, 0x45, 0x3E, 0x18, 0x17, 0x19, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x48, 0x44, 0x3D, 0x17, 0x17, 0x1A, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x47, 0x44, 0x3D, 0x18, 0x17, 0x19, 0x0A, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x47, 0x44, 0x3C, 0x17, 0x16, 0x19, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x46, 0x43, 0x3C, 0x17, 0x16, 0x19, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x46, 0x43, 0x3B, 0x17, 0x16, 0x19, 0x0A, 0x0B, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x46, 0x42, 0x3B, 0x16, 0x16, 0x19, 0x0B, 0x0C, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x45, 0x42, 0x3B, 0x17, 0x16, 0x18, 0x0A, 0x0B, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x45, 0x41, 0x3A, 0x16, 0x16, 0x19, 0x0A, 0x0C, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x44, 0x41, 0x3A, 0x16, 0x15, 0x18, 0x0A, 0x0C, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x44, 0x40, 0x39, 0x16, 0x16, 0x18, 0x0A, 0x0B, 0x0C},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x43, 0x40, 0x39, 0x16, 0x15, 0x18, 0x0A, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x42, 0x3F, 0x39, 0x16, 0x16, 0x17, 0x0A, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x42, 0x3F, 0x38, 0x16, 0x15, 0x18, 0x0A, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x42, 0x3F, 0x38, 0x15, 0x15, 0x17, 0x0A, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x41, 0x3E, 0x37, 0x16, 0x15, 0x18, 0x09, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x41, 0x3E, 0x37, 0x15, 0x14, 0x17, 0x0A, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x40, 0x3D, 0x37, 0x16, 0x15, 0x17, 0x09, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x40, 0x3D, 0x36, 0x15, 0x14, 0x17, 0x09, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3F, 0x3C, 0x36, 0x15, 0x15, 0x16, 0x0A, 0x0A, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3F, 0x3C, 0x36, 0x15, 0x14, 0x16, 0x09, 0x0B, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3E, 0x3C, 0x35, 0x15, 0x13, 0x16, 0x09, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3E, 0x3B, 0x35, 0x14, 0x14, 0x16, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3E, 0x3B, 0x34, 0x14, 0x13, 0x16, 0x09, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3D, 0x3A, 0x34, 0x14, 0x14, 0x16, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3D, 0x3A, 0x34, 0x13, 0x13, 0x15, 0x0A, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3C, 0x39, 0x33, 0x14, 0x13, 0x15, 0x09, 0x0B, 0x0B},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3C, 0x39, 0x33, 0x13, 0x13, 0x15, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3B, 0x38, 0x32, 0x13, 0x13, 0x15, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3B, 0x38, 0x32, 0x13, 0x13, 0x15, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3A, 0x38, 0x32, 0x13, 0x12, 0x14, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x3A, 0x37, 0x31, 0x13, 0x13, 0x15, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x39, 0x37, 0x31, 0x13, 0x12, 0x14, 0x09, 0x0A, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x39, 0x36, 0x31, 0x13, 0x13, 0x14, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x39, 0x36, 0x30, 0x12, 0x12, 0x14, 0x09, 0x09, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x38, 0x35, 0x30, 0x12, 0x12, 0x14, 0x09, 0x0A, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x38, 0x35, 0x2F, 0x12, 0x12, 0x14, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x37, 0x34, 0x2F, 0x12, 0x12, 0x13, 0x08, 0x09, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x37, 0x34, 0x2E, 0x11, 0x11, 0x14, 0x09, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x36, 0x34, 0x2E, 0x12, 0x11, 0x13, 0x08, 0x09, 0x0A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x36, 0x33, 0x2E, 0x11, 0x11, 0x13, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x35, 0x33, 0x2D, 0x12, 0x11, 0x13, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x35, 0x32, 0x2D, 0x11, 0x11, 0x13, 0x08, 0x09, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x34, 0x32, 0x2C, 0x11, 0x10, 0x13, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x34, 0x32, 0x2C, 0x11, 0x10, 0x13, 0x08, 0x09, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x33, 0x31, 0x2C, 0x11, 0x10, 0x12, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x33, 0x31, 0x2B, 0x11, 0x10, 0x12, 0x07, 0x08, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x32, 0x30, 0x2B, 0x11, 0x10, 0x12, 0x07, 0x09, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x32, 0x30, 0x2B, 0x11, 0x10, 0x12, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x32, 0x2F, 0x2A, 0x10, 0x10, 0x12, 0x07, 0x09, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x31, 0x2F, 0x2A, 0x10, 0x0F, 0x11, 0x08, 0x09, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x31, 0x2F, 0x2A, 0x10, 0x0F, 0x11, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x30, 0x2E, 0x29, 0x10, 0x10, 0x11, 0x07, 0x08, 0x09},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x30, 0x2E, 0x29, 0x10, 0x0F, 0x11, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2F, 0x2D, 0x28, 0x10, 0x0F, 0x11, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2F, 0x2D, 0x28, 0x10, 0x0F, 0x11, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2F, 0x2C, 0x28, 0x0F, 0x0F, 0x10, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2E, 0x2C, 0x27, 0x0F, 0x0F, 0x11, 0x07, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2E, 0x2C, 0x27, 0x0F, 0x0E, 0x10, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2D, 0x2B, 0x27, 0x0F, 0x0F, 0x10, 0x07, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2D, 0x2B, 0x26, 0x0F, 0x0E, 0x10, 0x07, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2D, 0x2B, 0x26, 0x0E, 0x0E, 0x10, 0x07, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2C, 0x2A, 0x26, 0x0F, 0x0E, 0x0F, 0x06, 0x08, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2C, 0x2A, 0x25, 0x0E, 0x0E, 0x10, 0x07, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2B, 0x29, 0x25, 0x0E, 0x0E, 0x0F, 0x07, 0x07, 0x08},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2B, 0x29, 0x24, 0x0E, 0x0D, 0x10, 0x06, 0x08, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2A, 0x28, 0x24, 0x0E, 0x0E, 0x0F, 0x06, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x2A, 0x28, 0x24, 0x0D, 0x0D, 0x0E, 0x07, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x29, 0x27, 0x23, 0x0E, 0x0D, 0x0F, 0x06, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x29, 0x27, 0x23, 0x0D, 0x0D, 0x0E, 0x06, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x29, 0x27, 0x22, 0x0D, 0x0C, 0x0F, 0x06, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x28, 0x26, 0x22, 0x0D, 0x0D, 0x0E, 0x06, 0x07, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x27, 0x26, 0x22, 0x0D, 0x0C, 0x0E, 0x06, 0x07, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x27, 0x25, 0x21, 0x0D, 0x0D, 0x0E, 0x06, 0x06, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x27, 0x25, 0x21, 0x0C, 0x0C, 0x0E, 0x06, 0x07, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x26, 0x25, 0x21, 0x0D, 0x0C, 0x0D, 0x06, 0x06, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x26, 0x24, 0x20, 0x0C, 0x0C, 0x0E, 0x06, 0x07, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x26, 0x24, 0x20, 0x0C, 0x0C, 0x0D, 0x06, 0x06, 0x07},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x25, 0x23, 0x20, 0x0C, 0x0C, 0x0D, 0x06, 0x07, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x25, 0x23, 0x1F, 0x0C, 0x0C, 0x0E, 0x06, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x24, 0x23, 0x1F, 0x0C, 0x0B, 0x0D, 0x06, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x24, 0x22, 0x1F, 0x0C, 0x0C, 0x0C, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x23, 0x22, 0x1E, 0x0C, 0x0B, 0x0D, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x23, 0x22, 0x1E, 0x0C, 0x0B, 0x0C, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x23, 0x21, 0x1D, 0x0B, 0x0B, 0x0D, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x22, 0x21, 0x1D, 0x0B, 0x0A, 0x0C, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x22, 0x20, 0x1D, 0x0B, 0x0B, 0x0C, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x21, 0x20, 0x1C, 0x0B, 0x0A, 0x0C, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x21, 0x1F, 0x1C, 0x0B, 0x0B, 0x0C, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x21, 0x1F, 0x1C, 0x0A, 0x0A, 0x0B, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x20, 0x1F, 0x1B, 0x0B, 0x0A, 0x0C, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x20, 0x1E, 0x1B, 0x0A, 0x0A, 0x0B, 0x05, 0x06, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1F, 0x1E, 0x1B, 0x0B, 0x0A, 0x0B, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1F, 0x1E, 0x1A, 0x0A, 0x0A, 0x0B, 0x05, 0x05, 0x06},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1F, 0x1D, 0x1A, 0x0A, 0x0A, 0x0B, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1E, 0x1D, 0x1A, 0x0A, 0x0A, 0x0B, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1E, 0x1C, 0x19, 0x0A, 0x0A, 0x0B, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1E, 0x1C, 0x19, 0x09, 0x0A, 0x0B, 0x05, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1D, 0x1C, 0x19, 0x0A, 0x09, 0x0A, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1D, 0x1B, 0x18, 0x09, 0x0A, 0x0B, 0x05, 0x04, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1B, 0x18, 0x0A, 0x09, 0x0A, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1B, 0x18, 0x09, 0x09, 0x0A, 0x04, 0x04, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1C, 0x1A, 0x17, 0x09, 0x09, 0x0A, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1B, 0x1A, 0x17, 0x09, 0x09, 0x0A, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1B, 0x1A, 0x17, 0x09, 0x08, 0x09, 0x04, 0x05, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1A, 0x19, 0x16, 0x09, 0x08, 0x0A, 0x04, 0x05, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x1A, 0x19, 0x16, 0x08, 0x08, 0x09, 0x04, 0x04, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x19, 0x18, 0x16, 0x09, 0x08, 0x09, 0x04, 0x05, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x19, 0x18, 0x15, 0x08, 0x08, 0x09, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x18, 0x17, 0x15, 0x08, 0x08, 0x08, 0x04, 0x04, 0x05},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x18, 0x17, 0x15, 0x08, 0x08, 0x08, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x18, 0x17, 0x14, 0x07, 0x07, 0x09, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x16, 0x14, 0x08, 0x08, 0x08, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x16, 0x14, 0x07, 0x07, 0x08, 0x04, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x17, 0x16, 0x13, 0x07, 0x07, 0x08, 0x03, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x16, 0x15, 0x13, 0x07, 0x07, 0x08, 0x04, 0x04, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x16, 0x15, 0x13, 0x07, 0x07, 0x07, 0x03, 0x04, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x16, 0x15, 0x12, 0x07, 0x07, 0x08, 0x03, 0x03, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x15, 0x14, 0x12, 0x07, 0x07, 0x07, 0x03, 0x03, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x15, 0x14, 0x12, 0x07, 0x07, 0x07, 0x03, 0x03, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x13, 0x11, 0x07, 0x07, 0x08, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x13, 0x11, 0x07, 0x07, 0x07, 0x03, 0x03, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x14, 0x13, 0x11, 0x06, 0x06, 0x07, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x13, 0x12, 0x10, 0x07, 0x07, 0x07, 0x03, 0x03, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x13, 0x12, 0x10, 0x06, 0x06, 0x07, 0x03, 0x04, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x13, 0x12, 0x10, 0x06, 0x06, 0x06, 0x02, 0x03, 0x04},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x12, 0x11, 0x10, 0x06, 0x06, 0x06, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x12, 0x11, 0x0F, 0x06, 0x06, 0x07, 0x02, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x11, 0x0F, 0x06, 0x05, 0x06, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x10, 0x0F, 0x06, 0x06, 0x06, 0x02, 0x03, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x11, 0x10, 0x0E, 0x05, 0x05, 0x06, 0x02, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x10, 0x0E, 0x06, 0x05, 0x06, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x0F, 0x0E, 0x05, 0x05, 0x05, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x10, 0x0F, 0x0D, 0x05, 0x05, 0x06, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0F, 0x0F, 0x0D, 0x05, 0x04, 0x05, 0x03, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0F, 0x0E, 0x0D, 0x05, 0x05, 0x05, 0x02, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0F, 0x0E, 0x0C, 0x04, 0x04, 0x06, 0x02, 0x03, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0E, 0x0C, 0x05, 0x04, 0x05, 0x02, 0x03, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0D, 0x0C, 0x05, 0x05, 0x05, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0E, 0x0D, 0x0B, 0x04, 0x04, 0x05, 0x02, 0x02, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0D, 0x0D, 0x0B, 0x05, 0x04, 0x05, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0D, 0x0C, 0x0B, 0x04, 0x04, 0x04, 0x02, 0x02, 0x03},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0C, 0x0B, 0x05, 0x04, 0x04, 0x01, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0B, 0x0A, 0x04, 0x04, 0x04, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0C, 0x0B, 0x0A, 0x04, 0x04, 0x04, 0x01, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0B, 0x0B, 0x0A, 0x04, 0x04, 0x04, 0x02, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0B, 0x0B, 0x0A, 0x04, 0x03, 0x04, 0x02, 0x02, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0B, 0x0A, 0x09, 0x04, 0x04, 0x04, 0x01, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x0A, 0x09, 0x04, 0x03, 0x04, 0x01, 0x02, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x0A, 0x09, 0x04, 0x03, 0x03, 0x01, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x0A, 0x09, 0x08, 0x03, 0x04, 0x04, 0x02, 0x01, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x09, 0x09, 0x08, 0x04, 0x03, 0x03, 0x01, 0x02, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x09, 0x09, 0x08, 0x03, 0x03, 0x03, 0x02, 0x01, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x09, 0x08, 0x07, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x07, 0x03, 0x03, 0x03, 0x01, 0x01, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x07, 0x03, 0x02, 0x03, 0x01, 0x02, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x08, 0x07, 0x02, 0x02, 0x03, 0x02, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x08, 0x07, 0x07, 0x02, 0x03, 0x02, 0x01, 0x01, 0x02},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x07, 0x06, 0x03, 0x02, 0x03, 0x01, 0x02, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x07, 0x06, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x07, 0x06, 0x06, 0x02, 0x02, 0x02, 0x01, 0x02, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x06, 0x05, 0x02, 0x02, 0x03, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x06, 0x05, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x06, 0x05, 0x05, 0x01, 0x02, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x05, 0x05, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x05, 0x04, 0x02, 0x01, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x05, 0x04, 0x01, 0x01, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x05, 0x04, 0x04, 0x01, 0x02, 0x02, 0x01, 0x01, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x04, 0x02, 0x02, 0x01, 0x00, 0x00, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x03, 0x01, 0x01, 0x02, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x04, 0x03, 0x01, 0x01, 0x02, 0x01, 0x01, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x04, 0x03, 0x03, 0x01, 0x02, 0x01, 0x00, 0x00, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x03, 0x02, 0x01, 0x01, 0x01, 0x00, 0x00, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x03, 0x02, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
};

static struct dsi_panel_cmds *mdss_irc(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int para_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (IS_ERR_OR_NULL(vdd->dtsi_data[ctrl->ndx].irc_tx_cmds)) {
		LCD_ERR("No irc_tx_cmds\n");
		return NULL;
	}

	if (!vdd->samsung_support_irc)
		return NULL;

	*level_key = PANEL_LEVE1_KEY;

	if (vdd->ldu_correction_state || vdd->color_weakness_value)
		return &(vdd->dtsi_data[ctrl->ndx].irc_off_tx_cmds[vdd->panel_revision]);

	para_idx = 255 - vdd->bl_level;

	/* copy irc default setting */
	if (mdss_get_hero2_fab_type() == S6E3HF4_AMB526JS01_A2_LINE)
		memcpy(&vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
			a2_line_data.irc_normal[para_idx], 20);
	else {
		if (hero2_a2_revM) {
			memcpy(&vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
					irc_dimming_para_revM[para_idx], 20);
		}
		else if (mdss_panel_rev_get(ctrl) >= 0x07)
			memcpy(&vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
					irc_dimming_para_revJ[para_idx], 20);
		else
			memcpy(&vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
					irc_dimming_para[para_idx], 20);
	}

	/* update irc enable*/
	vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = vdd->irc_info.irc_enable_status;

	/* update irc mode*/
	vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = vdd->irc_info.irc_mode;

	return &(vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision]);
}

static char irc_hbm_para[8][20] = {
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x6B, 0x65, 0x5E, 0x24, 0x24, 0x27, 0x15, 0x18, 0x15},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x67, 0x62, 0x5B, 0x23, 0x22, 0x25, 0x14, 0x18, 0x15},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x63, 0x5E, 0x57, 0x21, 0x21, 0x24, 0x14, 0x16, 0x14},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x5F, 0x5A, 0x54, 0x20, 0x20, 0x22, 0x13, 0x15, 0x13},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x5B, 0x56, 0x50, 0x1E, 0x1F, 0x21, 0x13, 0x14, 0x12},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x57, 0x52, 0x4D, 0x1D, 0x1E, 0x1F, 0x12, 0x13, 0x11},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x53, 0x4F, 0x49, 0x1C, 0x1B, 0x1E, 0x10, 0x13, 0x11},
	{0x01, 0x6D, 0x46, 0x4E, 0x95, 0xEC, 0x33, 0x69, 0x12, 0x7A, 0xFF, 0x4F, 0x4B, 0x46, 0x1A, 0x1A, 0x1C, 0x10, 0x12, 0x10},
};

static char irc_hbm_para_revJ[8][20] = {
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x6E, 0x6B, 0x63, 0x2F, 0x2E, 0x2F, 0x22, 0x1C, 0x1E},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x6A, 0x67, 0x5F, 0x2D, 0x2C, 0x2D, 0x21, 0x1C, 0x1D},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x66, 0x63, 0x5B, 0x2B, 0x2A, 0x2C, 0x20, 0x1B, 0x1C},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x62, 0x5F, 0x58, 0x2A, 0x29, 0x29, 0x1E, 0x19, 0x1B},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x5E, 0x5B, 0x54, 0x28, 0x27, 0x28, 0x1D, 0x18, 0x19},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x59, 0x57, 0x50, 0x27, 0x25, 0x27, 0x1C, 0x18, 0x18},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x55, 0x53, 0x4C, 0x25, 0x23, 0x25, 0x1A, 0x17, 0x17},
	{0x01, 0x6D, 0x53, 0x4A, 0x8C, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x51, 0x4F, 0x49, 0x23, 0x22, 0x23, 0x19, 0x15, 0x16},
};

static char irc_hbm_para_revM[8][20] = {
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x99, 0x92, 0x82, 0x32, 0x30, 0x36, 0x17, 0x1A, 0x1A},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x93, 0x8C, 0x7D, 0x30, 0x2F, 0x35, 0x16, 0x19, 0x18},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x8D, 0x87, 0x78, 0x2F, 0x2D, 0x32, 0x15, 0x18, 0x18},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x88, 0x81, 0x73, 0x2C, 0x2C, 0x31, 0x15, 0x16, 0x17},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x82, 0x7C, 0x6F, 0x2A, 0x29, 0x2E, 0x14, 0x16, 0x16},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x7C, 0x77, 0x6A, 0x29, 0x27, 0x2C, 0x13, 0x15, 0x15},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x76, 0x71, 0x65, 0x27, 0x26, 0x2A, 0x12, 0x14, 0x14},
	{0x01, 0x6D, 0x4A, 0x4E, 0x91, 0xBD, 0x33, 0x69, 0x12, 0x7A, 0xA0, 0x71, 0x6C, 0x60, 0x25, 0x23, 0x28, 0x11, 0x13, 0x13},
};

static struct dsi_panel_cmds *mdss_hbm_irc(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int para_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (IS_ERR_OR_NULL(vdd->dtsi_data[ctrl->ndx].irc_tx_cmds)) {
		LCD_ERR("No irc_tx_cmds\n");
		return NULL;
	}

	if (!vdd->samsung_support_irc)
		return NULL;

	*level_key = PANEL_LEVE1_KEY;

	if (vdd->ldu_correction_state || vdd->color_weakness_value)
		return &(vdd->dtsi_data[ctrl->ndx].irc_off_tx_cmds[vdd->panel_revision]);

	/*auto_brightness is 13 to use 443cd of hbm step on color weakness mode*/
	if (vdd->auto_brightness == HBM_MODE + 7)
		para_idx = 7;
	else
		para_idx = vdd->auto_brightness_level - vdd->auto_brightness;

	/* copy irc default setting */
	if (mdss_get_hero2_fab_type() == S6E3HF4_AMB526JS01_A2_LINE)
		memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
			a2_line_data.irc_hbm[para_idx], 20);
	else {
		if (hero2_a2_revM)
			memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
					irc_hbm_para_revM[para_idx], 20);
		else if (mdss_panel_rev_get(ctrl) >= 0x07)
			memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
					irc_hbm_para_revJ[para_idx], 20);
		else
			memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
					irc_hbm_para[para_idx], 20);
	}

	/* update irc enable*/
	vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = vdd->irc_info.irc_enable_status;

	/* update irc mode*/
	vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = vdd->irc_info.irc_mode;

	return &(vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision]);
}

static struct dsi_panel_cmds * mdss_gamma(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	LCD_DEBUG("bl_level : %d candela : %dCD\n", vdd->bl_level, vdd->candela_level);

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi[ctrl->ndx]->generate_gamma)) {
		LCD_ERR("generate_gamma is NULL error");
		return NULL;
	} else {
		vdd->smart_dimming_dsi[ctrl->ndx]->generate_gamma(
			vdd->smart_dimming_dsi[ctrl->ndx],
			vdd->candela_level,
			&vdd->dtsi_data[ctrl->ndx].gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1]);

		*level_key = PANEL_LEVE1_KEY;

		return &vdd->dtsi_data[ctrl->ndx].gamma_tx_cmds[vdd->panel_revision];
	}
}

#define LUT_SIZE (sizeof(freq_cal_lut_offset) / sizeof(int)) / 3

static int freq_cal_lut_offset[][3] = {
	{56568, 56686,	-25},
	{56686, 56805,	-24},
	{56805, 56924,	-23},
	{56924, 57044,	-22},
	{57044, 57165,	-21},
	{57165, 57286,	-20},
	{57286, 57407,	-19},
	{57407, 57529,	-18},
	{57529, 57651,	-17},
	{57651, 57774,	-16},
	{57774, 57898,	-15},
	{57898, 58022,	-14},
	{58022, 58146,	-13},
	{58146, 58272,	-12},
	{58272, 58397,	-11},
	{58397, 58523,	-10},
	{58523, 58650,	-9},
	{58650, 58777,	-8},
	{58777, 58905,	-7},
	{58905, 59034,	-6},
	{59034, 59163,	-5},
	{59163, 59292,	-4},
	{59292, 59422,	-3},
	{59422, 59553,	-2},
	{59553, 59684,	-1},
	{59684, 59816,	0},
	{59816, 59948,	1},
	{59948, 60081,	2},
	{60081, 60215,	3},
	{60215, 60349,	4},
	{60349, 60484,	5},
	{60484, 60619,	6},
	{60619, 60755,	7},
	{60755, 60892,	8},
	{60892, 61029,	9},
	{61029, 61167,	10},
	{61167, 61305,	11},
	{61305, 61444,	12},
	{61444, 61584,	13},
	{61584, 61724,	14},
	{61724, 61865,	15},
	{61865, 62007,	16},
	{62007, 62149,	17},
	{62149, 62292,	18},
	{62292, 62436,	19},
	{62436, 62580,	20},
	{62580, 62725,	21},
	{62725, 62871,	22},
	{62871, 63017,	23},
	{63017, 63164,	24},
	{63164, 63312,	25},
	{63312, 63460,	26},
	{63460, 63609,	27},
	{63609, 63759,	28},
	{63759, 63909,	29},
	{63909, 64010,	30},
};

#define FPS_CMD_INDEX 2

static int mdss_samsung_change_ldi_fps(struct mdss_dsi_ctrl_pdata *ctrl, unsigned int input_fps)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	int offset = 0;
	int i;
	int dest_cal_val;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return 0;
	}

	LCD_DEBUG("input_fps (%d), lut size (%d)\n", input_fps, (int)LUT_SIZE);

	if (mdss_panel_rev_get(ctrl) <= 0x01) {
		LCD_ERR("LDI EVT0 Not Support. Skip!!\n");
		return 0;
	}

	for (i = 0; i < LUT_SIZE; i++) {
		if (input_fps >= freq_cal_lut_offset[i][0] &&
			input_fps < freq_cal_lut_offset[i][1]) {
			offset = freq_cal_lut_offset[i][2];
			break;
		}
	}

	if (i == LUT_SIZE) {
		LCD_ERR("can not find offset !!\n");
		return 0;
	}

	LCD_DEBUG("current comp value(0x%x),offset(%d)\n",
		vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3], offset);

	dest_cal_val = vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3] + offset;

	if((dest_cal_val < 0xAC) || (dest_cal_val > 0xE3)) {
		LCD_ERR("Invalid cal value(0x%x)", dest_cal_val);
		return 0;
	} else
		LCD_INFO("input : %d dest write value (0x%x)\n", input_fps, dest_cal_val);

	vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3] = dest_cal_val;

	return 1;
}

static int init_ldi_fps(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return 0;
	}

	if (ctrl->cmd_sync_wait_broadcast)
		if (!ctrl->cmd_sync_wait_trigger)
			return 0;

	vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3] = 0xC5;
	LCD_ERR("(0x%x)\n", vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3]);

	return 1;
}

static int samsung_panel_off_post(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int rc = 0;


	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	return rc;
}

static struct dsi_panel_cmds * mdss_gamma_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	LCD_INFO("hmt_bl_level : %d candela : %dCD\n", vdd->hmt_stat.hmt_bl_level, vdd->hmt_stat.candela_level_hmt);

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi_hmt[ctrl->ndx]->generate_gamma)) {
		LCD_ERR("generate_gamma is NULL");
		return NULL;
	} else {
		vdd->smart_dimming_dsi_hmt[ctrl->ndx]->generate_gamma(
			vdd->smart_dimming_dsi_hmt[ctrl->ndx],
			vdd->hmt_stat.candela_level_hmt,
			&vdd->dtsi_data[ctrl->ndx].hmt_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1]);

		*level_key = PANEL_LEVE1_KEY;

		return &vdd->dtsi_data[ctrl->ndx].hmt_gamma_tx_cmds[vdd->panel_revision];
	}
}

static struct dsi_panel_cmds hmt_aid_cmd;
static struct dsi_panel_cmds *mdss_aid_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (!vdd->dtsi_data[ctrl->ndx].hmt_reverse_aid_map_table[vdd->panel_revision].size ||
		vdd->hmt_stat.cmd_idx_hmt > vdd->dtsi_data[ctrl->ndx].hmt_reverse_aid_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].hmt_reverse_aid_map_table[vdd->panel_revision].cmd_idx[vdd->hmt_stat.cmd_idx_hmt];

	hmt_aid_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].hmt_aid_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	hmt_aid_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &hmt_aid_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static struct dsi_panel_cmds *mdss_elvss_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (vdd->acl_status || vdd->siop_status) {
		vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = 0x8C;
	} else {
		vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = 0x9C;
	}

	if (vdd->bl_level < 0 || vdd->bl_level > 255)
		vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = 0x00;

	*level_key = PANEL_LEVE1_KEY;

	return &vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision];
}

static struct dsi_panel_cmds *mdss_vint_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	return &vdd->dtsi_data[ctrl->ndx].hmt_vint_tx_cmds[vdd->panel_revision];
}

static void mdss_make_sdimconf_hmt(struct mdss_dsi_ctrl_pdata *ctrl, struct samsung_display_driver_data *vdd) {
	/* Set the mtp read buffer pointer and read the NVM value*/
	mdss_samsung_panel_data_read(ctrl,
				&(vdd->dtsi_data[ctrl->ndx].smart_dimming_mtp_rx_cmds[vdd->panel_revision]),
				vdd->smart_dimming_dsi_hmt[ctrl->ndx]->mtp_buffer, PANEL_LEVE1_KEY);

	/* Initialize smart dimming related things here */
	/* lux_tab setting for 350cd */
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->lux_tab = vdd->dtsi_data[ctrl->ndx].hmt_candela_map_table[vdd->panel_revision].lux_tab;
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->lux_tabsize = vdd->dtsi_data[ctrl->ndx].hmt_candela_map_table[vdd->panel_revision].lux_tab_size;
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->man_id = vdd->manufacture_id_dsi[ctrl->ndx];

	/* Just a safety check to ensure smart dimming data is initialised well */
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->init(vdd->smart_dimming_dsi_hmt[ctrl->ndx]);

	LCD_INFO("[HMT] smart dimming done!\n");
}

static int mdss_samart_dimming_init_hmt(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	LCD_INFO("DSI%d : ++\n", ctrl->ndx);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	vdd->smart_dimming_dsi_hmt[ctrl->ndx] = vdd->panel_func.samsung_smart_get_conf_hmt();

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi_hmt[ctrl->ndx])) {
		LCD_ERR("DSI%d error", ctrl->ndx);
		return false;
	} else {
		vdd->hmt_stat.hmt_on = 0;
		vdd->hmt_stat.hmt_bl_level = 0;
		vdd->hmt_stat.hmt_reverse = 0;
		vdd->hmt_stat.hmt_is_first = 1;

		mdss_make_sdimconf_hmt(ctrl, vdd);

		vdd->smart_dimming_hmt_loaded_dsi[ctrl->ndx] = true;
	}

	LCD_INFO("DSI%d : --\n", ctrl->ndx);

	return true;
}

/*
 * This function will update parameters for ALPM_REG/ALPM_CTRL_REG
 * Parameter for ALPM_REG : Control brightness for panel LPM
 * Parameter for ALPM_CTRL_REG : Change panel LPM mode for ALPM/HLPM
 */
static int mdss_update_panel_lpm_cmds(struct mdss_dsi_ctrl_pdata *ctrl, int bl_level, int mode)
{
	struct samsung_display_driver_data *vdd = NULL;
	struct dsi_panel_cmds *alpm_brightness[PANEL_LPM_BRIGHTNESS_MAX] = {NULL, };
	struct dsi_panel_cmds *alpm_ctrl[MAX_LPM_CTRL] = {NULL, };
	struct dsi_panel_cmds *cmd_list[2];
	/*
	 * Init reg_list and cmd list
	 * reg_list[X][0] is reg value
	 * reg_list[X][1] is offset for reg value
	 * cmd_list is the target cmds for searching reg value
	 */
	static int reg_list[2][2] = {
		{ALPM_REG, -EINVAL},
		{ALPM_CTRL_REG, -EINVAL}};

	if (IS_ERR_OR_NULL(ctrl))
		goto end;

	vdd = check_valid_ctrl(ctrl);

	cmd_list[0] = &vdd->dtsi_data[ctrl->ndx].alpm_on_tx_cmds[vdd->panel_revision];
	cmd_list[1] = &vdd->dtsi_data[ctrl->ndx].alpm_on_tx_cmds[vdd->panel_revision];

	/* Init alpm_brightness and alpm_ctrl cmds */
	alpm_brightness[PANEL_LPM_2NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_2nit_tx_cmds[vdd->panel_revision];
	alpm_brightness[PANEL_LPM_40NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_40nit_tx_cmds[vdd->panel_revision];
	alpm_brightness[PANEL_LPM_60NIT] = NULL;

	alpm_ctrl[ALPM_CTRL_2NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_alpm_2nit_tx_cmds[vdd->panel_revision];
	alpm_ctrl[ALPM_CTRL_40NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_alpm_40nit_tx_cmds[vdd->panel_revision];
	alpm_ctrl[HLPM_CTRL_2NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_hlpm_2nit_tx_cmds[vdd->panel_revision];
	alpm_ctrl[HLPM_CTRL_40NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_hlpm_40nit_tx_cmds[vdd->panel_revision];


	/*
	 * Find offset for alpm_reg and alpm_ctrl_reg
	 * alpm_reg		 : Control register for ALPM/HLPM on/off
	 * alpm_ctrl_reg : Control register for changing ALPM/HLPM mode
	 */
	mdss_init_panel_lpm_reg_offset(ctrl, reg_list, cmd_list,
			sizeof(cmd_list) / sizeof(cmd_list[0]));

	if (reg_list[0][1] != -EINVAL) {
		/* Update parameter for ALPM_REG */
		memcpy(cmd_list[0]->cmds[reg_list[0][1]].payload,
				alpm_brightness[bl_level]->cmds[0].payload,
				sizeof(char) * cmd_list[0]->cmds[reg_list[0][1]].dchdr.dlen);

		LCD_DEBUG("[Panel LPM] change brightness cmd : %x, %x\n",
				cmd_list[0]->cmds[reg_list[0][1]].payload[1],
				alpm_brightness[bl_level]->cmds[0].payload[1]);

	}

	if (reg_list[1][1] != -EINVAL) {
		/* Initialize ALPM/HLPM cmds */
		switch (bl_level) {
			case PANEL_LPM_40NIT:
				mode = (mode == ALPM_MODE_ON) ? ALPM_CTRL_40NIT :
					(mode == HLPM_MODE_ON) ? HLPM_CTRL_40NIT : ALPM_CTRL_40NIT;
				break;
			case PANEL_LPM_2NIT:
			default:
				mode = (mode == ALPM_MODE_ON) ? ALPM_CTRL_2NIT :
					(mode == HLPM_MODE_ON) ? HLPM_CTRL_2NIT : ALPM_CTRL_2NIT;
				break;
		}

		/* Update parameter for ALPM_CTRL_REG */
		memcpy(cmd_list[1]->cmds[reg_list[1][1]].payload,
				alpm_ctrl[mode]->cmds[0].payload,
				sizeof(char) * cmd_list[1]->cmds[reg_list[1][1]].dchdr.dlen);
		LCD_DEBUG("[Panel LPM] update alpm ctrl reg(%d)\n", mode);
	}

end:
	return 0;
}

static void mdss_get_panel_lpm_mode(struct mdss_dsi_ctrl_pdata *ctrl, u8 *mode)
{
	struct samsung_display_driver_data *vdd = NULL;
	int panel_lpm_mode = 0, lpm_bl_level = 0;

	if (IS_ERR_OR_NULL(ctrl))
		return;

	/*
	 * if the mode value is lower than MAX_LPM_MODE
	 * this function was not called by mdss_samsung_alpm_store()
	 * so the mode will not be changed
	 */
	if (*mode < MAX_LPM_MODE)
		return;

	vdd = check_valid_ctrl(ctrl);

	/* default Hz is 30Hz */
	vdd->panel_lpm.hz = PANEL_LPM_30HZ;

	/* Check mode and bl_level */
	switch (*mode) {
		case AOD_MODE_ALPM_2NIT_ON:
			panel_lpm_mode = ALPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_2NIT;
			break;
		case AOD_MODE_HLPM_2NIT_ON:
			panel_lpm_mode = HLPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_2NIT;
			break;
		case AOD_MODE_ALPM_40NIT_ON:
			panel_lpm_mode = ALPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_40NIT;
			/* 2Hz can be supported after REV D */
			/*
			if (vdd->panel_revision >= (int)'F' - (int)'A')
				vdd->panel_lpm.hz = PANEL_LPM_2HZ;
			else
				vdd->panel_lpm.hz = PANEL_LPM_1HZ;
			*/
			break;
		case AOD_MODE_HLPM_40NIT_ON:
			panel_lpm_mode = HLPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_40NIT;
			break;
		default:
			panel_lpm_mode = MODE_OFF;
			break;
	}

	*mode = panel_lpm_mode;

	/* Save mode and bl_level */
	vdd->panel_lpm.lpm_bl_level = lpm_bl_level;

	mdss_update_panel_lpm_cmds(ctrl, lpm_bl_level, panel_lpm_mode);
}

static int ddi_hw_cursor(struct mdss_dsi_ctrl_pdata *ctrl, int *input)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	char *payload;

	if (IS_ERR_OR_NULL(ctrl)) {
		LCD_ERR("dsi_ctrl is NULL\n");
		return 0;
	}

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return 0;
	}

	if (IS_ERR_OR_NULL(input)) {
		LCD_ERR("input is NULL\n");
		return 0;
	}

	if (IS_ERR_OR_NULL(vdd->dtsi_data[ctrl->ndx].hw_cursor_tx_cmds[vdd->panel_revision].cmds[0].payload)) {
		LCD_ERR("hw_cursor_tx_cmds is NULL\n");
		return 0;
	}


	LCD_INFO("On/Off:(%d), Por/Land:(%d), On_Select:(%d), X:(%d), Y:(%d), Width:(%d), Length:(%d), Color:(0x%x), Period:(%x), TR_TIME(%x)\n",
		input[0], input[1], input[2], input[3], input[4], input[5],
		input[6], input[7], input[8], input[9]);

	payload = vdd->dtsi_data[ctrl->ndx].hw_cursor_tx_cmds[vdd->panel_revision].cmds[0].payload;

	/* Cursor On&Off (0:Off, 1:On) */
	payload[1] = input[0] & 0x1;

	/* 3rd bit : CURSOR_ON_SEL, 2nd bit : Port/Land, 1st bit : BLINK_ON(Default On)*/
	payload[2] = (input[2] & 0x1) << 2 | (input[1] & 0x1) << 1 | 0x1;

	/* Start X address */
	payload[3] = (input[3] & 0x700) >> 8;
	payload[4] = input[3] & 0xFF;

	/* Start Y address */
	payload[5] = (input[4] & 0x700) >> 8;
	payload[6] = input[4] & 0xFF;

	/* Width */
	payload[7] = input[5] & 0xF;

	/* Length */
	payload[8] = (input[6] & 0x100) >> 8;
	payload[9] = input[6] & 0xFF;

	/* Color */
	payload[10] = (input[7] & 0xFF0000) >> 16;
	payload[11] = (input[7] & 0xFF00) >> 8;
	payload[12] = input[7] & 0xFF;

	/* Period */
	payload[13] = input[8] & 0xFF;

	/* TR_TIME */
	payload[14] = input[9] & 0xFF;

	mdss_samsung_send_cmd(ctrl, PANEL_LEVE1_KEY_ENABLE);
	mdss_samsung_send_cmd(ctrl, PANEL_HW_CURSOR);
	mdss_samsung_send_cmd(ctrl, PANEL_LEVE1_KEY_DISABLE);

	return 1;
}

static int parse_basic_data(char *buf)
{
	int ret;
	int idx;
	unsigned int br, base_br;
	int aid[2];

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %x, %x",
						&idx, &br, &base_br,
						&aid[0], &aid[1]);


	if (ret != BASIC_DATA_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	a2_line_data.base_luminance[idx][0] = br;
	a2_line_data.base_luminance[idx][1] = base_br << 22 /*BIT_SHIFT*/;

	a2_line_data.base_luminance_aid[idx][0] = (char)aid[0];
	a2_line_data.base_luminance_aid[idx][1] = (char)aid[1];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), br(%d), base_br(%d), aid[0](%x), aid[1](%x)\n",
				idx, br, base_br, aid[0], aid[1]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_ref_shift(char *buf)
{
	int ret;
	int idx;
	int shift[9];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %d, %d, %d, %d, %d, %d, %d",
						&idx, &shift[0], &shift[1], &shift[2], &shift[3], &shift[4],
						&shift[5], &shift[6], &shift[7], &shift[8]);

	if (ret != REF_SHIFT_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < REF_SHIFT_ELEMENT_CNT-1; i++)
		a2_line_data.gradation_offset[idx][i] = shift[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %d, %d, %d, %d, %d, %d, %d, %d, %d\n",
				idx, shift[0], shift[1], shift[2], shift[3], shift[4],
						shift[5], shift[6], shift[7], shift[8]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_color_shift(char *buf)
{
	int ret;
	int idx;
	int shift[27];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %d, %d, %d, %d, %d, %d, %d,\
							%d, %d, %d, %d, %d, %d, %d, %d, %d,\
							%d, %d, %d, %d, %d, %d, %d, %d, %d",
						&idx, &shift[0], &shift[1], &shift[2], &shift[3], &shift[4],
							  &shift[5], &shift[6], &shift[7], &shift[8], &shift[9],
							  &shift[10], &shift[11], &shift[12], &shift[13], &shift[14],
							  &shift[15], &shift[16], &shift[17], &shift[18], &shift[19],
							  &shift[20], &shift[21], &shift[22], &shift[23], &shift[24],
							  &shift[25], &shift[26]);

	if (ret != COLOR_SHIFT_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < COLOR_SHIFT_ELEMENT_CNT-1; i++)
		a2_line_data.rgb_offset[idx][i] = shift[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d\n",
				idx, shift[0], shift[1], shift[2], shift[3], shift[4],
					  shift[5], shift[6], shift[7], shift[8], shift[9],
					  shift[10], shift[11], shift[12], shift[13], shift[14],
					  shift[15], shift[16], shift[17], shift[18], shift[19],
					  shift[20], shift[21], shift[22], shift[23], shift[24],
					  shift[25], shift[26]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_elvss_low_temp(char *buf)
{
	int ret;
	int idx;
	int elvss[5];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %d, %d, %d",
						&idx, &elvss[0], &elvss[1], &elvss[2], &elvss[3], &elvss[4]);

	if (ret != ELVSS_LOW_TEMP_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < ELVSS_LOW_TEMP_ELEMENT_CNT-1; i++)
		a2_line_data.elvss_offset[idx][i] = (char)elvss[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %d, %d, %d, %d, %d\n",
				idx, elvss[0], elvss[1], elvss[2], elvss[3], elvss[4]);
#endif
	LCD_DEBUG("--\n");

	return ret;
}

static int parse_elvss_data(char *buf)
{
	int ret;
	int idx;
	int elvss_data;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %x",
						&idx, &elvss_data);

	if (ret != ELVSS_DATA_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	a2_line_data.elvss_data[idx] = (char)elvss_data;

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d),  %x",
						idx, elvss_data);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_vint_tbl(char *buf)
{
	int ret;
	int idx;
	int vint[10];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %x, %x, %x, %x, %x,\
							%x, %x, %x, %x, %x",
						&idx, &vint[0], &vint[1], &vint[2], &vint[3], &vint[4],
								&vint[5], &vint[6], &vint[7], &vint[8], &vint[9]);

	if (ret != VINT_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < VINT_ELEMENT_CNT-1; i++)
		a2_line_data.vint[i] = (char)vint[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d),  %x, %x, %x, %x, %x, %x, %x, %x, %x, %x",
						idx, vint[0], vint[1], vint[2], vint[3], vint[4],
							vint[5], vint[6], vint[7], vint[8], vint[9]);
#endif
	LCD_DEBUG("--\n");

	return ret;
}

static int parse_aor_interpolation(char *buf)
{
	int ret;
	int idx;
	int aor[3];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %x, %x",
						&idx, &aor[0], &aor[1], &aor[2]);

	if (ret != AOR_INTERPOLATION_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < AOR_INTERPOLATION_ELEMENT_CNT-2; i++)
		a2_line_data.aor_interpolation[idx][i] = (char)aor[i+1];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d),  %d, %x, %x",
						idx, aor[0], aor[1], aor[2]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_hmt_basic_data(char *buf)
{
	int ret;
	int idx;
	unsigned int br, base_br;
	int aid[2];

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %x, %x",
						&idx, &br, &base_br,
						&aid[0], &aid[1]);

	if (ret != BASIC_DATA_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	a2_line_data.hmt_base_luminance[idx][0] = br;
	a2_line_data.hmt_base_luminance[idx][1] = base_br << 22 /*BIT_SHIFT*/;

	a2_line_data.hmt_base_luminance_aid[idx][0] = (char)aid[0];
	a2_line_data.hmt_base_luminance_aid[idx][1] = (char)aid[1];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), br(%d), base_br(%d)), aid[0](%x), aid[1](%x)\n",
				idx, br, base_br, aid[0], aid[1]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_hmt_ref_shift(char *buf)
{
	int ret;
	int idx;
	int shift[9];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %d, %d, %d, %d, %d, %d, %d",
						&idx, &shift[0], &shift[1], &shift[2], &shift[3], &shift[4],
						&shift[5], &shift[6], &shift[7], &shift[8]);

	if (ret != REF_SHIFT_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < REF_SHIFT_ELEMENT_CNT-1; i++)
		a2_line_data.hmt_gradation_offset[idx][i] = shift[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %d, %d, %d, %d, %d, %d, %d, %d, %d\n",
				idx, shift[0], shift[1], shift[2], shift[3], shift[4],
						shift[5], shift[6], shift[7], shift[8]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_hmt_color_shift(char *buf)
{
	int ret;
	int idx;
	int shift[27];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %d, %d, %d, %d, %d, %d, %d, %d, %d,\
							%d, %d, %d, %d, %d, %d, %d, %d, %d,\
							%d, %d, %d, %d, %d, %d, %d, %d, %d",
						&idx, &shift[0], &shift[1], &shift[2], &shift[3], &shift[4],
							  &shift[5], &shift[6], &shift[7], &shift[8], &shift[9],
							  &shift[10], &shift[11], &shift[12], &shift[13], &shift[14],
							  &shift[15], &shift[16], &shift[17], &shift[18], &shift[19],
							  &shift[20], &shift[21], &shift[22], &shift[23], &shift[24],
							  &shift[25], &shift[26]);

	if (ret != COLOR_SHIFT_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < COLOR_SHIFT_ELEMENT_CNT-1; i++)
		a2_line_data.hmt_rgb_offset[idx][i] = shift[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d\n",
				idx, shift[0], shift[1], shift[2], shift[3], shift[4],
					  shift[5], shift[6], shift[7], shift[8], shift[9],
					  shift[10], shift[11], shift[12], shift[13], shift[14],
					  shift[15], shift[16], shift[17], shift[18], shift[19],
					  shift[20], shift[21], shift[22], shift[23], shift[24],
					  shift[25], shift[26]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_irc_normal(char *buf)
{
	int ret;
	int idx;
	int shift[20];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %x, %x, %x, %x, %x, %x, %x, %x, %x,\
							%x, %x, %x, %x, %x, %x, %x, %x, %x,\
							%x, %x",
						&idx, &shift[0], &shift[1], &shift[2], &shift[3], &shift[4],
							  &shift[5], &shift[6], &shift[7], &shift[8], &shift[9],
							  &shift[10], &shift[11], &shift[12], &shift[13], &shift[14],
							  &shift[15], &shift[16], &shift[17], &shift[18], &shift[19]);

	if (ret != IRC_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret );
		return 0;
	}

	for (i = 0; i < IRC_ELEMENT_CNT-1; i++)
		a2_line_data.irc_normal[idx][i] = (char)shift[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x\n",
				idx, shift[0], shift[1], shift[2], shift[3], shift[4],
					  shift[5], shift[6], shift[7], shift[8], shift[9],
					  shift[10], shift[11], shift[12], shift[13], shift[14],
					  shift[15], shift[16], shift[17], shift[18], shift[19]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_irc_hbm(char *buf)
{
	int ret;
	int idx;
	int shift[20];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %x, %x, %x, %x, %x, %x, %x, %x, %x,\
							%x, %x, %x, %x, %x, %x, %x, %x, %x,\
							%x, %x",
						&idx, &shift[0], &shift[1], &shift[2], &shift[3], &shift[4],
							  &shift[5], &shift[6], &shift[7], &shift[8], &shift[9],
							  &shift[10], &shift[11], &shift[12], &shift[13], &shift[14],
							  &shift[15], &shift[16], &shift[17], &shift[18], &shift[19]);


	if (ret != IRC_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < IRC_ELEMENT_CNT-1; i++)
		a2_line_data.irc_hbm[idx][i] = (char)shift[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d), %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x, %x\n",
				idx, shift[0], shift[1], shift[2], shift[3], shift[4],
					  shift[5], shift[6], shift[7], shift[8], shift[9],
					  shift[10], shift[11], shift[12], shift[13], shift[14],
					  shift[15], shift[16], shift[17], shift[18], shift[19]);
#endif

	LCD_DEBUG("--\n");

	return ret;
}

static int parse_hbm_elvss(char *buf)
{
	int ret;
	int idx;
	int helvss[8];
	int i;

	LCD_DEBUG("++\n");

	ret = sscanf(buf, "%d, %x, %x, %x, %x, %x,\
							%x, %x, %x",
						&idx, &helvss[0], &helvss[1], &helvss[2], &helvss[3], &helvss[4],
								&helvss[5], &helvss[6], &helvss[7]);

	if (ret != HBM_ELVSS_ELEMENT_CNT) {
		LCD_ERR("invalid arg.. ret(%d)\n", ret);
		return 0;
	}

	for (i = 0; i < HBM_ELVSS_ELEMENT_CNT-1; i++)
		a2_line_data.hbm_elvss[i] = (char)helvss[i];

#ifdef OTHER_LINE_PANEL_SUPPORT_DEBUG
	LCD_ERR("idx(%d),  %x, %x, %x, %x, %x, %x, %x, %x",
						idx, helvss[0], helvss[1], helvss[2], helvss[3], helvss[4],
								helvss[5], helvss[6], helvss[7]);
#endif
	LCD_DEBUG("--\n");

	return ret;
}

#define PARSE_LIST_IDX 13
int search_fn_idx(char *read_buf, int read_size, int *fn_idx, int *dtsi_data)
{
	int i;
	int ret = 0;

	struct OtherLine_ParseList parse_list[PARSE_LIST_IDX]= {
		{"basic data", 0},
		{"reference shift data", 0},
		{"color shift data", 0},
		{"elvss low temperature", 0},
		{"elvss data", 0},
		{"vint data", 0},
		{"aor interpolation", 0},
		{"hmt basic data", 0},
		{"hmt reference shift data", 0},
		{"hmt color shift data", 0},
		{"normal irc", 0},
		{"hbm irc", 0},
		{"hbm elvss", 0},
	};

	for (i=0; i < PARSE_LIST_IDX; i++) {
		if (!strncmp(read_buf, parse_list[i].title, read_size)) {
			LCD_ERR("%s ! fx_idx(%d) dtsi_data(%d) \n", parse_list[i].title, *fn_idx, *dtsi_data);
			*fn_idx = i;
			*dtsi_data = parse_list[i].dtsi_data;
			return 1;
		}
	}

	return ret;
}

static void set_otherline_panel_data(struct samsung_display_driver_data *vdd)
{
	int i;
	int vint_cmd_size, elvss_data_size;

	/*vint: index : 0~9 */
	vint_cmd_size = vdd->dtsi_data[DISPLAY_1].vint_tx_cmds[vdd->panel_revision].cmd_cnt;
	for (i = 0; i < vint_cmd_size; i++)
		vdd->dtsi_data[DISPLAY_1].vint_tx_cmds[vdd->panel_revision].cmds[9-i].payload[2] = a2_line_data.vint[i];

	/*elvss data*/
	elvss_data_size = vdd->dtsi_data[DISPLAY_1].elvss_tx_cmds[vdd->panel_revision].cmd_cnt;
	for (i = 0; i < elvss_data_size ; i++)
		vdd->dtsi_data[DISPLAY_1].elvss_tx_cmds[vdd->panel_revision].cmds[i].payload[3] = a2_line_data.elvss_data[i];

	/*hmt_aid_tx_cmd*/
	for (i = 0; i < Hmt_Luminance; i++){
		vdd->dtsi_data[DISPLAY_1].hmt_aid_tx_cmds[vdd->panel_revision].cmds[i].payload[1] = a2_line_data.hmt_base_luminance_aid[i][0];
		vdd->dtsi_data[DISPLAY_1].hmt_aid_tx_cmds[vdd->panel_revision].cmds[i].payload[2] = a2_line_data.hmt_base_luminance_aid[i][1];
	}

	/*aor interpolation*/
	for (i = 0; i < Gray_Scale_Max; i++){
		vdd->dtsi_data[DISPLAY_1].aid_subdivision_tx_cmds[vdd->panel_revision].cmds[i].payload[1] = a2_line_data.aor_interpolation[i][0];
		vdd->dtsi_data[DISPLAY_1].aid_subdivision_tx_cmds[vdd->panel_revision].cmds[i].payload[2] = a2_line_data.aor_interpolation[i][1] ;
	}
	LCD_ERR("-- vint_table_size(%d) elvss_table_size(%d)\n", vint_cmd_size, elvss_data_size);
}

int mdss_parsing_otherline_pdata(struct file *f, struct samsung_display_driver_data *vdd,
		char *src, int len)
{
	int pos = 0;
	int read_size;
	int fn_idx, dtsi;
	char read_buf[MAX_READ_LINE_SIZE];
	int ret = 0;
	int ndx;
	int invalid_arg = 0;

	int (*parse_fn[PARSE_LIST_IDX])(char *) = {
		parse_basic_data,
		parse_ref_shift,
		parse_color_shift,
		parse_elvss_low_temp,
		parse_elvss_data,
		parse_vint_tbl,
		parse_aor_interpolation,
		parse_hmt_basic_data,
		parse_hmt_ref_shift,
		parse_hmt_color_shift,
		parse_irc_normal,
		parse_irc_hbm,
		parse_hbm_elvss,
	};

	LCD_ERR("++\n");

	ndx = display_ndx_check(vdd->ctrl_dsi[DSI_CTRL_0]);

	while (pos < len) {
		memset(read_buf, 0, MAX_READ_LINE_SIZE);

		read_size = read_line(src, read_buf, &pos, len);
		read_buf[MAX_READ_LINE_SIZE-1] = 0;

		LCD_INFO("read_buf (%s) cnt (%d) pos(%d)\n", read_buf, read_size, pos);

		if (!search_fn_idx(read_buf, read_size, &fn_idx, &dtsi))
			if (!parse_fn[fn_idx](read_buf)){
				invalid_arg = 1;
				mdss_set_hero2_fab_type(S6E3HF4_AMB526JS01_A3_LINE); /*to work as a3 line panel*/
				goto end;
			}
/*
		if (!ret)
			goto end;
*/
		if (pos >= len) {
			LCD_INFO("End of File (%d) / (%d)\n", pos, len);
			goto end;
		}
	}

end:
	if (invalid_arg == 0){
		mdss_set_hero2_fab_type(S6E3HF4_AMB526JS01_A2_LINE);
		set_otherline_panel_data(vdd);
		set_otherline_dimming_data();
	}
	LCD_ERR("-- (%d)\n", ret);

	return ret;
}

static void mdss_send_colorweakness_ccb_cmd(struct samsung_display_driver_data *vdd, int mode)
{
	LCD_INFO("mode (%d) color_weakness_value (%x) \n", mode, vdd->color_weakness_value);

	if (mode) {
		vdd->dtsi_data[DISPLAY_1].ccb_on_tx_cmds[vdd->panel_revision].cmds[1].payload[1] = vdd->color_weakness_value;
		mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_COLOR_WEAKNESS_ENABLE);
	} else
		mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_COLOR_WEAKNESS_DISABLE);
}

static int default_res_set = 0;

static void mdss_panel_multires(struct samsung_display_driver_data *vdd)
{
	if (is_boot_recovery && !default_res_set) {
		LCD_INFO("recovery booting!! set to default : %s\n", vdd->resolution_default);
		sec_set_param(param_index_lcd_resolution,  vdd->resolution_default);
		default_res_set = 1;
	}

	if(vdd->multires_stat.prev_mode != vdd->multires_stat.curr_mode)
	{
		LCD_INFO("vdd->multires_stat.prev_mode = %d, vdd-multires_stat.curr_mode = %d\n",
			vdd->multires_stat.prev_mode, vdd->multires_stat.curr_mode);

		switch(vdd->multires_stat.curr_mode)
		{
			case 0:
				sec_set_param(param_index_lcd_resolution, "WQHD");
				break;
			case 1:
				sec_set_param(param_index_lcd_resolution, "FHD");
				break;
			case 2:
				sec_set_param(param_index_lcd_resolution, "HD");
				break;

		}
		mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_DISPLAY_OFF);
		vdd->multires_stat.black_frame_cnt = 5;

		usleep_range(20000, 20000);

		if(vdd->multires_stat.curr_mode == MULTIRES_FHD)
			mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_FHD);
		else if(vdd->multires_stat.curr_mode == MULTIRES_HD)
			mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_HD);
		else if(vdd->multires_stat.curr_mode == MULTIRES_WQHD)
		{
			if(vdd->multires_stat.prev_mode == MULTIRES_FHD)
				mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_FHD_TO_WQHD);
			else
				mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_HD_TO_WQHD);
		}
		vdd->multires_stat.prev_mode = vdd->multires_stat.curr_mode;
	}
}

static void dsi_update_mdnie_data(void)
{
	/* Update mdnie command */
	mdnie_data.DSI0_COLOR_BLIND_MDNIE_1 = DSI0_COLOR_BLIND_MDNIE_1;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_1 = DSI0_RGB_SENSOR_MDNIE_1;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_2 = DSI0_RGB_SENSOR_MDNIE_2;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_3 = DSI0_RGB_SENSOR_MDNIE_3;
	mdnie_data.DSI0_TRANS_DIMMING_MDNIE = DSI0_RGB_SENSOR_MDNIE_3;
	mdnie_data.DSI0_UI_DYNAMIC_MDNIE_2 = DSI0_UI_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_UI_STANDARD_MDNIE_2 = DSI0_UI_STANDARD_MDNIE_2;
	mdnie_data.DSI0_UI_AUTO_MDNIE_2 = DSI0_UI_AUTO_MDNIE_2;
	mdnie_data.DSI0_VIDEO_DYNAMIC_MDNIE_2 = DSI0_VIDEO_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_VIDEO_STANDARD_MDNIE_2 = DSI0_VIDEO_STANDARD_MDNIE_2;
	mdnie_data.DSI0_VIDEO_AUTO_MDNIE_2 = DSI0_VIDEO_AUTO_MDNIE_2;
	mdnie_data.DSI0_CAMERA_MDNIE_2 = DSI0_CAMERA_MDNIE_2;
	mdnie_data.DSI0_CAMERA_AUTO_MDNIE_2 = DSI0_CAMERA_AUTO_MDNIE_2;
	mdnie_data.DSI0_GALLERY_DYNAMIC_MDNIE_2 = DSI0_GALLERY_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_GALLERY_STANDARD_MDNIE_2 = DSI0_GALLERY_STANDARD_MDNIE_2;
	mdnie_data.DSI0_GALLERY_AUTO_MDNIE_2 = DSI0_GALLERY_AUTO_MDNIE_2;
	mdnie_data.DSI0_VT_DYNAMIC_MDNIE_2 = DSI0_VT_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_VT_STANDARD_MDNIE_2 = DSI0_VT_STANDARD_MDNIE_2;
	mdnie_data.DSI0_VT_AUTO_MDNIE_2 = DSI0_VT_AUTO_MDNIE_2;
	mdnie_data.DSI0_BROWSER_DYNAMIC_MDNIE_2 = DSI0_BROWSER_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_BROWSER_STANDARD_MDNIE_2 = DSI0_BROWSER_STANDARD_MDNIE_2;
	mdnie_data.DSI0_BROWSER_AUTO_MDNIE_2 = DSI0_BROWSER_AUTO_MDNIE_2;
	mdnie_data.DSI0_EBOOK_DYNAMIC_MDNIE_2 = DSI0_EBOOK_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_EBOOK_STANDARD_MDNIE_2 = DSI0_EBOOK_STANDARD_MDNIE_2;
	mdnie_data.DSI0_EBOOK_AUTO_MDNIE_2 = DSI0_EBOOK_AUTO_MDNIE_2;
	mdnie_data.DSI0_TDMB_DYNAMIC_MDNIE_2 = DSI0_TDMB_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_TDMB_STANDARD_MDNIE_2 = DSI0_TDMB_STANDARD_MDNIE_2;
	mdnie_data.DSI0_TDMB_AUTO_MDNIE_2 = DSI0_TDMB_AUTO_MDNIE_2;

	mdnie_data.DSI0_BYPASS_MDNIE = DSI0_BYPASS_MDNIE;
	mdnie_data.DSI0_NEGATIVE_MDNIE = DSI0_NEGATIVE_MDNIE;
	mdnie_data.DSI0_COLOR_BLIND_MDNIE = DSI0_COLOR_BLIND_MDNIE;
	mdnie_data.DSI0_HBM_CE_MDNIE = DSI0_HBM_CE_MDNIE;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE = DSI0_RGB_SENSOR_MDNIE;
	mdnie_data.DSI0_UI_DYNAMIC_MDNIE = DSI0_UI_DYNAMIC_MDNIE;
	mdnie_data.DSI0_UI_STANDARD_MDNIE = DSI0_UI_STANDARD_MDNIE;
	mdnie_data.DSI0_UI_NATURAL_MDNIE = DSI0_UI_NATURAL_MDNIE;
	mdnie_data.DSI0_UI_MOVIE_MDNIE = DSI0_UI_MOVIE_MDNIE;
	mdnie_data.DSI0_UI_AUTO_MDNIE = DSI0_UI_AUTO_MDNIE;
	mdnie_data.DSI0_VIDEO_OUTDOOR_MDNIE = NULL;
	mdnie_data.DSI0_VIDEO_DYNAMIC_MDNIE = DSI0_VIDEO_DYNAMIC_MDNIE;
	mdnie_data.DSI0_VIDEO_STANDARD_MDNIE = DSI0_VIDEO_STANDARD_MDNIE;
	mdnie_data.DSI0_VIDEO_NATURAL_MDNIE = DSI0_VIDEO_NATURAL_MDNIE;
	mdnie_data.DSI0_VIDEO_MOVIE_MDNIE = DSI0_VIDEO_MOVIE_MDNIE;
	mdnie_data.DSI0_VIDEO_AUTO_MDNIE = DSI0_VIDEO_AUTO_MDNIE;
	mdnie_data.DSI0_VIDEO_WARM_OUTDOOR_MDNIE = DSI0_VIDEO_WARM_OUTDOOR_MDNIE;
	mdnie_data.DSI0_VIDEO_WARM_MDNIE = DSI0_VIDEO_WARM_MDNIE;
	mdnie_data.DSI0_VIDEO_COLD_OUTDOOR_MDNIE = DSI0_VIDEO_COLD_OUTDOOR_MDNIE;
	mdnie_data.DSI0_VIDEO_COLD_MDNIE = DSI0_VIDEO_COLD_MDNIE;
	mdnie_data.DSI0_CAMERA_OUTDOOR_MDNIE = DSI0_CAMERA_OUTDOOR_MDNIE;
	mdnie_data.DSI0_CAMERA_MDNIE = DSI0_CAMERA_MDNIE;
	mdnie_data.DSI0_CAMERA_AUTO_MDNIE = DSI0_CAMERA_AUTO_MDNIE;
	mdnie_data.DSI0_GALLERY_DYNAMIC_MDNIE = DSI0_GALLERY_DYNAMIC_MDNIE;
	mdnie_data.DSI0_GALLERY_STANDARD_MDNIE = DSI0_GALLERY_STANDARD_MDNIE;
	mdnie_data.DSI0_GALLERY_NATURAL_MDNIE = DSI0_GALLERY_NATURAL_MDNIE;
	mdnie_data.DSI0_GALLERY_MOVIE_MDNIE = DSI0_GALLERY_MOVIE_MDNIE;
	mdnie_data.DSI0_GALLERY_AUTO_MDNIE = DSI0_GALLERY_AUTO_MDNIE;
	mdnie_data.DSI0_VT_DYNAMIC_MDNIE = DSI0_VT_DYNAMIC_MDNIE;
	mdnie_data.DSI0_VT_STANDARD_MDNIE = DSI0_VT_STANDARD_MDNIE;
	mdnie_data.DSI0_VT_NATURAL_MDNIE = DSI0_VT_NATURAL_MDNIE;
	mdnie_data.DSI0_VT_MOVIE_MDNIE = DSI0_VT_MOVIE_MDNIE;
	mdnie_data.DSI0_VT_AUTO_MDNIE = DSI0_VT_AUTO_MDNIE;
	mdnie_data.DSI0_BROWSER_DYNAMIC_MDNIE = DSI0_BROWSER_DYNAMIC_MDNIE;
	mdnie_data.DSI0_BROWSER_STANDARD_MDNIE = DSI0_BROWSER_STANDARD_MDNIE;
	mdnie_data.DSI0_BROWSER_NATURAL_MDNIE = DSI0_BROWSER_NATURAL_MDNIE;
	mdnie_data.DSI0_BROWSER_MOVIE_MDNIE = DSI0_BROWSER_MOVIE_MDNIE;
	mdnie_data.DSI0_BROWSER_AUTO_MDNIE = DSI0_BROWSER_AUTO_MDNIE;
	mdnie_data.DSI0_EBOOK_DYNAMIC_MDNIE = DSI0_EBOOK_DYNAMIC_MDNIE;
	mdnie_data.DSI0_EBOOK_STANDARD_MDNIE = DSI0_EBOOK_STANDARD_MDNIE;
	mdnie_data.DSI0_EBOOK_NATURAL_MDNIE = DSI0_EBOOK_NATURAL_MDNIE;
	mdnie_data.DSI0_EBOOK_MOVIE_MDNIE = DSI0_EBOOK_MOVIE_MDNIE;
	mdnie_data.DSI0_EBOOK_AUTO_MDNIE = DSI0_EBOOK_AUTO_MDNIE;
	mdnie_data.DSI0_EMAIL_AUTO_MDNIE = DSI0_EMAIL_AUTO_MDNIE;
	mdnie_data.DSI0_GAME_LOW_MDNIE = DSI0_GAME_LOW_MDNIE;
	mdnie_data.DSI0_GAME_MID_MDNIE = DSI0_GAME_MID_MDNIE;
	mdnie_data.DSI0_GAME_HIGH_MDNIE = DSI0_GAME_HIGH_MDNIE;
	mdnie_data.DSI0_TDMB_DYNAMIC_MDNIE = DSI0_TDMB_DYNAMIC_MDNIE;
	mdnie_data.DSI0_TDMB_STANDARD_MDNIE = DSI0_TDMB_STANDARD_MDNIE;
	mdnie_data.DSI0_TDMB_NATURAL_MDNIE = DSI0_TDMB_NATURAL_MDNIE;
	mdnie_data.DSI0_TDMB_MOVIE_MDNIE = DSI0_TDMB_MOVIE_MDNIE;
	mdnie_data.DSI0_TDMB_AUTO_MDNIE = DSI0_TDMB_AUTO_MDNIE;
	mdnie_data.DSI0_GRAYSCALE_MDNIE = DSI0_GRAYSCALE_MDNIE;
	mdnie_data.DSI0_GRAYSCALE_NEGATIVE_MDNIE= DSI0_GRAYSCALE_NEGATIVE_MDNIE;
	mdnie_data.DSI0_CURTAIN = DSI0_SCREEN_CURTAIN_MDNIE;
	mdnie_data.DSI0_NIGHT_MODE_MDNIE = DSI0_NIGHT_MODE_MDNIE;
	mdnie_data.DSI0_NIGHT_MODE_MDNIE_SCR = DSI0_NIGHT_MODE_MDNIE_1;
	mdnie_data.DSI0_COLOR_BLIND_MDNIE_SCR = DSI0_COLOR_BLIND_MDNIE_1;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_SCR = DSI0_RGB_SENSOR_MDNIE_1;
	mdnie_data.DSI0_COLOR_LENS_MDNIE = DSI0_COLOR_LENS_MDNIE;
	mdnie_data.DSI0_COLOR_LENS_MDNIE_SCR = DSI0_COLOR_LENS_MDNIE_1;

	mdnie_data.mdnie_tune_value_dsi0 = mdnie_tune_value_dsi0;
	mdnie_data.mdnie_tune_value_dsi1 = mdnie_tune_value_dsi0;
	mdnie_data.hmt_color_temperature_tune_value_dsi0 = hmt_color_temperature_tune_value_dsi0;
	mdnie_data.hmt_color_temperature_tune_value_dsi1 = hmt_color_temperature_tune_value_dsi0;

	mdnie_data.hdr_tune_value_dsi0 = hdr_tune_value_dsi0;
	mdnie_data.hdr_tune_value_dsi1 = hdr_tune_value_dsi0;
	
	mdnie_data.light_notification_tune_value_dsi0 = light_notification_tune_value_dsi0;
	mdnie_data.light_notification_tune_value_dsi1 = light_notification_tune_value_dsi0;

	/* Update MDNIE data related with size, offset or index */
	mdnie_data.dsi0_bypass_mdnie_size = ARRAY_SIZE(DSI0_BYPASS_MDNIE);
	mdnie_data.mdnie_color_blinde_cmd_offset = MDNIE_COLOR_BLINDE_CMD_OFFSET;
	mdnie_data.mdnie_step_index[MDNIE_STEP1] = MDNIE_STEP1_INDEX;
	mdnie_data.mdnie_step_index[MDNIE_STEP2] = MDNIE_STEP2_INDEX;
	mdnie_data.mdnie_step_index[MDNIE_STEP3] = MDNIE_STEP3_INDEX;
	mdnie_data.address_scr_white[ADDRESS_SCR_WHITE_RED_OFFSET] = ADDRESS_SCR_WHITE_RED;
	mdnie_data.address_scr_white[ADDRESS_SCR_WHITE_GREEN_OFFSET] = ADDRESS_SCR_WHITE_GREEN;
	mdnie_data.address_scr_white[ADDRESS_SCR_WHITE_BLUE_OFFSET] = ADDRESS_SCR_WHITE_BLUE;
	mdnie_data.dsi0_rgb_sensor_mdnie_1_size = DSI0_RGB_SENSOR_MDNIE_1_SIZE;
	mdnie_data.dsi0_rgb_sensor_mdnie_2_size = DSI0_RGB_SENSOR_MDNIE_2_SIZE;
	mdnie_data.dsi0_rgb_sensor_mdnie_3_size = DSI0_RGB_SENSOR_MDNIE_3_SIZE;

	mdnie_data.dsi0_trans_dimming_data_index = MDNIE_TRANS_DIMMING_DATA_INDEX;

	mdnie_data.dsi0_adjust_ldu_table = adjust_ldu_data;
	mdnie_data.dsi1_adjust_ldu_table = adjust_ldu_data;
	mdnie_data.dsi0_max_adjust_ldu = 6;
	mdnie_data.dsi1_max_adjust_ldu = 6;
	mdnie_data.dsi0_night_mode_table = night_mode_data;
	mdnie_data.dsi1_night_mode_table = night_mode_data;
	mdnie_data.dsi0_max_night_mode_index = 11;
	mdnie_data.dsi1_max_night_mode_index = 11;
	mdnie_data.dsi0_color_lens_table = color_lens_data;
	mdnie_data.dsi1_color_lens_table = NULL;
	mdnie_data.dsi0_scr_step_index = MDNIE_STEP1_INDEX;
	mdnie_data.dsi1_scr_step_index = MDNIE_STEP1_INDEX;
	mdnie_data.dsi0_white_default_r = 0xff;
	mdnie_data.dsi0_white_default_g = 0xff;
	mdnie_data.dsi0_white_default_b = 0xff;
	mdnie_data.dsi0_white_rgb_enabled = 0;
	mdnie_data.dsi1_white_default_r = 0xff;
	mdnie_data.dsi1_white_default_g = 0xff;
	mdnie_data.dsi1_white_default_b = 0xff;
	mdnie_data.dsi1_white_rgb_enabled = 0;
}

static void  mdss_panel_init(struct samsung_display_driver_data *vdd)
{
	char resolution_default[] = "FHD";

	LCD_ERR("%s", vdd->panel_name);

	vdd->support_mdnie_lite = true;

	vdd->support_mdnie_trans_dimming = true;

	vdd->mdnie_tune_size1 = sizeof(DSI0_BYPASS_MDNIE_1);
	vdd->mdnie_tune_size2 = sizeof(DSI0_BYPASS_MDNIE_2);
	vdd->mdnie_tune_size3 = sizeof(DSI0_BYPASS_MDNIE_3);

	/* ON/OFF */
	vdd->panel_func.samsung_panel_on_pre = mdss_panel_on_pre;
	vdd->panel_func.samsung_panel_on_post = mdss_panel_on_post;
	vdd->panel_func.samsung_panel_off_post = samsung_panel_off_post;

	/* DDI RX */
	vdd->panel_func.samsung_panel_revision = mdss_panel_revision;
	vdd->panel_func.samsung_manufacture_date_read = mdss_manufacture_date_read;
	vdd->panel_func.samsung_ddi_id_read = mdss_ddi_id_read;

	vdd->panel_func.samsung_hbm_read = NULL;
	vdd->panel_func.samsung_mdnie_read = NULL;
	vdd->panel_func.samsung_smart_dimming_init = NULL;
	vdd->panel_func.samsung_smart_get_conf = NULL;

	vdd->panel_func.samsung_elvss_read = mdss_elvss_read;
	vdd->panel_func.samsung_hbm_read = mdss_hbm_read;
	vdd->panel_func.samsung_mdnie_read = mdss_mdnie_read;
	vdd->panel_func.samsung_smart_dimming_init = mdss_samart_dimming_init;
	vdd->panel_func.samsung_smart_get_conf = smart_get_conf_S6E3HF4_AMB526JS01;

	vdd->panel_func.samsung_cell_id_read = mdss_cell_id_read;

	/* Brightness */
	vdd->panel_func.samsung_brightness_hbm_off = NULL;
	vdd->panel_func.samsung_brightness_aid = mdss_aid;
	vdd->panel_func.samsung_brightness_acl_on = mdss_acl_on;
	vdd->panel_func.samsung_brightness_acl_percent = NULL;
	vdd->panel_func.samsung_brightness_acl_off = mdss_acl_off;
	vdd->panel_func.samsung_brightness_elvss = mdss_elvss;
	vdd->panel_func.samsung_brightness_elvss_temperature1 = mdss_elvss_temperature1;
	vdd->panel_func.samsung_brightness_elvss_temperature2 = NULL;
	vdd->panel_func.samsung_brightness_vint = mdss_vint;
	vdd->panel_func.samsung_brightness_irc = mdss_irc;
	vdd->panel_func.samsung_brightness_gamma = mdss_gamma;

	/* HBM */
	vdd->panel_func.samsung_hbm_gamma = mdss_hbm_gamma;
	vdd->panel_func.samsung_hbm_etc = mdss_hbm_etc;
	vdd->panel_func.samsung_hbm_irc = mdss_hbm_irc;
	vdd->panel_func.get_hbm_candela_value = get_hbm_candela_value;

	/* Event */
	vdd->panel_func.samsung_change_ldi_fps = mdss_samsung_change_ldi_fps;

	/* HMT */
	vdd->panel_func.samsung_brightness_gamma_hmt = mdss_gamma_hmt;
	vdd->panel_func.samsung_brightness_aid_hmt = mdss_aid_hmt;
	vdd->panel_func.samsung_brightness_elvss_hmt = mdss_elvss_hmt;
	vdd->panel_func.samsung_brightness_vint_hmt = mdss_vint_hmt;
	vdd->panel_func.samsung_smart_dimming_hmt_init = mdss_samart_dimming_init_hmt;
	vdd->panel_func.samsung_smart_get_conf_hmt = smart_get_conf_S6E3HF4_AMB526JS01_hmt;

	dsi_update_mdnie_data();

	/* Panel LPM */
	vdd->panel_func.samsung_get_panel_lpm_mode = mdss_get_panel_lpm_mode;

	/* other line panel data parsing fn */
	vdd->panel_func.parsing_otherline_pdata = NULL;
	vdd->panel_func.set_panel_fab_type = mdss_set_hero2_fab_type;
	vdd->panel_func.get_panel_fab_type = mdss_get_hero2_fab_type;

	/* send recovery pck before sending image date (for ESD recovery) */
	vdd->send_esd_recovery = false;

	vdd->auto_brightness_level = 12;

	/* Enable panic on first pingpong timeout */
	vdd->debug_data->panic_on_pptimeout = true;

	/* Set IRC init value */
	vdd->irc_info.irc_enable_status = true;
	vdd->irc_info.irc_mode = IRC_MODERATO_MODE;

	/* COLOR WEAKNESS */
	vdd->panel_func.color_weakness_ccb_on_off = mdss_send_colorweakness_ccb_cmd;

	/* Support DDI HW CURSOR */
	vdd->panel_func.ddi_hw_cursor = ddi_hw_cursor;

	/* MULTI_RESOLUTION */
	vdd->panel_func.samsung_multires = mdss_panel_multires;
	memcpy(vdd->resolution_default, resolution_default, sizeof(resolution_default));

	/* ACL default ON */
	vdd->acl_status = 1;

	mdss_init_hero2_fab_type(vdd);
}

static int __init samsung_panel_init(void)
{
	struct samsung_display_driver_data *vdd = samsung_get_vdd();
	char panel_string[] = "ss_dsi_panel_S6E3HF4_AMB526JS01_WQHD";

	vdd->panel_name = mdss_mdp_panel + 8;

	LCD_INFO("%s \n", vdd->panel_name);

	if (!strncmp(vdd->panel_name, panel_string, strlen(panel_string)))
		vdd->panel_func.samsung_panel_init = mdss_panel_init;

	return 0;
}

early_initcall(samsung_panel_init);
