/*
 * =================================================================
 *
 *
 *	Description:  samsung display panel file
 *
 *	Author: jb09.kim
 *	Company:  Samsung Electronics
 *
 * ================================================================
 */
/*
<one line to give the program's name and a brief idea of what it does.>
Copyright (C) 2012, Samsung Electronics. All rights reserved.

*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
*/
#include "ss_dsi_panel_S6E3HA3_AMS567JA01.h"
#include "ss_dsi_mdnie_S6E3HA3_AMS567JA01.h"
#include <linux/sec_param.h>

/* AOD Mode status on AOD Service */
enum {
	AOD_MODE_ALPM_2NIT_ON = MAX_LPM_MODE + 1,
	AOD_MODE_HLPM_2NIT_ON,
	AOD_MODE_ALPM_40NIT_ON,
	AOD_MODE_HLPM_40NIT_ON,
};

/* Register to control brightness level */
#define ALPM_REG		0x53
/* Register to cnotrol ALPM/HLPM mode */
#define ALPM_CTRL_REG	0xBB


/*cmd_idx(of vdd->bl_level)is  mapping
	to color weakness platform bl level :
	it should be changed according the
	data sheet from display lab.*/

#define LUMINANCE_MAX 74

static int init_ldi_fps(struct mdss_dsi_ctrl_pdata *ctrl);

static int mdss_panel_on_pre(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	LCD_INFO("+: ndx=%d \n", ctrl->ndx);
	mdss_panel_attach_set(ctrl, true);
	LCD_INFO("-: ndx=%d \n", ctrl->ndx);

	init_ldi_fps(ctrl);

	return true;
}

static int mdss_panel_on_post(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	return true;
}

static int mdss_panel_revision(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	if (vdd->manufacture_id_dsi[ctrl->ndx] == 0)
		mdss_panel_attach_set(ctrl, false);
	else
		mdss_panel_attach_set(ctrl, true);

	if (mdss_panel_rev_get(ctrl) == 0x00)
		vdd->panel_revision = 'A';
	else if (mdss_panel_rev_get(ctrl) == 0x01)
		vdd->panel_revision = 'A';
	else if (mdss_panel_rev_get(ctrl) == 0x02)
		vdd->panel_revision = 'A'; /* rev B is A */
	else if (mdss_panel_rev_get(ctrl) == 0x03)
		vdd->panel_revision = 'C';
	else {
		LCD_ERR("panel_rev not found\n");
		vdd->panel_revision = 'C';
	}

	vdd->panel_revision -= 'A';

	vdd->aid_subdivision_enable = true;

	LCD_DEBUG("panel_revision = %c %d \n",
					vdd->panel_revision + 'A', vdd->panel_revision);

	return true;
}

static int mdss_manufacture_date_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	unsigned char date[4];
	int year, month, day;
	int hour, min;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (C8h 41,42th) for manufacture date */
	if (vdd->dtsi_data[ctrl->ndx].manufacture_date_rx_cmds[vdd->panel_revision].cmd_cnt) {
		mdss_samsung_panel_data_read(ctrl,
			&vdd->dtsi_data[ctrl->ndx].manufacture_date_rx_cmds[vdd->panel_revision],
			date, PANEL_LEVE1_KEY);

		year = date[0] & 0xf0;
		year >>= 4;
		year += 2011; // 0 = 2011 year
		month = date[0] & 0x0f;
		day = date[1] & 0x1f;
		hour = date[2]& 0x0f;
		min = date[3] & 0x1f;

		vdd->manufacture_date_dsi[ctrl->ndx] = year * 10000 + month * 100 + day;
		vdd->manufacture_time_dsi[ctrl->ndx] = hour * 100 + min;

		LCD_ERR("manufacture_date DSI%d = (%d%04d) - year(%d) month(%d) day(%d) hour(%d) min(%d)\n",
			ctrl->ndx, vdd->manufacture_date_dsi[ctrl->ndx], vdd->manufacture_time_dsi[ctrl->ndx],
			year, month, day, hour, min);

	} else {
		LCD_ERR("DSI%d error (%d)", ctrl->ndx,vdd->panel_revision);
		return false;
	}

	return true;
}

static int mdss_ddi_id_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char ddi_id[5];
	int loop;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (D6h 1~5th) for ddi id */
	if (vdd->dtsi_data[ctrl->ndx].ddi_id_rx_cmds[vdd->panel_revision].cmd_cnt) {
		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].ddi_id_rx_cmds[vdd->panel_revision]),
			ddi_id, PANEL_LEVE1_KEY);

		for(loop = 0; loop < 5; loop++)
			vdd->ddi_id_dsi[ctrl->ndx][loop] = ddi_id[loop];

		LCD_INFO("DSI%d : %02x %02x %02x %02x %02x\n", ctrl->ndx,
			vdd->ddi_id_dsi[ctrl->ndx][0], vdd->ddi_id_dsi[ctrl->ndx][1],
			vdd->ddi_id_dsi[ctrl->ndx][2], vdd->ddi_id_dsi[ctrl->ndx][3],
			vdd->ddi_id_dsi[ctrl->ndx][4]);
	} else {
		LCD_ERR("DSI%d error", ctrl->ndx);
		return false;
	}

	return true;
}

static int get_hbm_candela_value(int level)
{
	if (level == 13)
		return 443;
	else if (level == 6)
		return 465;
	else if (level == 7)
		return 488;
	else if (level == 8)
		return 510;
	else if (level == 9)
		return 533;
	else if (level == 10)
		return 555;
	else if (level == 11)
		return 578;
	else if (level == 12)
		return 600;
	else
		return 600;
}

static struct dsi_panel_cmds *mdss_hbm_gamma(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi[ctrl->ndx]->generate_hbm_gamma)) {
		LCD_ERR("generate_hbm_gamma is NULL error");
		return NULL;
	} else {
		vdd->smart_dimming_dsi[ctrl->ndx]->generate_hbm_gamma(
			vdd->smart_dimming_dsi[ctrl->ndx],
			vdd->auto_brightness,
			&vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1]);

		*level_key = PANEL_LEVE1_KEY;

		return &vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision];
	}
}

static struct dsi_panel_cmds *mdss_hbm_etc(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	char elvss_2th_val, elvss_22th_val;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	elvss_2th_val = 0;

	/* 0xB5 23th */
	elvss_22th_val = vdd->display_status_dsi[ctrl->ndx].elvss_value2;

	/* 0xB5 2th*/
	if (vdd->auto_brightness == HBM_MODE) /* 465CD */
		elvss_2th_val = 0x13;
	else if (vdd->auto_brightness == HBM_MODE + 1) /* 488CD */
		elvss_2th_val = 0x11;
	else if (vdd->auto_brightness == HBM_MODE + 2) /* 510CD */
		elvss_2th_val = 0x10;
	else if (vdd->auto_brightness == HBM_MODE + 3) /* 533CD */
		elvss_2th_val = 0x0E;
	else if (vdd->auto_brightness == HBM_MODE + 4) /* 555CD */
		elvss_2th_val = 0x0D;
	else if (vdd->auto_brightness == HBM_MODE + 5) /* 578CD */
		elvss_2th_val = 0x0B;
	else if (vdd->auto_brightness == HBM_MODE + 6) /* 600CD */
		elvss_2th_val = 0x0A;
	else if (vdd->auto_brightness == HBM_MODE + 7) /* 443CD */
		elvss_2th_val = 0x14;

	/* ELVSS 0xB5 2th, 22th */
	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[1].payload[2] = elvss_2th_val;
	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[3].payload[1] = elvss_22th_val;

	/* ACL 15% in LDU/CCB mode */
	if (vdd->color_weakness_mode || vdd->ldu_correction_state)
		vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[4].payload[4] = 0x12; /* 0x12 : 15% */
	else
		vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[4].payload[4] = 0x0A; /* 0x0A : 8% */

	vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision].cmds[5].payload[1] = 0x02; /* ACL ON */

	LCD_DEBUG("0xB5 2th : 0x%x 0xB5 22th(elvss val) : 0x%x\n", elvss_2th_val, elvss_22th_val);

	return &vdd->dtsi_data[ctrl->ndx].hbm_etc_tx_cmds[vdd->panel_revision];
}

static int mdss_elvss_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char elvss_b5[2];
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (B5h 22th,23th) for elvss*/
	mdss_samsung_panel_data_read(ctrl,
		&(vdd->dtsi_data[ctrl->ndx].elvss_rx_cmds[vdd->panel_revision]),
		elvss_b5, PANEL_LEVE1_KEY);
	vdd->display_status_dsi[ctrl->ndx].elvss_value1 = elvss_b5[0]; /*0xB5 22th OTP value*/
	vdd->display_status_dsi[ctrl->ndx].elvss_value2 = elvss_b5[1]; /*0xB5 23th */

	return true;
}

static int mdss_hbm_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char hbm_buffer1[33];
	int loop;

	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (B3h 1~33th) for hbm gamma */
	mdss_samsung_panel_data_read(ctrl,
		&(vdd->dtsi_data[ctrl->ndx].hbm_rx_cmds[vdd->panel_revision]),
		hbm_buffer1, PANEL_LEVE1_KEY);

	for (loop = 0; loop < 33; loop++) {
		if (loop == 0) {
			/* V255 RGB MSB */
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = (hbm_buffer1[loop] & 0x04) >> 2;
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[3] = (hbm_buffer1[loop] & 0x02) >> 1;
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[5] = hbm_buffer1[loop] & 0x01;
		} else if (loop == 1) {
			/* V255 R LSB */
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = hbm_buffer1[loop];
		} else if (loop == 2) {
			/* V255 G LSB */
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[4] = hbm_buffer1[loop];
		} else if (loop == 3) {
			/* V255 B LSB */
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[6] = hbm_buffer1[loop];
		} else {
			/* +3 means V255 RGB MSB */
			vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[loop + 3] = hbm_buffer1[loop];
		}
	}

	return true;
}

#define COORDINATE_DATA_SIZE 6
#define MDNIE_SCR_WR_ADDR 	0x34

#define F1(x,y) ((y)-((43*(x))/40)+45)
#define F2(x,y) ((y)-((310*(x))/297)-3)
#define F3(x,y) ((y)+((367*(x))/84)-16305)
#define F4(x,y) ((y)+((333*(x))/107)-12396)

static char coordinate_data_1[][COORDINATE_DATA_SIZE] = {
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00}, /* dummy */
	{0xff, 0x00, 0xfb, 0x00, 0xfb, 0x00}, /* Tune_1 */
	{0xff, 0x00, 0xfd, 0x00, 0xff, 0x00}, /* Tune_2 */
	{0xfb, 0x00, 0xfa, 0x00, 0xff, 0x00}, /* Tune_3 */
	{0xff, 0x00, 0xfe, 0x00, 0xfc, 0x00}, /* Tune_4 */
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00}, /* Tune_5 */
	{0xfb, 0x00, 0xfc, 0x00, 0xff, 0x00}, /* Tune_6 */
	{0xfd, 0x00, 0xff, 0x00, 0xfa, 0x00}, /* Tune_7 */
	{0xfd, 0x00, 0xff, 0x00, 0xfd, 0x00}, /* Tune_8 */
	{0xfb, 0x00, 0xff, 0x00, 0xff, 0x00}, /* Tune_9 */
};

static char coordinate_data_2[][COORDINATE_DATA_SIZE] = {
	{0xff, 0x00, 0xff, 0x00, 0xff, 0x00}, /* dummy */
	{0xff, 0x00, 0xf7, 0x00, 0xee, 0x00}, /* Tune_1 */
	{0xff, 0x00, 0xf8, 0x00, 0xf1, 0x00}, /* Tune_2 */
	{0xff, 0x00, 0xf9, 0x00, 0xf5, 0x00}, /* Tune_3 */
	{0xff, 0x00, 0xf9, 0x00, 0xee, 0x00}, /* Tune_4 */
	{0xff, 0x00, 0xfa, 0x00, 0xf1, 0x00}, /* Tune_5 */
	{0xff, 0x00, 0xfb, 0x00, 0xf5, 0x00}, /* Tune_6 */
	{0xff, 0x00, 0xfc, 0x00, 0xee, 0x00}, /* Tune_7 */
	{0xff, 0x00, 0xfc, 0x00, 0xf1, 0x00}, /* Tune_8 */
	{0xff, 0x00, 0xfd, 0x00, 0xf4, 0x00}, /* Tune_9 */
};

static char (*coordinate_data[MAX_MODE])[COORDINATE_DATA_SIZE] = {
	coordinate_data_1,
	coordinate_data_2,
	coordinate_data_2,
	coordinate_data_1,
	coordinate_data_1,
	coordinate_data_1
};

static int mdnie_coordinate_index(int x, int y)
{
	int tune_number = 0;

	if (F1(x,y) > 0) {
		if (F3(x,y) > 0) {
			tune_number = 3;
		} else {
			if (F4(x,y) < 0)
				tune_number = 1;
			else
				tune_number = 2;
		}
	} else {
		if (F2(x,y) < 0) {
			if (F3(x,y) > 0) {
				tune_number = 9;
			} else {
				if (F4(x,y) < 0)
					tune_number = 7;
				else
					tune_number = 8;
			}
		} else {
			if (F3(x,y) > 0)
				tune_number = 6;
			else {
				if (F4(x,y) < 0)
					tune_number = 4;
				else
					tune_number = 5;
			}
		}
	}

	return tune_number;
}

static int mdss_mdnie_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char x_y_location[4];
	int mdnie_tune_index = 0;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read mtp (D6h 1~5th) for ddi id */
	if (vdd->dtsi_data[ctrl->ndx].mdnie_read_rx_cmds[vdd->panel_revision].cmd_cnt) {
		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].mdnie_read_rx_cmds[vdd->panel_revision]),
			x_y_location, PANEL_LEVE1_KEY);

		vdd->mdnie_x[ctrl->ndx] = x_y_location[0] << 8 | x_y_location[1];	/* X */
		vdd->mdnie_y[ctrl->ndx] = x_y_location[2] << 8 | x_y_location[3];	/* Y */

		mdnie_tune_index = mdnie_coordinate_index(vdd->mdnie_x[ctrl->ndx], vdd->mdnie_y[ctrl->ndx]);
		coordinate_tunning_multi(ctrl->ndx, coordinate_data, mdnie_tune_index,
			MDNIE_SCR_WR_ADDR, COORDINATE_DATA_SIZE);

		LCD_INFO("DSI%d : X-%d Y-%d \n", ctrl->ndx,
			vdd->mdnie_x[ctrl->ndx], vdd->mdnie_y[ctrl->ndx]);
	} else {
		LCD_ERR("DSI%d error", ctrl->ndx);
		return false;
	}

	return true;
}

static int mdss_samart_dimming_init(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	vdd->smart_dimming_dsi[ctrl->ndx] = vdd->panel_func.samsung_smart_get_conf();

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi[ctrl->ndx])) {
		LCD_ERR("DSI%d error", ctrl->ndx);
		return false;
	} else {
		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].smart_dimming_mtp_rx_cmds[vdd->panel_revision]),
			vdd->smart_dimming_dsi[ctrl->ndx]->mtp_buffer, PANEL_LEVE1_KEY);

		/* Initialize smart dimming related things here */
		/* lux_tab setting for 350cd */
		vdd->smart_dimming_dsi[ctrl->ndx]->lux_tab = vdd->dtsi_data[ctrl->ndx].candela_map_table[vdd->panel_revision].lux_tab;
		vdd->smart_dimming_dsi[ctrl->ndx]->lux_tabsize = vdd->dtsi_data[ctrl->ndx].candela_map_table[vdd->panel_revision].lux_tab_size;
		vdd->smart_dimming_dsi[ctrl->ndx]->man_id = vdd->manufacture_id_dsi[ctrl->ndx];

		/* copy hbm gamma payload for hbm interpolation calc */
		vdd->smart_dimming_dsi[ctrl->ndx]->hbm_payload = &vdd->dtsi_data[ctrl->ndx].hbm_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1];

		/* Just a safety check to ensure smart dimming data is initialised well */
		vdd->smart_dimming_dsi[ctrl->ndx]->init(vdd->smart_dimming_dsi[ctrl->ndx]);

		vdd->temperature = 20; // default temperature

		vdd->smart_dimming_loaded_dsi[ctrl->ndx] = true;
	}

	LCD_INFO("DSI%d : --\n", ctrl->ndx);

	return true;
}

static int mdss_cell_id_read(struct mdss_dsi_ctrl_pdata *ctrl)
{
	char cell_id_buffer[MAX_CELL_ID] = {0,};
	int loop;
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	/* Read Panel Unique Cell ID (C8h 41~51th) */
	if (vdd->dtsi_data[ctrl->ndx].cell_id_rx_cmds[vdd->panel_revision].cmd_cnt) {
		memset(cell_id_buffer, 0x00, MAX_CELL_ID);

		mdss_samsung_panel_data_read(ctrl,
			&(vdd->dtsi_data[ctrl->ndx].cell_id_rx_cmds[vdd->panel_revision]),
			cell_id_buffer, PANEL_LEVE1_KEY);

		for(loop = 0; loop < MAX_CELL_ID; loop++)
			vdd->cell_id_dsi[ctrl->ndx][loop] = cell_id_buffer[loop];

		LCD_INFO("DSI%d: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			ctrl->ndx, vdd->cell_id_dsi[ctrl->ndx][0],
			vdd->cell_id_dsi[ctrl->ndx][1], vdd->cell_id_dsi[ctrl->ndx][2],
			vdd->cell_id_dsi[ctrl->ndx][3], vdd->cell_id_dsi[ctrl->ndx][4],
			vdd->cell_id_dsi[ctrl->ndx][5], vdd->cell_id_dsi[ctrl->ndx][6],
			vdd->cell_id_dsi[ctrl->ndx][7], vdd->cell_id_dsi[ctrl->ndx][8],
			vdd->cell_id_dsi[ctrl->ndx][9], vdd->cell_id_dsi[ctrl->ndx][10]);

	} else {
		LCD_ERR("DSI%d error", ctrl->ndx);
		return false;
	}

	return true;
}

static struct dsi_panel_cmds aid_cmd;
static struct dsi_panel_cmds *mdss_aid(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (vdd->aid_subdivision_enable) {
		aid_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].aid_subdivision_tx_cmds[vdd->panel_revision].cmds[vdd->bl_level]);
		LCD_ERR("level(%d), aid(%x %x)\n", vdd->bl_level, aid_cmd.cmds->payload[9], aid_cmd.cmds->payload[10]);
	} else {
		cd_index = get_cmd_index(vdd, ctrl->ndx);

		if (!vdd->dtsi_data[ctrl->ndx].aid_map_table[vdd->panel_revision].size ||
			cd_index > vdd->dtsi_data[ctrl->ndx].aid_map_table[vdd->panel_revision].size)
			goto end;

		cmd_idx = vdd->dtsi_data[ctrl->ndx].aid_map_table[vdd->panel_revision].cmd_idx[cd_index];

		aid_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].aid_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	}

	aid_cmd.cmd_cnt = 1;
	*level_key = PANEL_LEVE1_KEY;

	return &aid_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static struct dsi_panel_cmds * mdss_acl_on(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	if (vdd->gradual_acl_val)
		vdd->dtsi_data[ctrl->ndx].acl_on_tx_cmds[vdd->panel_revision].cmds[0].payload[4] = vdd->gradual_acl_val;

	return &(vdd->dtsi_data[ctrl->ndx].acl_on_tx_cmds[vdd->panel_revision]);
}

static struct dsi_panel_cmds * mdss_acl_off(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	return &(vdd->dtsi_data[ctrl->ndx].acl_off_tx_cmds[vdd->panel_revision]);
}

#if 0
static struct dsi_panel_cmds acl_percent_cmd;
static struct dsi_panel_cmds * mdss_acl_precent(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	cd_index = get_cmd_index(vdd, ctrl->ndx);

	if (!vdd->dtsi_data[ctrl->ndx].acl_map_table[vdd->panel_revision].size ||
		cd_index > vdd->dtsi_data[ctrl->ndx].acl_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].acl_map_table[vdd->panel_revision].cmd_idx[cd_index];

	acl_percent_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].acl_percent_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	acl_percent_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &acl_percent_cmd;

end :
	LCD_ERR("error");
	return NULL;

}
#endif

static struct dsi_panel_cmds elvss_cmd;
static struct dsi_panel_cmds * mdss_elvss(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	cd_index = get_cmd_index(vdd, ctrl->ndx);

	if (!vdd->dtsi_data[ctrl->ndx].smart_acl_elvss_map_table[vdd->panel_revision].size ||
		cd_index > vdd->dtsi_data[ctrl->ndx].smart_acl_elvss_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].smart_acl_elvss_map_table[vdd->panel_revision].cmd_idx[cd_index];

	elvss_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].elvss_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);

	elvss_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &elvss_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static struct dsi_panel_cmds * mdss_elvss_temperature1(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	char elvss_22th_val;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	/* OTP value - B5 22th */
	elvss_22th_val = vdd->display_status_dsi[ctrl->ndx].elvss_value1;

	if (mdss_panel_rev_get(ctrl) >= S6E3HA3_AMS567JA01_REV_C)
		elvss_22th_val += 0; /*nothing to do */
	else {
		if (vdd->candela_level >= 6) {
			if (vdd->temperature <= vdd->elvss_interpolation_temperature)
				elvss_22th_val -= 4;
		} else {
			if (vdd->temperature > 0) {
				elvss_22th_val += 0;
			} else if (vdd->temperature > vdd->elvss_interpolation_temperature) {
				if (vdd->candela_level == 5)
					elvss_22th_val += 1;
				else if (vdd->candela_level == 4)
					elvss_22th_val += 2;
				else if (vdd->candela_level == 3)
					elvss_22th_val += 3;
				else
					elvss_22th_val += 4;

			} else {
				if (vdd->candela_level == 5)
					elvss_22th_val -= 3;
				else if (vdd->candela_level == 4)
					elvss_22th_val -= 2;
				else if (vdd->candela_level == 3)
					elvss_22th_val -= 1;
				else
					elvss_22th_val += 0;
			}
		}
	}

	/* 0xB5 29th TSET */
	vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[1].payload[1] =
		vdd->temperature > 0 ? vdd->temperature : 0x80|(-1*vdd->temperature);

	/* 0xB5 22th ELVSS*/
	vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[3].payload[1] = elvss_22th_val;

	LCD_DEBUG("acl : %d, interpolation_temp : %d temp : %d, cd : %d, OTP(22th %x 23th %x), B5 22th :0x%x\n",
		vdd->acl_status, vdd->elvss_interpolation_temperature, vdd->temperature, vdd->candela_level,
		vdd->display_status_dsi[ctrl->ndx].elvss_value1, vdd->display_status_dsi[ctrl->ndx].elvss_value2,
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision].cmds[3].payload[1]);

	*level_key = PANEL_LEVE1_KEY;

	return &(vdd->dtsi_data[ctrl->ndx].elvss_lowtemp_tx_cmds[vdd->panel_revision]);
}

#if 0
static struct dsi_panel_cmds * mdss_elvss_temperature2(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (vdd->temperature > -20) {
		/*b6 21th para*/
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp2_tx_cmds[vdd->panel_revision].cmds[1].payload[1] =
			vdd->display_status_dsi[ctrl->ndx].elvss_value;
	} else {
		/*temp <= -20 : b6 21th para-0x05*/
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp2_tx_cmds[vdd->panel_revision].cmds[1].payload[1] =
			(vdd->display_status_dsi[ctrl->ndx].elvss_value - 0x04);
	}

	LCD_DEBUG("acl : %d temp : %d 0xB0 : 0x%x 0xB6 : 0x%x\n", vdd->acl_status, vdd->temperature,
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp2_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
		vdd->dtsi_data[ctrl->ndx].elvss_lowtemp2_tx_cmds[vdd->panel_revision].cmds[1].payload[1]);

	*level_key = PANEL_LEVE1_KEY;

	return &(vdd->dtsi_data[ctrl->ndx].elvss_lowtemp2_tx_cmds[vdd->panel_revision]);
}
#endif

static struct dsi_panel_cmds vint_cmd;
static struct dsi_panel_cmds *mdss_vint(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cd_index = 0;
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	cd_index = get_cmd_index(vdd, ctrl->ndx);

	if (!vdd->dtsi_data[ctrl->ndx].vint_map_table[vdd->panel_revision].size ||
		cd_index > vdd->dtsi_data[ctrl->ndx].vint_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].vint_map_table[vdd->panel_revision].cmd_idx[cd_index];

	if (vdd->temperature > vdd->elvss_interpolation_temperature )
		vint_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].vint_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	else
		vint_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].vint_tx_cmds[vdd->panel_revision].cmds[0]);
	vint_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &vint_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static char irc_dimming_para_revA [256][20] = {
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x5B, 0x52, 0x58, 0x1D, 0x22, 0x22, 0x11, 0x16, 0x17}, /* 255 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x5B, 0x52, 0x58, 0x1D, 0x22, 0x22, 0x11, 0x16, 0x17}, /* 254 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x5A, 0x51, 0x57, 0x1D, 0x22, 0x22, 0x11, 0x16, 0x17}, /* 253 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x5A, 0x51, 0x57, 0x1D, 0x22, 0x22, 0x10, 0x15, 0x16}, /* 252 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x5A, 0x51, 0x57, 0x1C, 0x21, 0x21, 0x11, 0x16, 0x17}, /* 251 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x59, 0x50, 0x56, 0x1D, 0x22, 0x22, 0x10, 0x15, 0x16}, /* 250 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x59, 0x50, 0x56, 0x1C, 0x21, 0x21, 0x11, 0x16, 0x17}, /* 249 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x58, 0x50, 0x55, 0x1D, 0x21, 0x22, 0x10, 0x15, 0x16}, /* 248 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x58, 0x4F, 0x55, 0x1C, 0x21, 0x21, 0x10, 0x15, 0x16}, /* 247 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x57, 0x4F, 0x54, 0x1C, 0x20, 0x21, 0x10, 0x15, 0x16}, /* 246 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x57, 0x4E, 0x54, 0x1C, 0x21, 0x20, 0x10, 0x15, 0x16}, /* 245 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x56, 0x4E, 0x54, 0x1C, 0x20, 0x20, 0x10, 0x15, 0x16}, /* 244 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x56, 0x4E, 0x53, 0x1B, 0x20, 0x20, 0x10, 0x14, 0x16}, /* 243 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x56, 0x4D, 0x53, 0x1B, 0x20, 0x20, 0x10, 0x15, 0x15}, /* 242 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x55, 0x4D, 0x52, 0x1B, 0x20, 0x20, 0x10, 0x14, 0x16}, /* 241 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x55, 0x4C, 0x52, 0x1B, 0x20, 0x20, 0x10, 0x14, 0x15}, /* 240 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x54, 0x4C, 0x52, 0x1B, 0x1F, 0x1F, 0x10, 0x15, 0x15}, /* 239 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x54, 0x4C, 0x51, 0x1B, 0x1F, 0x1F, 0x0F, 0x14, 0x16}, /* 238 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x53, 0x4B, 0x51, 0x1B, 0x1F, 0x1F, 0x10, 0x15, 0x15}, /* 237 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x53, 0x4B, 0x50, 0x1B, 0x1F, 0x20, 0x0F, 0x14, 0x15}, /* 236 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x53, 0x4B, 0x50, 0x1A, 0x1F, 0x1F, 0x10, 0x14, 0x15}, /* 235 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x52, 0x4A, 0x50, 0x1B, 0x1F, 0x1E, 0x0F, 0x14, 0x15}, /* 234 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x52, 0x4A, 0x4F, 0x1A, 0x1E, 0x1F, 0x0F, 0x14, 0x15}, /* 233 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x51, 0x49, 0x4F, 0x1A, 0x1F, 0x1E, 0x10, 0x14, 0x15}, /* 232 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x51, 0x49, 0x4E, 0x1A, 0x1E, 0x1F, 0x0F, 0x14, 0x14}, /* 231 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x51, 0x49, 0x4E, 0x19, 0x1E, 0x1E, 0x0F, 0x13, 0x14}, /* 230 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x50, 0x48, 0x4E, 0x1A, 0x1E, 0x1D, 0x0F, 0x14, 0x15}, /* 229 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x50, 0x48, 0x4D, 0x19, 0x1D, 0x1E, 0x0F, 0x14, 0x14}, /* 228 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4F, 0x47, 0x4C, 0x19, 0x1E, 0x1E, 0x0F, 0x13, 0x14}, /* 227 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4F, 0x47, 0x4C, 0x19, 0x1D, 0x1D, 0x0E, 0x13, 0x14}, /* 226 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4E, 0x46, 0x4C, 0x19, 0x1E, 0x1D, 0x0F, 0x13, 0x14}, /* 225 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4E, 0x46, 0x4B, 0x19, 0x1D, 0x1D, 0x0E, 0x13, 0x14}, /* 224 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4D, 0x46, 0x4B, 0x19, 0x1D, 0x1D, 0x0E, 0x12, 0x13}, /* 223 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4D, 0x45, 0x4A, 0x18, 0x1D, 0x1D, 0x0F, 0x13, 0x14}, /* 222 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4D, 0x45, 0x4A, 0x18, 0x1D, 0x1D, 0x0E, 0x12, 0x13}, /* 221 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4C, 0x45, 0x4A, 0x19, 0x1C, 0x1C, 0x0E, 0x13, 0x14}, /* 220 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4C, 0x44, 0x49, 0x18, 0x1D, 0x1D, 0x0E, 0x12, 0x13}, /* 219 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4B, 0x44, 0x49, 0x18, 0x1C, 0x1C, 0x0F, 0x12, 0x13}, /* 218 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4B, 0x44, 0x48, 0x18, 0x1C, 0x1D, 0x0E, 0x12, 0x12}, /* 217 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4B, 0x43, 0x48, 0x17, 0x1C, 0x1C, 0x0E, 0x12, 0x13}, /* 216 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4A, 0x43, 0x48, 0x18, 0x1B, 0x1B, 0x0E, 0x12, 0x13}, /* 215 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x4A, 0x43, 0x47, 0x17, 0x1B, 0x1C, 0x0E, 0x12, 0x13}, /* 214 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x49, 0x42, 0x47, 0x18, 0x1C, 0x1B, 0x0E, 0x11, 0x13}, /* 213 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x49, 0x42, 0x47, 0x17, 0x1B, 0x1B, 0x0E, 0x12, 0x12}, /* 212 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x49, 0x41, 0x46, 0x17, 0x1C, 0x1B, 0x0D, 0x11, 0x13}, /* 211 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x48, 0x41, 0x46, 0x17, 0x1B, 0x1B, 0x0E, 0x11, 0x12}, /* 210 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x48, 0x41, 0x45, 0x17, 0x1A, 0x1B, 0x0D, 0x12, 0x12}, /* 209 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x47, 0x40, 0x45, 0x17, 0x1B, 0x1B, 0x0D, 0x11, 0x12}, /* 208 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x47, 0x40, 0x45, 0x16, 0x1A, 0x1A, 0x0E, 0x11, 0x12}, /* 207 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x46, 0x3F, 0x44, 0x17, 0x1B, 0x1A, 0x0D, 0x11, 0x12}, /* 206 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x46, 0x3F, 0x44, 0x17, 0x1A, 0x1A, 0x0D, 0x11, 0x12}, /* 205 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x46, 0x3F, 0x43, 0x16, 0x1A, 0x1B, 0x0D, 0x11, 0x11}, /* 204 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x45, 0x3E, 0x43, 0x16, 0x1A, 0x1A, 0x0D, 0x11, 0x11}, /* 203 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x45, 0x3E, 0x43, 0x16, 0x1A, 0x19, 0x0D, 0x10, 0x12}, /* 202 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x44, 0x3E, 0x42, 0x16, 0x19, 0x1A, 0x0D, 0x11, 0x11}, /* 201 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x44, 0x3D, 0x42, 0x16, 0x1A, 0x19, 0x0C, 0x10, 0x11}, /* 200 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x44, 0x3D, 0x41, 0x15, 0x19, 0x1A, 0x0D, 0x11, 0x11}, /* 199 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x43, 0x3D, 0x41, 0x16, 0x19, 0x19, 0x0C, 0x10, 0x11}, /* 198 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x43, 0x3C, 0x41, 0x15, 0x19, 0x18, 0x0C, 0x10, 0x11}, /* 197 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x42, 0x3C, 0x40, 0x15, 0x19, 0x19, 0x0D, 0x10, 0x11}, /* 196 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x42, 0x3B, 0x40, 0x15, 0x19, 0x18, 0x0C, 0x10, 0x11}, /* 195 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x41, 0x3B, 0x3F, 0x15, 0x18, 0x19, 0x0D, 0x10, 0x10}, /* 194 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x41, 0x3B, 0x3F, 0x15, 0x18, 0x18, 0x0C, 0x10, 0x11}, /* 193 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x41, 0x3A, 0x3E, 0x14, 0x18, 0x19, 0x0C, 0x10, 0x10}, /* 192 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x40, 0x3A, 0x3E, 0x15, 0x18, 0x18, 0x0C, 0x0F, 0x10}, /* 191 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x40, 0x3A, 0x3E, 0x14, 0x17, 0x18, 0x0C, 0x10, 0x10}, /* 190 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3F, 0x39, 0x3D, 0x15, 0x18, 0x18, 0x0C, 0x0F, 0x10}, /* 189 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3F, 0x39, 0x3D, 0x14, 0x17, 0x18, 0x0C, 0x10, 0x0F}, /* 188 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3F, 0x38, 0x3D, 0x14, 0x18, 0x17, 0x0B, 0x0F, 0x10}, /* 187 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3E, 0x38, 0x3C, 0x14, 0x17, 0x17, 0x0C, 0x0F, 0x10}, /* 186 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3E, 0x38, 0x3C, 0x14, 0x17, 0x17, 0x0B, 0x0F, 0x10}, /* 185 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3E, 0x37, 0x3C, 0x13, 0x17, 0x16, 0x0C, 0x0F, 0x10}, /* 184 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3D, 0x37, 0x3B, 0x14, 0x17, 0x17, 0x0B, 0x0F, 0x0F}, /* 183 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3D, 0x37, 0x3B, 0x13, 0x16, 0x16, 0x0B, 0x0F, 0x10}, /* 182 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3C, 0x36, 0x3A, 0x13, 0x17, 0x17, 0x0C, 0x0E, 0x0F}, /* 181 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3C, 0x36, 0x3A, 0x13, 0x16, 0x16, 0x0B, 0x0F, 0x0F}, /* 180 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3B, 0x35, 0x39, 0x13, 0x17, 0x17, 0x0B, 0x0E, 0x0F}, /* 179 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3B, 0x35, 0x39, 0x13, 0x16, 0x16, 0x0B, 0x0F, 0x0F}, /* 178 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3B, 0x35, 0x39, 0x12, 0x16, 0x16, 0x0B, 0x0E, 0x0F}, /* 177 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3A, 0x35, 0x38, 0x13, 0x15, 0x16, 0x0B, 0x0E, 0x0F}, /* 176 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x3A, 0x34, 0x38, 0x12, 0x16, 0x16, 0x0B, 0x0E, 0x0E}, /* 175 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x39, 0x34, 0x38, 0x13, 0x15, 0x15, 0x0A, 0x0E, 0x0E}, /* 174 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x39, 0x33, 0x37, 0x12, 0x16, 0x15, 0x0B, 0x0D, 0x0F}, /* 173 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x39, 0x33, 0x37, 0x12, 0x15, 0x15, 0x0A, 0x0E, 0x0E}, /* 172 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x38, 0x33, 0x36, 0x12, 0x15, 0x16, 0x0B, 0x0D, 0x0E}, /* 171 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x38, 0x32, 0x36, 0x12, 0x15, 0x15, 0x0A, 0x0E, 0x0E}, /* 170 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x37, 0x32, 0x36, 0x12, 0x15, 0x14, 0x0B, 0x0D, 0x0E}, /* 169 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x37, 0x32, 0x35, 0x12, 0x14, 0x15, 0x0A, 0x0D, 0x0E}, /* 168 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x37, 0x31, 0x35, 0x11, 0x15, 0x14, 0x0B, 0x0D, 0x0E}, /* 167 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x36, 0x31, 0x35, 0x12, 0x14, 0x14, 0x0A, 0x0D, 0x0E}, /* 166 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x36, 0x31, 0x34, 0x11, 0x14, 0x14, 0x0A, 0x0D, 0x0E}, /* 165 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x36, 0x30, 0x34, 0x11, 0x14, 0x14, 0x0A, 0x0D, 0x0D}, /* 164 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x35, 0x30, 0x33, 0x11, 0x14, 0x14, 0x0A, 0x0D, 0x0E}, /* 163 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x35, 0x2F, 0x33, 0x10, 0x14, 0x14, 0x0A, 0x0D, 0x0D}, /* 162 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x34, 0x2F, 0x33, 0x11, 0x14, 0x13, 0x0A, 0x0D, 0x0E}, /* 161 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x34, 0x2F, 0x32, 0x11, 0x13, 0x14, 0x09, 0x0D, 0x0D}, /* 160 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x34, 0x2E, 0x32, 0x10, 0x14, 0x13, 0x0A, 0x0C, 0x0D}, /* 159 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x33, 0x2E, 0x31, 0x10, 0x13, 0x14, 0x0A, 0x0D, 0x0C}, /* 158 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x33, 0x2E, 0x31, 0x10, 0x13, 0x13, 0x09, 0x0C, 0x0D}, /* 157 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x32, 0x2D, 0x31, 0x10, 0x13, 0x12, 0x0A, 0x0C, 0x0D}, /* 156 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x32, 0x2D, 0x30, 0x10, 0x13, 0x13, 0x09, 0x0C, 0x0D}, /* 155 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x32, 0x2D, 0x30, 0x0F, 0x12, 0x13, 0x0A, 0x0C, 0x0C}, /* 154 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x31, 0x2C, 0x30, 0x10, 0x13, 0x12, 0x09, 0x0C, 0x0C}, /* 153 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x31, 0x2C, 0x2F, 0x0F, 0x12, 0x12, 0x09, 0x0C, 0x0D}, /* 152 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x31, 0x2C, 0x2F, 0x0F, 0x12, 0x12, 0x09, 0x0C, 0x0C}, /* 151 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x30, 0x2B, 0x2F, 0x0F, 0x12, 0x11, 0x09, 0x0C, 0x0D}, /* 150 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x30, 0x2B, 0x2E, 0x0F, 0x12, 0x12, 0x09, 0x0B, 0x0C}, /* 149 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2F, 0x2B, 0x2E, 0x0F, 0x11, 0x11, 0x09, 0x0C, 0x0C}, /* 148 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2F, 0x2A, 0x2D, 0x0F, 0x12, 0x12, 0x08, 0x0B, 0x0C}, /* 147 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2E, 0x2A, 0x2D, 0x0F, 0x11, 0x11, 0x09, 0x0B, 0x0C}, /* 146 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2E, 0x2A, 0x2D, 0x0F, 0x11, 0x11, 0x08, 0x0B, 0x0C}, /* 145 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2E, 0x29, 0x2C, 0x0E, 0x11, 0x11, 0x09, 0x0B, 0x0C}, /* 144 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2D, 0x29, 0x2C, 0x0F, 0x11, 0x11, 0x08, 0x0B, 0x0B}, /* 143 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2D, 0x28, 0x2B, 0x0E, 0x11, 0x11, 0x09, 0x0B, 0x0B}, /* 142 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2C, 0x28, 0x2B, 0x0F, 0x11, 0x11, 0x08, 0x0A, 0x0B}, /* 141 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2C, 0x28, 0x2B, 0x0E, 0x10, 0x10, 0x09, 0x0B, 0x0B}, /* 140 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2C, 0x27, 0x2A, 0x0E, 0x11, 0x11, 0x08, 0x0A, 0x0B}, /* 139 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2B, 0x27, 0x2A, 0x0E, 0x10, 0x10, 0x08, 0x0B, 0x0B}, /* 138 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2B, 0x27, 0x29, 0x0E, 0x10, 0x11, 0x08, 0x0A, 0x0A}, /* 137 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2B, 0x26, 0x29, 0x0D, 0x10, 0x10, 0x08, 0x0B, 0x0B}, /* 136 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2A, 0x26, 0x29, 0x0E, 0x10, 0x10, 0x08, 0x0A, 0x0A}, /* 135 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2A, 0x26, 0x28, 0x0D, 0x0F, 0x10, 0x08, 0x0A, 0x0B}, /* 134 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x2A, 0x25, 0x28, 0x0D, 0x10, 0x10, 0x08, 0x0A, 0x0A}, /* 133 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x29, 0x25, 0x28, 0x0D, 0x0F, 0x0F, 0x08, 0x0A, 0x0B}, /* 132 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x29, 0x25, 0x27, 0x0D, 0x0F, 0x10, 0x07, 0x0A, 0x0A}, /* 131 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x28, 0x24, 0x27, 0x0D, 0x0F, 0x0F, 0x08, 0x0A, 0x0A}, /* 130 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x28, 0x24, 0x27, 0x0D, 0x0F, 0x0F, 0x07, 0x0A, 0x0A}, /* 129 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x28, 0x24, 0x26, 0x0C, 0x0F, 0x0F, 0x08, 0x09, 0x0A}, /* 128 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x27, 0x23, 0x26, 0x0D, 0x0F, 0x0F, 0x07, 0x09, 0x09}, /* 127 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x27, 0x23, 0x26, 0x0C, 0x0F, 0x0E, 0x08, 0x09, 0x0A}, /* 126 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x27, 0x23, 0x25, 0x0C, 0x0E, 0x0F, 0x07, 0x09, 0x09}, /* 125 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x26, 0x23, 0x25, 0x0D, 0x0E, 0x0E, 0x07, 0x09, 0x0A}, /* 124 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x26, 0x22, 0x25, 0x0C, 0x0E, 0x0E, 0x07, 0x0A, 0x09}, /* 123 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x26, 0x22, 0x24, 0x0C, 0x0E, 0x0F, 0x07, 0x09, 0x09}, /* 122 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x25, 0x22, 0x24, 0x0C, 0x0E, 0x0E, 0x07, 0x09, 0x09}, /* 121 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x25, 0x21, 0x24, 0x0C, 0x0E, 0x0D, 0x06, 0x09, 0x0A}, /* 120 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x24, 0x21, 0x23, 0x0C, 0x0D, 0x0E, 0x07, 0x09, 0x09}, /* 119 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x24, 0x20, 0x23, 0x0B, 0x0E, 0x0D, 0x07, 0x09, 0x09}, /* 118 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x24, 0x20, 0x22, 0x0B, 0x0D, 0x0E, 0x06, 0x09, 0x09}, /* 117 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x23, 0x20, 0x22, 0x0B, 0x0D, 0x0D, 0x07, 0x08, 0x09}, /* 116 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x23, 0x1F, 0x22, 0x0B, 0x0D, 0x0C, 0x06, 0x09, 0x09}, /* 115 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x22, 0x1F, 0x21, 0x0B, 0x0D, 0x0D, 0x07, 0x08, 0x09}, /* 114 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x22, 0x1F, 0x21, 0x0B, 0x0C, 0x0D, 0x06, 0x09, 0x08}, /* 113 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x22, 0x1E, 0x20, 0x0A, 0x0D, 0x0D, 0x07, 0x08, 0x09}, /* 112 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x21, 0x1E, 0x20, 0x0B, 0x0D, 0x0D, 0x06, 0x08, 0x08}, /* 111 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x21, 0x1E, 0x20, 0x0A, 0x0C, 0x0C, 0x07, 0x08, 0x08}, /* 110 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x21, 0x1D, 0x20, 0x0A, 0x0D, 0x0C, 0x06, 0x08, 0x08}, /* 109 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x20, 0x1D, 0x1F, 0x0B, 0x0C, 0x0C, 0x06, 0x08, 0x08}, /* 108 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x20, 0x1D, 0x1F, 0x0A, 0x0C, 0x0C, 0x06, 0x08, 0x08}, /* 107 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x20, 0x1D, 0x1F, 0x0A, 0x0B, 0x0B, 0x06, 0x08, 0x08}, /* 106 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1F, 0x1C, 0x1E, 0x0A, 0x0C, 0x0C, 0x06, 0x08, 0x08}, /* 105 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1F, 0x1C, 0x1E, 0x0A, 0x0B, 0x0C, 0x06, 0x08, 0x07}, /* 104 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1F, 0x1C, 0x1E, 0x09, 0x0B, 0x0B, 0x06, 0x07, 0x08}, /* 103 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1E, 0x1B, 0x1D, 0x0A, 0x0B, 0x0B, 0x05, 0x08, 0x08}, /* 102 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1E, 0x1B, 0x1D, 0x09, 0x0B, 0x0B, 0x06, 0x07, 0x08}, /* 101 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1D, 0x1B, 0x1C, 0x0A, 0x0B, 0x0C, 0x05, 0x07, 0x07}, /* 100 */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1D, 0x1A, 0x1C, 0x09, 0x0B, 0x0B, 0x06, 0x07, 0x07}, /* 99  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1D, 0x1A, 0x1C, 0x09, 0x0A, 0x0A, 0x05, 0x07, 0x08}, /* 98  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1C, 0x1A, 0x1B, 0x09, 0x0A, 0x0B, 0x06, 0x07, 0x07}, /* 97  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1C, 0x19, 0x1B, 0x09, 0x0B, 0x0A, 0x05, 0x06, 0x08}, /* 96  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1C, 0x19, 0x1B, 0x09, 0x0A, 0x0A, 0x05, 0x07, 0x07}, /* 95  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1B, 0x19, 0x1A, 0x09, 0x0A, 0x0B, 0x05, 0x06, 0x07}, /* 94  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1B, 0x18, 0x1A, 0x09, 0x0B, 0x0A, 0x05, 0x06, 0x07}, /* 93  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1B, 0x18, 0x1A, 0x08, 0x0A, 0x0A, 0x05, 0x06, 0x06}, /* 92  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1A, 0x18, 0x1A, 0x09, 0x0A, 0x09, 0x05, 0x06, 0x07}, /* 91  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1A, 0x17, 0x19, 0x08, 0x0A, 0x0A, 0x05, 0x06, 0x06}, /* 90  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x1A, 0x17, 0x19, 0x08, 0x0A, 0x0A, 0x05, 0x06, 0x06}, /* 89  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x19, 0x17, 0x19, 0x08, 0x09, 0x09, 0x05, 0x06, 0x06}, /* 88  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x19, 0x17, 0x18, 0x08, 0x09, 0x0A, 0x05, 0x06, 0x06}, /* 87  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x19, 0x16, 0x18, 0x08, 0x09, 0x09, 0x04, 0x06, 0x06}, /* 86  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x18, 0x16, 0x18, 0x08, 0x09, 0x09, 0x05, 0x06, 0x06}, /* 85  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x18, 0x16, 0x17, 0x08, 0x09, 0x09, 0x04, 0x05, 0x06}, /* 84  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x18, 0x15, 0x17, 0x07, 0x09, 0x09, 0x05, 0x06, 0x06}, /* 83  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x17, 0x15, 0x17, 0x08, 0x09, 0x08, 0x04, 0x05, 0x06}, /* 82  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x17, 0x15, 0x16, 0x08, 0x09, 0x09, 0x04, 0x05, 0x06}, /* 81  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x17, 0x15, 0x16, 0x07, 0x08, 0x09, 0x04, 0x06, 0x05}, /* 80  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x16, 0x14, 0x16, 0x07, 0x08, 0x08, 0x05, 0x06, 0x06}, /* 79  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x16, 0x14, 0x15, 0x07, 0x08, 0x09, 0x04, 0x06, 0x05}, /* 78  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x16, 0x14, 0x15, 0x07, 0x08, 0x08, 0x04, 0x05, 0x06}, /* 77  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x15, 0x13, 0x15, 0x07, 0x08, 0x07, 0x04, 0x05, 0x06}, /* 76  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x15, 0x13, 0x14, 0x06, 0x08, 0x08, 0x04, 0x05, 0x05}, /* 75  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x15, 0x13, 0x14, 0x06, 0x07, 0x08, 0x04, 0x05, 0x05}, /* 74  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x14, 0x12, 0x13, 0x07, 0x08, 0x08, 0x03, 0x05, 0x05}, /* 73  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x14, 0x12, 0x13, 0x06, 0x07, 0x08, 0x04, 0x05, 0x05}, /* 72  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x14, 0x12, 0x13, 0x06, 0x07, 0x07, 0x03, 0x05, 0x05}, /* 71  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x13, 0x11, 0x13, 0x06, 0x08, 0x07, 0x04, 0x04, 0x05}, /* 70  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x13, 0x11, 0x12, 0x06, 0x07, 0x07, 0x03, 0x05, 0x05}, /* 69  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x13, 0x11, 0x12, 0x06, 0x07, 0x07, 0x03, 0x04, 0x05}, /* 68  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x12, 0x11, 0x12, 0x06, 0x06, 0x07, 0x04, 0x05, 0x04}, /* 67  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x12, 0x10, 0x11, 0x06, 0x07, 0x07, 0x03, 0x04, 0x05}, /* 66  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x12, 0x10, 0x11, 0x05, 0x07, 0x07, 0x04, 0x04, 0x04}, /* 65  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x11, 0x10, 0x11, 0x06, 0x06, 0x06, 0x03, 0x04, 0x05}, /* 64  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x11, 0x0F, 0x11, 0x06, 0x07, 0x06, 0x03, 0x04, 0x04}, /* 63  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x11, 0x0F, 0x10, 0x05, 0x06, 0x06, 0x03, 0x04, 0x05}, /* 62  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x10, 0x0F, 0x10, 0x06, 0x06, 0x06, 0x03, 0x04, 0x04}, /* 61  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x10, 0x0F, 0x10, 0x05, 0x06, 0x06, 0x03, 0x04, 0x04}, /* 60  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x10, 0x0E, 0x0F, 0x05, 0x06, 0x06, 0x03, 0x04, 0x04}, /* 59  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x10, 0x0E, 0x0F, 0x05, 0x06, 0x06, 0x02, 0x04, 0x04}, /* 58  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0F, 0x0E, 0x0F, 0x05, 0x05, 0x05, 0x03, 0x04, 0x04}, /* 57  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0F, 0x0D, 0x0E, 0x04, 0x06, 0x06, 0x03, 0x03, 0x03}, /* 56  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0F, 0x0D, 0x0E, 0x04, 0x06, 0x05, 0x03, 0x03, 0x04}, /* 55  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0E, 0x0D, 0x0E, 0x05, 0x05, 0x05, 0x02, 0x03, 0x03}, /* 54  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0E, 0x0C, 0x0D, 0x04, 0x06, 0x06, 0x03, 0x03, 0x03}, /* 53  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0E, 0x0C, 0x0D, 0x04, 0x05, 0x05, 0x03, 0x04, 0x04}, /* 52  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0D, 0x0C, 0x0D, 0x04, 0x05, 0x05, 0x03, 0x03, 0x03}, /* 51  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0D, 0x0C, 0x0D, 0x04, 0x05, 0x04, 0x03, 0x03, 0x04}, /* 50  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0D, 0x0C, 0x0C, 0x04, 0x04, 0x05, 0x02, 0x03, 0x03}, /* 49  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0C, 0x0B, 0x0C, 0x04, 0x05, 0x05, 0x03, 0x03, 0x03}, /* 48  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0C, 0x0B, 0x0C, 0x04, 0x04, 0x04, 0x02, 0x03, 0x03}, /* 47  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0C, 0x0B, 0x0C, 0x04, 0x04, 0x04, 0x02, 0x03, 0x03}, /* 46  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0B, 0x0A, 0x0B, 0x04, 0x05, 0x04, 0x02, 0x02, 0x03}, /* 45  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0B, 0x0A, 0x0B, 0x04, 0x04, 0x04, 0x02, 0x03, 0x03}, /* 44  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0B, 0x0A, 0x0A, 0x03, 0x04, 0x05, 0x02, 0x02, 0x02}, /* 43  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0B, 0x0A, 0x0A, 0x03, 0x04, 0x04, 0x02, 0x02, 0x03}, /* 42  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0A, 0x09, 0x0A, 0x03, 0x04, 0x04, 0x02, 0x02, 0x02}, /* 41  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0A, 0x09, 0x0A, 0x03, 0x04, 0x03, 0x02, 0x02, 0x03}, /* 40  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0A, 0x09, 0x09, 0x03, 0x03, 0x04, 0x02, 0x03, 0x03}, /* 39  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x0A, 0x09, 0x09, 0x03, 0x03, 0x04, 0x01, 0x02, 0x02}, /* 38  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x09, 0x08, 0x09, 0x03, 0x04, 0x03, 0x02, 0x02, 0x03}, /* 37  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x09, 0x08, 0x09, 0x03, 0x03, 0x03, 0x01, 0x02, 0x02}, /* 36  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x09, 0x08, 0x08, 0x02, 0x03, 0x04, 0x02, 0x02, 0x02}, /* 35  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x08, 0x08, 0x08, 0x03, 0x03, 0x03, 0x02, 0x02, 0x02}, /* 34  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x08, 0x07, 0x08, 0x03, 0x03, 0x03, 0x01, 0x02, 0x02}, /* 33  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x08, 0x07, 0x08, 0x02, 0x03, 0x02, 0x02, 0x02, 0x02}, /* 32  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x07, 0x07, 0x07, 0x03, 0x02, 0x03, 0x01, 0x02, 0x02}, /* 31  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x07, 0x06, 0x07, 0x02, 0x03, 0x03, 0x02, 0x02, 0x01}, /* 30  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x07, 0x06, 0x07, 0x02, 0x03, 0x02, 0x01, 0x02, 0x02}, /* 29  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x07, 0x06, 0x06, 0x02, 0x03, 0x03, 0x01, 0x01, 0x02}, /* 28  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x07, 0x06, 0x06, 0x02, 0x02, 0x03, 0x01, 0x02, 0x01}, /* 27  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x06, 0x06, 0x06, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02}, /* 26  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x06, 0x05, 0x06, 0x02, 0x02, 0x02, 0x01, 0x02, 0x01}, /* 25  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x06, 0x05, 0x05, 0x01, 0x02, 0x03, 0x01, 0x02, 0x01}, /* 24  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x05, 0x05, 0x05, 0x02, 0x02, 0x02, 0x01, 0x01, 0x02}, /* 23  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x05, 0x05, 0x05, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01}, /* 22  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x05, 0x04, 0x05, 0x01, 0x02, 0x01, 0x01, 0x01, 0x02}, /* 21  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x05, 0x04, 0x04, 0x01, 0x02, 0x02, 0x01, 0x01, 0x01}, /* 20  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x04, 0x04, 0x04, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01}, /* 19  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x04, 0x04, 0x04, 0x01, 0x01, 0x02, 0x01, 0x01, 0x01}, /* 18  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x04, 0x04, 0x04, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01}, /* 17  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x04, 0x03, 0x04, 0x01, 0x02, 0x01, 0x01, 0x01, 0x01}, /* 16  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x03, 0x03, 0x03, 0x02, 0x01, 0x02, 0x00, 0x01, 0x01}, /* 15  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01}, /* 14  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01}, /* 13  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00}, /* 12  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x03, 0x02, 0x03, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01}, /* 11  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01}, /* 10  */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00}, /* 9   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 8   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 7   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 6   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 5   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 4   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 3   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 2   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 1   */
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* 0   */
};

static char irc_dimming_para_revC [256][20] = {
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x54, 0x47, 0x48, 0x0F, 0x17, 0x17, 0x15, 0x19, 0x19},	/*	255	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x54, 0x47, 0x48, 0x0F, 0x17, 0x17, 0x15, 0x19, 0x19},	/*	254	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x53, 0x46, 0x47, 0x0F, 0x17, 0x17, 0x15, 0x19, 0x19},	/*	253	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x53, 0x46, 0x47, 0x0F, 0x17, 0x17, 0x15, 0x19, 0x19},	/*	252	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x53, 0x46, 0x47, 0x0F, 0x17, 0x17, 0x14, 0x18, 0x18},	/*	251	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x52, 0x46, 0x47, 0x0F, 0x16, 0x16, 0x15, 0x19, 0x19},	/*	250	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x52, 0x45, 0x46, 0x0F, 0x17, 0x17, 0x14, 0x18, 0x18},	/*	249	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x52, 0x45, 0x46, 0x0E, 0x16, 0x16, 0x15, 0x19, 0x19},	/*	248	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x51, 0x44, 0x45, 0x0E, 0x17, 0x17, 0x15, 0x18, 0x18},	/*	247	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x51, 0x44, 0x45, 0x0E, 0x16, 0x16, 0x14, 0x18, 0x18},	/*	246	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x50, 0x44, 0x45, 0x0F, 0x16, 0x16, 0x14, 0x18, 0x18},	/*	245	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x50, 0x43, 0x44, 0x0E, 0x16, 0x16, 0x14, 0x18, 0x18},	/*	244	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4F, 0x43, 0x44, 0x0F, 0x16, 0x16, 0x13, 0x17, 0x17},	/*	243	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4F, 0x43, 0x44, 0x0E, 0x15, 0x15, 0x14, 0x18, 0x18},	/*	242	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4F, 0x42, 0x43, 0x0E, 0x16, 0x16, 0x13, 0x17, 0x17},	/*	241	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4E, 0x42, 0x43, 0x0E, 0x16, 0x15, 0x14, 0x17, 0x18},	/*	240	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4E, 0x42, 0x43, 0x0E, 0x15, 0x15, 0x13, 0x17, 0x17},	/*	239	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4D, 0x41, 0x42, 0x0E, 0x16, 0x16, 0x14, 0x17, 0x17},	/*	238	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4D, 0x41, 0x42, 0x0E, 0x15, 0x15, 0x13, 0x17, 0x17},	/*	237	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4D, 0x41, 0x42, 0x0E, 0x15, 0x15, 0x13, 0x17, 0x17},	/*	236	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4C, 0x41, 0x41, 0x0E, 0x14, 0x15, 0x13, 0x17, 0x17},	/*	235	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4C, 0x40, 0x41, 0x0E, 0x15, 0x15, 0x13, 0x17, 0x17},	/*	234	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4C, 0x40, 0x41, 0x0D, 0x15, 0x15, 0x13, 0x16, 0x16},	/*	233	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4B, 0x40, 0x40, 0x0E, 0x14, 0x15, 0x12, 0x17, 0x16},	/*	232	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4B, 0x3F, 0x40, 0x0D, 0x15, 0x15, 0x13, 0x16, 0x16},	/*	231	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4A, 0x3F, 0x40, 0x0E, 0x14, 0x14, 0x12, 0x16, 0x16},	/*	230	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x4A, 0x3F, 0x3F, 0x0D, 0x14, 0x15, 0x13, 0x16, 0x16},	/*	229	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x49, 0x3E, 0x3F, 0x0E, 0x14, 0x14, 0x12, 0x16, 0x16},	/*	228	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x49, 0x3E, 0x3F, 0x0D, 0x14, 0x14, 0x12, 0x15, 0x15},	/*	227	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x49, 0x3D, 0x3E, 0x0D, 0x14, 0x14, 0x12, 0x16, 0x16},	/*	226	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x48, 0x3D, 0x3E, 0x0D, 0x14, 0x14, 0x12, 0x15, 0x15},	/*	225	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x48, 0x3D, 0x3E, 0x0D, 0x13, 0x13, 0x12, 0x16, 0x16},	/*	224	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x47, 0x3C, 0x3D, 0x0D, 0x14, 0x14, 0x12, 0x15, 0x15},	/*	223	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x47, 0x3C, 0x3D, 0x0D, 0x13, 0x13, 0x11, 0x16, 0x15},	/*	222	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x47, 0x3C, 0x3D, 0x0C, 0x13, 0x13, 0x12, 0x15, 0x15},	/*	221	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x46, 0x3C, 0x3C, 0x0D, 0x13, 0x14, 0x12, 0x15, 0x15},	/*	220	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x46, 0x3B, 0x3C, 0x0D, 0x13, 0x13, 0x11, 0x15, 0x15},	/*	219	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x46, 0x3B, 0x3C, 0x0C, 0x13, 0x13, 0x11, 0x15, 0x14},	/*	218	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x45, 0x3A, 0x3B, 0x0D, 0x13, 0x13, 0x11, 0x15, 0x15},	/*	217	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x45, 0x3A, 0x3B, 0x0C, 0x13, 0x13, 0x11, 0x14, 0x14},	/*	216	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x44, 0x3A, 0x3B, 0x0D, 0x13, 0x12, 0x11, 0x14, 0x15},	/*	215	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x44, 0x3A, 0x3A, 0x0C, 0x12, 0x13, 0x11, 0x15, 0x14},	/*	214	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x44, 0x39, 0x3A, 0x0C, 0x13, 0x13, 0x11, 0x14, 0x14},	/*	213	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x43, 0x39, 0x3A, 0x0C, 0x12, 0x12, 0x11, 0x14, 0x14},	/*	212	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x43, 0x39, 0x39, 0x0C, 0x12, 0x13, 0x11, 0x14, 0x14},	/*	211	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x43, 0x38, 0x39, 0x0B, 0x13, 0x12, 0x11, 0x13, 0x14},	/*	210	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x42, 0x38, 0x39, 0x0C, 0x12, 0x12, 0x11, 0x14, 0x14},	/*	209	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x42, 0x38, 0x38, 0x0C, 0x12, 0x12, 0x10, 0x13, 0x14},	/*	208	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x41, 0x37, 0x38, 0x0C, 0x12, 0x12, 0x10, 0x14, 0x13},	/*	207	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x41, 0x37, 0x38, 0x0C, 0x12, 0x12, 0x10, 0x13, 0x13},	/*	206	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x41, 0x37, 0x38, 0x0B, 0x12, 0x11, 0x11, 0x13, 0x14},	/*	205	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x40, 0x36, 0x37, 0x0C, 0x12, 0x12, 0x10, 0x13, 0x13},	/*	204	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x40, 0x36, 0x37, 0x0B, 0x12, 0x11, 0x10, 0x13, 0x13},	/*	203	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x40, 0x36, 0x37, 0x0B, 0x11, 0x11, 0x10, 0x13, 0x13},	/*	202	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3F, 0x35, 0x36, 0x0B, 0x12, 0x11, 0x10, 0x13, 0x13},	/*	201	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3F, 0x35, 0x36, 0x0B, 0x11, 0x11, 0x10, 0x13, 0x13},	/*	200	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3E, 0x35, 0x35, 0x0C, 0x11, 0x12, 0x0F, 0x12, 0x12},	/*	199	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3E, 0x34, 0x35, 0x0B, 0x11, 0x11, 0x10, 0x13, 0x13},	/*	198	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3E, 0x34, 0x35, 0x0B, 0x11, 0x11, 0x0F, 0x12, 0x12},	/*	197	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3D, 0x34, 0x34, 0x0B, 0x10, 0x11, 0x0F, 0x13, 0x12},	/*	196	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3D, 0x33, 0x34, 0x0B, 0x11, 0x11, 0x0F, 0x12, 0x12},	/*	195	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3C, 0x33, 0x34, 0x0B, 0x11, 0x10, 0x0F, 0x12, 0x12},	/*	194	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3C, 0x33, 0x33, 0x0B, 0x10, 0x11, 0x0F, 0x12, 0x12},	/*	193	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3C, 0x32, 0x33, 0x0A, 0x11, 0x10, 0x0F, 0x11, 0x12},	/*	192	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3B, 0x32, 0x33, 0x0B, 0x10, 0x10, 0x0F, 0x12, 0x12},	/*	191	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3B, 0x32, 0x33, 0x0B, 0x10, 0x10, 0x0E, 0x12, 0x11},	/*	190	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3B, 0x32, 0x32, 0x0A, 0x10, 0x10, 0x0F, 0x11, 0x12},	/*	189	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3A, 0x31, 0x32, 0x0B, 0x10, 0x10, 0x0E, 0x11, 0x11},	/*	188	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x3A, 0x31, 0x32, 0x0A, 0x10, 0x0F, 0x0F, 0x11, 0x12},	/*	187	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x39, 0x31, 0x31, 0x0B, 0x0F, 0x10, 0x0E, 0x11, 0x11},	/*	186	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x39, 0x30, 0x31, 0x0A, 0x10, 0x10, 0x0F, 0x11, 0x11},	/*	185	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x39, 0x30, 0x31, 0x0A, 0x10, 0x0F, 0x0E, 0x10, 0x11},	/*	184	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x38, 0x30, 0x30, 0x0A, 0x0F, 0x10, 0x0F, 0x11, 0x11},	/*	183	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x38, 0x2F, 0x30, 0x0A, 0x10, 0x0F, 0x0E, 0x10, 0x11},	/*	182	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x38, 0x2F, 0x30, 0x0A, 0x0F, 0x0F, 0x0D, 0x11, 0x10},	/*	181	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x37, 0x2F, 0x2F, 0x0A, 0x0F, 0x0F, 0x0E, 0x10, 0x11},	/*	180	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x37, 0x2E, 0x2F, 0x0A, 0x0F, 0x0F, 0x0D, 0x11, 0x10},	/*	179	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x37, 0x2E, 0x2F, 0x09, 0x0F, 0x0F, 0x0E, 0x10, 0x10},	/*	178	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x36, 0x2E, 0x2E, 0x0A, 0x0F, 0x0F, 0x0D, 0x10, 0x10},	/*	177	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x36, 0x2D, 0x2E, 0x09, 0x0F, 0x0F, 0x0E, 0x10, 0x10},	/*	176	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x35, 0x2D, 0x2E, 0x0A, 0x0F, 0x0E, 0x0D, 0x10, 0x10},	/*	175	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x35, 0x2D, 0x2D, 0x09, 0x0E, 0x0F, 0x0E, 0x10, 0x10},	/*	174	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x35, 0x2C, 0x2D, 0x09, 0x0F, 0x0E, 0x0D, 0x10, 0x10},	/*	173	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x34, 0x2C, 0x2D, 0x0A, 0x0E, 0x0E, 0x0D, 0x10, 0x10},	/*	172	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x34, 0x2C, 0x2D, 0x09, 0x0E, 0x0E, 0x0D, 0x10, 0x0F},	/*	171	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x34, 0x2C, 0x2C, 0x09, 0x0E, 0x0E, 0x0D, 0x0F, 0x10},	/*	170	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x33, 0x2B, 0x2C, 0x09, 0x0E, 0x0E, 0x0D, 0x10, 0x0F},	/*	169	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x33, 0x2B, 0x2C, 0x09, 0x0E, 0x0D, 0x0D, 0x0F, 0x10},	/*	168	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x33, 0x2B, 0x2B, 0x09, 0x0E, 0x0E, 0x0C, 0x0F, 0x0F},	/*	167	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x32, 0x2A, 0x2B, 0x09, 0x0E, 0x0E, 0x0D, 0x0F, 0x0F},	/*	166	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x32, 0x2A, 0x2B, 0x09, 0x0E, 0x0D, 0x0C, 0x0F, 0x0F},	/*	165	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x31, 0x2A, 0x2A, 0x09, 0x0D, 0x0E, 0x0D, 0x0F, 0x0F},	/*	164	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x31, 0x29, 0x2A, 0x09, 0x0E, 0x0D, 0x0C, 0x0E, 0x0F},	/*	163	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x31, 0x29, 0x2A, 0x08, 0x0D, 0x0D, 0x0C, 0x0F, 0x0E},	/*	162	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x30, 0x29, 0x29, 0x09, 0x0D, 0x0E, 0x0C, 0x0F, 0x0E},	/*	161	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x30, 0x29, 0x29, 0x09, 0x0D, 0x0D, 0x0C, 0x0E, 0x0F},	/*	160	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x30, 0x28, 0x29, 0x08, 0x0D, 0x0D, 0x0C, 0x0E, 0x0E},	/*	159	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2F, 0x28, 0x28, 0x09, 0x0D, 0x0D, 0x0B, 0x0E, 0x0E},	/*	158	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2F, 0x28, 0x28, 0x08, 0x0C, 0x0D, 0x0C, 0x0E, 0x0E},	/*	157	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2E, 0x27, 0x28, 0x09, 0x0D, 0x0C, 0x0B, 0x0E, 0x0E},	/*	156	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2E, 0x27, 0x28, 0x08, 0x0D, 0x0C, 0x0C, 0x0D, 0x0E},	/*	155	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2E, 0x27, 0x27, 0x08, 0x0C, 0x0D, 0x0B, 0x0E, 0x0D},	/*	154	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2D, 0x26, 0x27, 0x09, 0x0D, 0x0C, 0x0B, 0x0D, 0x0E},	/*	153	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2D, 0x26, 0x27, 0x08, 0x0C, 0x0C, 0x0B, 0x0E, 0x0D},	/*	152	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2D, 0x26, 0x26, 0x08, 0x0C, 0x0D, 0x0B, 0x0D, 0x0D},	/*	151	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2C, 0x26, 0x26, 0x08, 0x0C, 0x0C, 0x0B, 0x0D, 0x0D},	/*	150	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2C, 0x25, 0x26, 0x08, 0x0C, 0x0C, 0x0B, 0x0D, 0x0D},	/*	149	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2C, 0x25, 0x25, 0x07, 0x0C, 0x0C, 0x0B, 0x0D, 0x0D},	/*	148	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2B, 0x25, 0x25, 0x08, 0x0B, 0x0C, 0x0B, 0x0D, 0x0D},	/*	147	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2B, 0x24, 0x25, 0x07, 0x0C, 0x0B, 0x0B, 0x0D, 0x0D},	/*	146	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2B, 0x24, 0x25, 0x07, 0x0C, 0x0B, 0x0B, 0x0C, 0x0D},	/*	145	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2A, 0x24, 0x24, 0x08, 0x0B, 0x0C, 0x0A, 0x0D, 0x0C},	/*	144	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x2A, 0x23, 0x24, 0x07, 0x0C, 0x0B, 0x0B, 0x0C, 0x0D},	/*	143	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x29, 0x23, 0x23, 0x08, 0x0B, 0x0C, 0x0A, 0x0D, 0x0C},	/*	142	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x29, 0x23, 0x23, 0x07, 0x0B, 0x0B, 0x0B, 0x0C, 0x0D},	/*	141	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x29, 0x22, 0x23, 0x07, 0x0C, 0x0B, 0x0A, 0x0C, 0x0C},	/*	140	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x28, 0x22, 0x23, 0x08, 0x0B, 0x0B, 0x0A, 0x0C, 0x0C},	/*	139	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x28, 0x22, 0x22, 0x07, 0x0B, 0x0B, 0x0A, 0x0C, 0x0C},	/*	138	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x28, 0x21, 0x22, 0x07, 0x0B, 0x0B, 0x0A, 0x0C, 0x0C},	/*	137	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x27, 0x21, 0x22, 0x07, 0x0B, 0x0B, 0x0A, 0x0C, 0x0B},	/*	136	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x27, 0x21, 0x21, 0x07, 0x0B, 0x0B, 0x0A, 0x0B, 0x0C},	/*	135	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x27, 0x21, 0x21, 0x06, 0x0A, 0x0B, 0x0A, 0x0C, 0x0B},	/*	134	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x26, 0x20, 0x21, 0x07, 0x0B, 0x0A, 0x0A, 0x0B, 0x0C},	/*	133	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x26, 0x20, 0x21, 0x07, 0x0B, 0x0A, 0x09, 0x0B, 0x0B},	/*	132	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x26, 0x20, 0x20, 0x06, 0x0A, 0x0B, 0x0A, 0x0B, 0x0B},	/*	131	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x25, 0x1F, 0x20, 0x07, 0x0B, 0x0A, 0x09, 0x0B, 0x0B},	/*	130	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x25, 0x1F, 0x20, 0x07, 0x0A, 0x0A, 0x09, 0x0B, 0x0B},	/*	129	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x25, 0x1F, 0x1F, 0x06, 0x0A, 0x0A, 0x09, 0x0B, 0x0B},	/*	128	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x24, 0x1F, 0x1F, 0x07, 0x0A, 0x0A, 0x09, 0x0A, 0x0B},	/*	127	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x24, 0x1E, 0x1F, 0x06, 0x0A, 0x0A, 0x09, 0x0B, 0x0A},	/*	126	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x24, 0x1E, 0x1F, 0x06, 0x0A, 0x09, 0x09, 0x0A, 0x0B},	/*	125	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x23, 0x1E, 0x1E, 0x07, 0x0A, 0x0A, 0x09, 0x0A, 0x0B},	/*	124	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x23, 0x1E, 0x1E, 0x06, 0x09, 0x0A, 0x09, 0x0B, 0x0A},	/*	123	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x23, 0x1D, 0x1E, 0x06, 0x0A, 0x09, 0x09, 0x0A, 0x0B},	/*	122	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x22, 0x1D, 0x1D, 0x07, 0x09, 0x0A, 0x08, 0x0B, 0x0A},	/*	121	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x22, 0x1D, 0x1D, 0x06, 0x09, 0x09, 0x09, 0x0A, 0x0B},	/*	120	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x22, 0x1C, 0x1D, 0x06, 0x0A, 0x09, 0x08, 0x0A, 0x0A},	/*	119	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x21, 0x1C, 0x1C, 0x06, 0x09, 0x0A, 0x08, 0x0A, 0x09},	/*	118	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x21, 0x1C, 0x1C, 0x06, 0x09, 0x09, 0x08, 0x09, 0x0A},	/*	117	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x20, 0x1B, 0x1C, 0x06, 0x09, 0x09, 0x08, 0x0A, 0x09},	/*	116	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x20, 0x1B, 0x1B, 0x06, 0x09, 0x09, 0x08, 0x09, 0x0A},	/*	115	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x20, 0x1B, 0x1B, 0x05, 0x09, 0x09, 0x08, 0x09, 0x09},	/*	114	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1F, 0x1B, 0x1B, 0x06, 0x08, 0x09, 0x08, 0x09, 0x09},	/*	113	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1F, 0x1A, 0x1B, 0x06, 0x09, 0x08, 0x07, 0x09, 0x09},	/*	112	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1F, 0x1A, 0x1A, 0x05, 0x08, 0x09, 0x08, 0x0A, 0x09},	/*	111	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1E, 0x1A, 0x1A, 0x06, 0x08, 0x08, 0x07, 0x09, 0x09},	/*	110	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1E, 0x1A, 0x1A, 0x06, 0x08, 0x08, 0x07, 0x09, 0x09},	/*	109	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1E, 0x19, 0x1A, 0x05, 0x08, 0x08, 0x08, 0x09, 0x09},	/*	108	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1E, 0x19, 0x19, 0x05, 0x08, 0x08, 0x07, 0x09, 0x09},	/*	107	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1D, 0x19, 0x19, 0x05, 0x08, 0x08, 0x08, 0x08, 0x09},	/*	106	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1D, 0x19, 0x19, 0x05, 0x07, 0x08, 0x07, 0x09, 0x08},	/*	105	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1D, 0x18, 0x19, 0x05, 0x08, 0x07, 0x07, 0x09, 0x09},	/*	104	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1C, 0x18, 0x18, 0x05, 0x08, 0x08, 0x07, 0x08, 0x08},	/*	103	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1C, 0x17, 0x18, 0x05, 0x08, 0x07, 0x07, 0x08, 0x09},	/*	102	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1C, 0x17, 0x18, 0x05, 0x08, 0x07, 0x06, 0x08, 0x08},	/*	101	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1B, 0x17, 0x17, 0x05, 0x07, 0x08, 0x07, 0x09, 0x08},	/*	100	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1B, 0x17, 0x17, 0x05, 0x07, 0x07, 0x06, 0x08, 0x08},	/*	99	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1A, 0x16, 0x17, 0x05, 0x08, 0x07, 0x07, 0x07, 0x08},	/*	98	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1A, 0x16, 0x16, 0x05, 0x07, 0x08, 0x06, 0x08, 0x07},	/*	97	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1A, 0x16, 0x16, 0x04, 0x07, 0x07, 0x07, 0x08, 0x08},	/*	96	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x1A, 0x16, 0x16, 0x04, 0x07, 0x07, 0x07, 0x07, 0x08},	/*	95	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x19, 0x15, 0x16, 0x05, 0x07, 0x07, 0x06, 0x08, 0x07},	/*	94	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x19, 0x15, 0x15, 0x04, 0x07, 0x07, 0x07, 0x07, 0x08},	/*	93	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x19, 0x15, 0x15, 0x04, 0x07, 0x07, 0x06, 0x07, 0x07},	/*	92	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x18, 0x15, 0x15, 0x05, 0x06, 0x07, 0x06, 0x08, 0x07},	/*	91	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x18, 0x14, 0x15, 0x04, 0x07, 0x06, 0x06, 0x07, 0x07},	/*	90	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x18, 0x14, 0x14, 0x04, 0x07, 0x07, 0x06, 0x07, 0x07},	/*	89	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x17, 0x14, 0x14, 0x05, 0x06, 0x06, 0x05, 0x07, 0x07},	/*	88	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x17, 0x14, 0x14, 0x04, 0x06, 0x06, 0x06, 0x07, 0x07},	/*	87	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x17, 0x13, 0x14, 0x04, 0x07, 0x06, 0x06, 0x06, 0x07},	/*	86	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x17, 0x13, 0x13, 0x04, 0x06, 0x07, 0x05, 0x07, 0x06},	/*	85	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x16, 0x13, 0x13, 0x04, 0x06, 0x06, 0x06, 0x06, 0x07},	/*	84	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x16, 0x13, 0x13, 0x04, 0x06, 0x06, 0x05, 0x06, 0x06},	/*	83	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x16, 0x12, 0x13, 0x03, 0x06, 0x05, 0x06, 0x07, 0x07},	/*	82	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x15, 0x12, 0x12, 0x04, 0x06, 0x06, 0x06, 0x06, 0x07},	/*	81	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x15, 0x12, 0x12, 0x04, 0x06, 0x06, 0x05, 0x06, 0x06},	/*	80	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x15, 0x11, 0x12, 0x03, 0x06, 0x05, 0x05, 0x06, 0x06},	/*	79	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x14, 0x11, 0x11, 0x04, 0x06, 0x06, 0x05, 0x06, 0x06},	/*	78	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x14, 0x11, 0x11, 0x04, 0x05, 0x06, 0x05, 0x06, 0x06},	/*	77	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x14, 0x11, 0x11, 0x03, 0x05, 0x05, 0x05, 0x06, 0x06},	/*	76	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x13, 0x10, 0x10, 0x04, 0x05, 0x06, 0x04, 0x06, 0x05},	/*	75	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x13, 0x10, 0x10, 0x03, 0x05, 0x05, 0x05, 0x06, 0x06},	/*	74	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x13, 0x10, 0x10, 0x03, 0x05, 0x05, 0x05, 0x05, 0x06},	/*	73	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x12, 0x10, 0x10, 0x04, 0x05, 0x05, 0x04, 0x05, 0x05},	/*	72	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x12, 0x0F, 0x0F, 0x03, 0x05, 0x05, 0x05, 0x06, 0x06},	/*	71	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x12, 0x0F, 0x0F, 0x03, 0x05, 0x05, 0x04, 0x05, 0x05},	/*	70	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x11, 0x0F, 0x0F, 0x04, 0x04, 0x05, 0x04, 0x06, 0x05},	/*	69	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x11, 0x0F, 0x0F, 0x03, 0x04, 0x04, 0x05, 0x05, 0x06},	/*	68	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x11, 0x0E, 0x0F, 0x03, 0x05, 0x04, 0x04, 0x05, 0x05},	/*	67	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x11, 0x0E, 0x0E, 0x03, 0x05, 0x05, 0x04, 0x05, 0x05},	/*	66	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x10, 0x0E, 0x0E, 0x03, 0x04, 0x05, 0x04, 0x05, 0x04},	/*	65	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x10, 0x0E, 0x0E, 0x03, 0x04, 0x04, 0x04, 0x05, 0x05},	/*	64	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x10, 0x0D, 0x0E, 0x03, 0x05, 0x04, 0x04, 0x04, 0x05},	/*	63	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0F, 0x0D, 0x0D, 0x03, 0x04, 0x04, 0x04, 0x05, 0x05},	/*	62	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0F, 0x0D, 0x0D, 0x03, 0x04, 0x04, 0x04, 0x05, 0x05},	/*	61	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0F, 0x0D, 0x0D, 0x03, 0x04, 0x04, 0x03, 0x04, 0x04},	/*	60	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0F, 0x0C, 0x0D, 0x02, 0x04, 0x04, 0x04, 0x05, 0x04},	/*	59	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0E, 0x0C, 0x0C, 0x03, 0x04, 0x04, 0x04, 0x04, 0x05},	/*	58	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0E, 0x0C, 0x0C, 0x03, 0x04, 0x04, 0x03, 0x04, 0x04},	/*	57	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0E, 0x0B, 0x0C, 0x02, 0x04, 0x03, 0x03, 0x04, 0x04},	/*	56	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0D, 0x0B, 0x0B, 0x03, 0x04, 0x04, 0x03, 0x04, 0x04},	/*	55	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0D, 0x0B, 0x0B, 0x02, 0x04, 0x04, 0x04, 0x03, 0x04},	/*	54	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0D, 0x0B, 0x0B, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04},	/*	53	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0D, 0x0B, 0x0B, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04},	/*	52	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0C, 0x0A, 0x0A, 0x02, 0x04, 0x04, 0x03, 0x03, 0x03},	/*	51	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0C, 0x0A, 0x0A, 0x02, 0x03, 0x04, 0x03, 0x04, 0x03},	/*	50	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0C, 0x0A, 0x0A, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04},	/*	49	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0B, 0x0A, 0x0A, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03},	/*	48	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0B, 0x09, 0x0A, 0x02, 0x04, 0x03, 0x03, 0x03, 0x03},	/*	47	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0B, 0x09, 0x09, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04},	/*	46	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0B, 0x09, 0x09, 0x01, 0x03, 0x03, 0x03, 0x03, 0x03},	/*	45	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0A, 0x09, 0x09, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03},	/*	44	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0A, 0x08, 0x09, 0x02, 0x03, 0x02, 0x02, 0x03, 0x03},	/*	43	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x0A, 0x08, 0x08, 0x02, 0x03, 0x03, 0x02, 0x03, 0x03},	/*	42	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x09, 0x08, 0x08, 0x02, 0x03, 0x03, 0x02, 0x02, 0x02},	/*	41	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x09, 0x08, 0x08, 0x02, 0x02, 0x02, 0x02, 0x03, 0x03},	/*	40	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x09, 0x08, 0x08, 0x02, 0x02, 0x02, 0x02, 0x03, 0x03},	/*	39	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x09, 0x07, 0x08, 0x01, 0x03, 0x02, 0x03, 0x02, 0x03},	/*	38	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x09, 0x07, 0x07, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02},	/*	37	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x08, 0x07, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x03},	/*	36	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x08, 0x07, 0x07, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02},	/*	35	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x08, 0x07, 0x07, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02},	/*	34	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x07, 0x06, 0x06, 0x02, 0x02, 0x02, 0x02, 0x02, 0x03},	/*	33	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x07, 0x06, 0x06, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02},	/*	32	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x07, 0x06, 0x06, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02},	/*	31	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x07, 0x06, 0x06, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02},	/*	30	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x06, 0x05, 0x05, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02},	/*	29	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x06, 0x05, 0x05, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02},	/*	28	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x06, 0x05, 0x05, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02},	/*	27	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x06, 0x05, 0x05, 0x01, 0x01, 0x02, 0x01, 0x02, 0x01},	/*	26	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x05, 0x05, 0x05, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02},	/*	25	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x05, 0x04, 0x04, 0x01, 0x02, 0x02, 0x01, 0x01, 0x01},	/*	24	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x05, 0x04, 0x04, 0x01, 0x02, 0x02, 0x01, 0x01, 0x01},	/*	23	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x05, 0x04, 0x04, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02},	/*	22	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x04, 0x04, 0x04, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	21	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x04, 0x04, 0x04, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	20	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x04, 0x03, 0x03, 0x01, 0x01, 0x02, 0x01, 0x02, 0x01},	/*	19	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x04, 0x03, 0x03, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	18	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x04, 0x03, 0x03, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	17	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	16	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x03, 0x03, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	15	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x03, 0x03, 0x03, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01},	/*	14	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x03, 0x02, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	13	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x03, 0x02, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01},	/*	12	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00},	/*	11	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x02, 0x02, 0x02, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01},	/*	10	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x02, 0x02, 0x02, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01},	/*	9	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	8	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	7	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	6	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	5	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	4	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	3	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	2	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	1	*/
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},	/*	0	*/
};

static struct dsi_panel_cmds *mdss_irc(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int para_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (IS_ERR_OR_NULL(vdd->dtsi_data[ctrl->ndx].irc_tx_cmds)) {
		LCD_ERR("No irc_tx_cmds\n");
		return NULL;
	}

	if (!vdd->samsung_support_irc)
		return NULL;

	*level_key = PANEL_LEVE1_KEY;

	if (vdd->ldu_correction_state || vdd->color_weakness_value)
		return &(vdd->dtsi_data[ctrl->ndx].irc_off_tx_cmds[vdd->panel_revision]);

	para_idx = 255 - vdd->bl_level;

	/* copy irc default setting */
	if (mdss_panel_rev_get(ctrl) >= S6E3HA3_AMS567JA01_REV_C)
		memcpy(&vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
			irc_dimming_para_revC[para_idx], 20);
	else
		memcpy(&vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
			irc_dimming_para_revA[para_idx], 20);

	/* update irc enable*/
	vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = vdd->irc_info.irc_enable_status;

	/* update irc mode*/
	vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = vdd->irc_info.irc_mode;

	return &(vdd->dtsi_data[ctrl->ndx].irc_tx_cmds[vdd->panel_revision]);
}

static char irc_hbm_para_revA[8][20] = {
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x82, 0x75, 0x7E, 0x29, 0x31, 0x30, 0x19, 0x1F, 0x21},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x7D, 0x71, 0x79, 0x28, 0x2F, 0x2F, 0x18, 0x1E, 0x20},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x78, 0x6C, 0x74, 0x27, 0x2D, 0x2D, 0x16, 0x1D, 0x1F},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x73, 0x68, 0x70, 0x25, 0x2B, 0x2B, 0x16, 0x1C, 0x1D},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x6F, 0x64, 0x6B, 0x23, 0x29, 0x29, 0x14, 0x1B, 0x1C},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x6A, 0x5F, 0x66, 0x21, 0x28, 0x28, 0x14, 0x19, 0x1A},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x65, 0x5B, 0x61, 0x20, 0x25, 0x26, 0x13, 0x19, 0x1A},
	{0x01, 0x0D, 0x60, 0x48, 0x81, 0xBB, 0x33, 0x69, 0x12, 0x7A, 0x9D, 0x60, 0x56, 0x5D, 0x1F, 0x24, 0x24, 0x12, 0x18, 0x18},
};

static char irc_hbm_para_revC[8][20] = {
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x78, 0x65, 0x67, 0x15, 0x21, 0x21, 0x1E, 0x24, 0x23},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x74, 0x62, 0x63, 0x14, 0x1F, 0x20, 0x1D, 0x23, 0x22},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x6F, 0x5E, 0x5F, 0x14, 0x1E, 0x1F, 0x1C, 0x21, 0x21},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x6B, 0x5A, 0x5B, 0x13, 0x1D, 0x1E, 0x1A, 0x20, 0x1F},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x66, 0x56, 0x57, 0x12, 0x1C, 0x1C, 0x1A, 0x1F, 0x1F},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x62, 0x52, 0x54, 0x11, 0x1B, 0x1A, 0x18, 0x1D, 0x1D},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x5D, 0x4F, 0x50, 0x11, 0x19, 0x19, 0x17, 0x1C, 0x1C},
	{0x01, 0x0D, 0x4D, 0x52, 0x8A, 0xCC, 0x33, 0x69, 0x12, 0x7A, 0xBE, 0x59, 0x4B, 0x4C, 0x0F, 0x18, 0x18, 0x17, 0x1B, 0x1B},
};

static struct dsi_panel_cmds *mdss_hbm_irc(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int para_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (IS_ERR_OR_NULL(vdd->dtsi_data[ctrl->ndx].irc_tx_cmds)) {
		LCD_ERR("No irc_tx_cmds\n");
		return NULL;
	}

	if (!vdd->samsung_support_irc)
		return NULL;

	*level_key = PANEL_LEVE1_KEY;

	if (vdd->ldu_correction_state || vdd->color_weakness_value)
		return &(vdd->dtsi_data[ctrl->ndx].irc_off_tx_cmds[vdd->panel_revision]);

	/*auto_brightness is 13 to use 443cd of hbm step on color weakness mode*/
	if (vdd->auto_brightness == HBM_MODE + 7)
		para_idx = 7;
	else
		para_idx = vdd->auto_brightness_level - vdd->auto_brightness;

	/* copy irc default setting */
	if (mdss_panel_rev_get(ctrl) >= S6E3HA3_AMS567JA01_REV_C)
		memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
			irc_hbm_para_revC[para_idx], 20);
	else
		memcpy(&vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1],
			irc_hbm_para_revA[para_idx], 20);

	/* update irc enable*/
	vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = vdd->irc_info.irc_enable_status;

	/* update irc mode*/
	vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = vdd->irc_info.irc_mode;

	return &(vdd->dtsi_data[ctrl->ndx].hbm_irc_tx_cmds[vdd->panel_revision]);
}

static struct dsi_panel_cmds * mdss_gamma(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	vdd->candela_level = get_candela_value(vdd, ctrl->ndx);

	LCD_DEBUG("bl_level : %d candela : %dCD\n", vdd->bl_level, vdd->candela_level);

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi[ctrl->ndx]->generate_gamma)) {
		LCD_ERR("generate_gamma is NULL error");
		return NULL;
	} else {
		vdd->smart_dimming_dsi[ctrl->ndx]->generate_gamma(
			vdd->smart_dimming_dsi[ctrl->ndx],
			vdd->candela_level,
			&vdd->dtsi_data[ctrl->ndx].gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1]);

		*level_key = PANEL_LEVE1_KEY;

		return &vdd->dtsi_data[ctrl->ndx].gamma_tx_cmds[vdd->panel_revision];
	}
}

#define LUT_SIZE (sizeof(freq_cal_lut_offset) / sizeof(int)) / 3

static int freq_cal_lut_offset[][3] = {
	{56568, 56686,	-25},
	{56686, 56805,	-24},
	{56805, 56924,	-23},
	{56924, 57044,	-22},
	{57044, 57165,	-21},
	{57165, 57286,	-20},
	{57286, 57407,	-19},
	{57407, 57529,	-18},
	{57529, 57651,	-17},
	{57651, 57774,	-16},
	{57774, 57898,	-15},
	{57898, 58022,	-14},
	{58022, 58146,	-13},
	{58146, 58272,	-12},
	{58272, 58397,	-11},
	{58397, 58523,	-10},
	{58523, 58650,	-9},
	{58650, 58777,	-8},
	{58777, 58905,	-7},
	{58905, 59034,	-6},
	{59034, 59163,	-5},
	{59163, 59292,	-4},
	{59292, 59422,	-3},
	{59422, 59553,	-2},
	{59553, 59684,	-1},
	{59684, 59816,	0},
	{59816, 59948,	1},
	{59948, 60081,	2},
	{60081, 60215,	3},
	{60215, 60349,	4},
	{60349, 60484,	5},
	{60484, 60619,	6},
	{60619, 60755,	7},
	{60755, 60892,	8},
	{60892, 61029,	9},
	{61029, 61167,	10},
	{61167, 61305,	11},
	{61305, 61444,	12},
	{61444, 61584,	13},
	{61584, 61724,	14},
	{61724, 61865,	15},
	{61865, 62007,	16},
	{62007, 62149,	17},
	{62149, 62292,	18},
	{62292, 62436,	19},
	{62436, 62580,	20},
	{62580, 62725,	21},
	{62725, 62871,	22},
	{62871, 63017,	23},
	{63017, 63164,	24},
	{63164, 63312,	25},
	{63312, 63460,	26},
	{63460, 63609,	27},
	{63609, 63759,	28},
	{63759, 63909,	29},
	{63909, 64010,	30},
};

#define FPS_CMD_INDEX 2

static int mdss_samsung_change_ldi_fps(struct mdss_dsi_ctrl_pdata *ctrl, unsigned int input_fps)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	int offset = 0;
	int i;
	int dest_cal_val;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return 0;
	}

	LCD_DEBUG("input_fps (%d), lut size (%d)\n", input_fps, (int)LUT_SIZE);

	if (mdss_panel_id2_get(ctrl) <= 0x01) {
		LCD_ERR("LDI EVT0 Not Support. Skip!!\n");
		return 0;
	}

	for (i = 0; i < LUT_SIZE; i++) {
		if (input_fps >= freq_cal_lut_offset[i][0] &&
			input_fps < freq_cal_lut_offset[i][1]) {
			offset = freq_cal_lut_offset[i][2];
			break;
		}
	}

	if (i == LUT_SIZE) {
		LCD_ERR("can not find offset !!\n");
		return 0;
	}

	LCD_DEBUG("current comp value(0x%x),offset(%d)\n",
		vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3], offset);

	dest_cal_val = vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3] + offset;

	if((dest_cal_val < 0xAC) || (dest_cal_val > 0xE3)) {
		LCD_ERR("Invalid cal value(0x%x)", dest_cal_val);
		return 0;
	} else
		LCD_INFO("input : %d dest write value (0x%x)\n", input_fps, dest_cal_val);

	vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3] = dest_cal_val;

	return 1;
}

static int init_ldi_fps(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return 0;
	}

	if (ctrl->cmd_sync_wait_broadcast)
		if (!ctrl->cmd_sync_wait_trigger)
			return 0;

	vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3] = 0xC5;
	LCD_ERR("(0x%x)\n", vdd->dtsi_data[ctrl->ndx].ldi_fps_change_tx_cmds[vdd->panel_revision].cmds[FPS_CMD_INDEX].payload[3]);

	return 1;
}

static int samsung_panel_off_post(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int rc = 0;


	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	return rc;
}

static struct dsi_panel_cmds * mdss_gamma_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	LCD_INFO("hmt_bl_level : %d candela : %dCD\n", vdd->hmt_stat.hmt_bl_level, vdd->hmt_stat.candela_level_hmt);

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi_hmt[ctrl->ndx]->generate_gamma)) {
		LCD_ERR("generate_gamma is NULL error");
		return NULL;
	} else {
		vdd->smart_dimming_dsi_hmt[ctrl->ndx]->generate_gamma(
			vdd->smart_dimming_dsi_hmt[ctrl->ndx],
			vdd->hmt_stat.candela_level_hmt,
			&vdd->dtsi_data[ctrl->ndx].hmt_gamma_tx_cmds[vdd->panel_revision].cmds[0].payload[1]);

		*level_key = PANEL_LEVE1_KEY;

		return &vdd->dtsi_data[ctrl->ndx].hmt_gamma_tx_cmds[vdd->panel_revision];
	}
}

static struct dsi_panel_cmds hmt_aid_cmd;
static struct dsi_panel_cmds *mdss_aid_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);
	int cmd_idx = 0;

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (!vdd->dtsi_data[ctrl->ndx].hmt_reverse_aid_map_table[vdd->panel_revision].size ||
		vdd->hmt_stat.cmd_idx_hmt > vdd->dtsi_data[ctrl->ndx].hmt_reverse_aid_map_table[vdd->panel_revision].size)
		goto end;

	cmd_idx = vdd->dtsi_data[ctrl->ndx].hmt_reverse_aid_map_table[vdd->panel_revision].cmd_idx[vdd->hmt_stat.cmd_idx_hmt];

	hmt_aid_cmd.cmds = &(vdd->dtsi_data[ctrl->ndx].hmt_aid_tx_cmds[vdd->panel_revision].cmds[cmd_idx]);
	hmt_aid_cmd.cmd_cnt = 1;

	*level_key = PANEL_LEVE1_KEY;

	return &hmt_aid_cmd;

end :
	LCD_ERR("error");
	return NULL;
}

static struct dsi_panel_cmds *mdss_elvss_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	if (vdd->acl_status || vdd->siop_status) {
		vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = 0x8C;
	} else {
		vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = 0x9C;
	}

	if (vdd->bl_level < 0 || vdd->bl_level > 255)
		vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = 0x00;

	*level_key = PANEL_LEVE1_KEY;

	return &vdd->dtsi_data[ctrl->ndx].hmt_elvss_tx_cmds[vdd->panel_revision];
}

static struct dsi_panel_cmds *mdss_vint_hmt(struct mdss_dsi_ctrl_pdata *ctrl, int *level_key)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return NULL;
	}

	*level_key = PANEL_LEVE1_KEY;

	return &vdd->dtsi_data[ctrl->ndx].hmt_vint_tx_cmds[vdd->panel_revision];
}

static void mdss_make_sdimconf_hmt(struct mdss_dsi_ctrl_pdata *ctrl, struct samsung_display_driver_data *vdd) {
	/* Set the mtp read buffer pointer and read the NVM value*/
	mdss_samsung_panel_data_read(ctrl,
				&(vdd->dtsi_data[ctrl->ndx].smart_dimming_mtp_rx_cmds[vdd->panel_revision]),
				vdd->smart_dimming_dsi_hmt[ctrl->ndx]->mtp_buffer, PANEL_LEVE1_KEY);

	/* Initialize smart dimming related things here */
	/* lux_tab setting for 350cd */
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->lux_tab = vdd->dtsi_data[ctrl->ndx].hmt_candela_map_table[vdd->panel_revision].lux_tab;
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->lux_tabsize = vdd->dtsi_data[ctrl->ndx].hmt_candela_map_table[vdd->panel_revision].lux_tab_size;
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->man_id = vdd->manufacture_id_dsi[ctrl->ndx];

	/* Just a safety check to ensure smart dimming data is initialised well */
	vdd->smart_dimming_dsi_hmt[ctrl->ndx]->init(vdd->smart_dimming_dsi_hmt[ctrl->ndx]);

	LCD_INFO("[HMT] smart dimming done!\n");
}

static int mdss_samart_dimming_init_hmt(struct mdss_dsi_ctrl_pdata *ctrl)
{
	struct samsung_display_driver_data *vdd = check_valid_ctrl(ctrl);

	LCD_INFO("DSI%d : ++\n", ctrl->ndx);

	if (IS_ERR_OR_NULL(vdd)) {
		LCD_ERR("Invalid data ctrl : 0x%zx vdd : 0x%zx", (size_t)ctrl, (size_t)vdd);
		return false;
	}

	vdd->smart_dimming_dsi_hmt[ctrl->ndx] = vdd->panel_func.samsung_smart_get_conf_hmt();

	if (IS_ERR_OR_NULL(vdd->smart_dimming_dsi_hmt[ctrl->ndx])) {
		LCD_ERR("DSI%d error", ctrl->ndx);
		return false;
	} else {
		vdd->hmt_stat.hmt_on = 0;
		vdd->hmt_stat.hmt_bl_level = 0;
		vdd->hmt_stat.hmt_reverse = 0;
		vdd->hmt_stat.hmt_is_first = 1;

		mdss_make_sdimconf_hmt(ctrl, vdd);

		vdd->smart_dimming_hmt_loaded_dsi[ctrl->ndx] = true;
	}

	LCD_INFO("DSI%d : --\n", ctrl->ndx);

	return true;
}

/*
 * This function will update parameters for ALPM_REG/ALPM_CTRL_REG
 * Parameter for ALPM_REG : Control brightness for panel LPM
 * Parameter for ALPM_CTRL_REG : Change panel LPM mode for ALPM/HLPM
 */
static int mdss_update_panel_lpm_cmds(struct mdss_dsi_ctrl_pdata *ctrl, int bl_level, int mode)
{
	struct samsung_display_driver_data *vdd = NULL;
	struct dsi_panel_cmds *alpm_brightness[PANEL_LPM_BRIGHTNESS_MAX] = {NULL, };
	struct dsi_panel_cmds *alpm_ctrl[MAX_LPM_MODE] = {NULL, };
	struct dsi_panel_cmds *cmd_list[2];
	/*
	 * Init reg_list and cmd list
	 * reg_list[X][0] is reg value
	 * reg_list[X][1] is offset for reg value
	 * cmd_list is the target cmds for searching reg value
	 */
	static int reg_list[2][2] = {
		{ALPM_REG, -EINVAL},
		{ALPM_CTRL_REG, -EINVAL}};

	if (IS_ERR_OR_NULL(ctrl))
		goto end;

	vdd = check_valid_ctrl(ctrl);

	cmd_list[0] = &vdd->dtsi_data[ctrl->ndx].alpm_on_tx_cmds[vdd->panel_revision];
	cmd_list[1] = &vdd->dtsi_data[ctrl->ndx].alpm_on_tx_cmds[vdd->panel_revision];
	/* Init alpm_brightness and alpm_ctrl cmds */
	alpm_brightness[PANEL_LPM_2NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_2nit_tx_cmds[vdd->panel_revision];
	alpm_brightness[PANEL_LPM_40NIT] = NULL;
	alpm_brightness[PANEL_LPM_60NIT] =
		&vdd->dtsi_data[ctrl->ndx].lpm_40nit_tx_cmds[vdd->panel_revision];

	alpm_ctrl[MODE_OFF] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_alpm_tx_cmds[vdd->panel_revision];
	alpm_ctrl[ALPM_MODE_ON] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_alpm_tx_cmds[vdd->panel_revision];
	alpm_ctrl[HLPM_MODE_ON] =
		&vdd->dtsi_data[ctrl->ndx].lpm_ctrl_hlpm_tx_cmds[vdd->panel_revision];

	/*
	 * Find offset for alpm_reg and alpm_ctrl_reg
	 * alpm_reg		 : Control register for ALPM/HLPM on/off
	 * alpm_ctrl_reg : Control register for changing ALPM/HLPM mode
	 */
	mdss_init_panel_lpm_reg_offset(ctrl, reg_list, cmd_list,
			sizeof(cmd_list) / sizeof(cmd_list[0]));

	if (reg_list[0][1] != -EINVAL) {
		/* Update parameter for ALPM_REG */
		memcpy(cmd_list[0]->cmds[reg_list[0][1]].payload,
				alpm_brightness[bl_level]->cmds[0].payload,
				sizeof(char) * cmd_list[0]->cmds[reg_list[0][1]].dchdr.dlen);

		LCD_DEBUG("[Panel LPM] change brightness cmd : %x, %x\n",
				cmd_list[0]->cmds[reg_list[0][1]].payload[1],
				alpm_brightness[bl_level]->cmds[0].payload[1]);
	}

	if (reg_list[1][1] != -EINVAL) {
		/* Initialize ALPM/HLPM cmds */
		/* Update parameter for ALPM_CTRL_REG */
		memcpy(cmd_list[1]->cmds[reg_list[1][1]].payload,
				alpm_ctrl[mode]->cmds[0].payload,
				sizeof(char) * cmd_list[1]->cmds[reg_list[1][1]].dchdr.dlen);
		LCD_DEBUG("[Panel LPM] update alpm ctrl reg(%d)\n", mode);
	}

end:
	return 0;
}

static void mdss_get_panel_lpm_mode(struct mdss_dsi_ctrl_pdata *ctrl, u8 *mode)
{
	struct samsung_display_driver_data *vdd = NULL;
	int panel_lpm_mode = 0, lpm_bl_level = 0;

	if (IS_ERR_OR_NULL(ctrl))
		return;

	/*
	 * if the mode value is lower than MAX_LPM_MODE
	 * this function was not called by mdss_samsung_alpm_store()
	 * so the mode will not be changed
	 */
	if (*mode < MAX_LPM_MODE)
		return;

	vdd = check_valid_ctrl(ctrl);

	/* default Hz is 30Hz */
	vdd->panel_lpm.hz = PANEL_LPM_30HZ;

	/* Check mode and bl_level */
	switch (*mode) {
		case AOD_MODE_ALPM_2NIT_ON:
			panel_lpm_mode = ALPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_2NIT;
			break;
		case AOD_MODE_HLPM_2NIT_ON:
			panel_lpm_mode = HLPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_2NIT;
			break;
		case AOD_MODE_ALPM_40NIT_ON:
			panel_lpm_mode = ALPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_60NIT;
			break;
		case AOD_MODE_HLPM_40NIT_ON:
			panel_lpm_mode = HLPM_MODE_ON;
			lpm_bl_level = PANEL_LPM_60NIT;
			break;
		default:
			panel_lpm_mode = MODE_OFF;
			break;
	}

	*mode = panel_lpm_mode;

	/* Save mode and bl_level */
	vdd->panel_lpm.lpm_bl_level = lpm_bl_level;

	mdss_update_panel_lpm_cmds(ctrl, lpm_bl_level, panel_lpm_mode);
}

static void mdss_send_colorweakness_ccb_cmd(struct samsung_display_driver_data *vdd, int mode)
{
	int i;
	int step_size = (0x3F - 0x2A)+1;

	LCD_INFO("mode (%d) color_weakness_value (%x) \n", mode, vdd->color_weakness_value);

	mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_LEVE1_KEY_ENABLE);
	if (mode) {
		vdd->dtsi_data[DISPLAY_1].ccb_on_tx_cmds[vdd->panel_revision].cmds[0].payload[1] = vdd->color_weakness_value;
		mutex_lock(&vdd->mfd_dsi[DISPLAY_1]->bl_lock);
		for (i = 0; i < step_size; i++){
			vdd->dtsi_data[DISPLAY_1].ccb_on_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = 0x3F - i;
			mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_COLOR_WEAKNESS_ENABLE);
			msleep(10);
		}
		mutex_unlock(&vdd->mfd_dsi[DISPLAY_1]->bl_lock);
	} else {
		mutex_lock(&vdd->mfd_dsi[DISPLAY_1]->bl_lock);
		for (i = 0; i < step_size; i++){
			vdd->dtsi_data[DISPLAY_1].ccb_off_tx_cmds[vdd->panel_revision].cmds[0].payload[2] = 0x2A + i;
			mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_COLOR_WEAKNESS_DISABLE);
			msleep(10);
		}
		mutex_unlock(&vdd->mfd_dsi[DISPLAY_1]->bl_lock);
	}
	mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_LEVE1_KEY_DISABLE);
}

static int default_res_set = 0;

static void mdss_panel_multires(struct samsung_display_driver_data *vdd)
{
	if (is_boot_recovery && !default_res_set) {
		LCD_INFO("recovery booting!! set to default : %s\n", vdd->resolution_default);
		sec_set_param(param_index_lcd_resolution,  vdd->resolution_default);
		default_res_set = 1;
	}

	if(vdd->multires_stat.prev_mode != vdd->multires_stat.curr_mode)
	{
		LCD_INFO("vdd->multires_stat.prev_mode = %d, vdd-multires_stat.curr_mode = %d\n",
			vdd->multires_stat.prev_mode, vdd->multires_stat.curr_mode);

		switch(vdd->multires_stat.curr_mode)
		{
			case 0:
				sec_set_param(param_index_lcd_resolution, "WQHD");
				break;
			case 1:
				sec_set_param(param_index_lcd_resolution, "FHD");
				break;
			case 2:
				sec_set_param(param_index_lcd_resolution, "HD");
				break;
		}
		mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_DISPLAY_OFF);
		vdd->multires_stat.black_frame_cnt = 5;

		usleep_range(20000, 20000);

		if(vdd->multires_stat.curr_mode == MULTIRES_FHD)
			mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_FHD);
		else if(vdd->multires_stat.curr_mode == MULTIRES_HD)
			mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_HD);
		else if(vdd->multires_stat.curr_mode == MULTIRES_WQHD)
		{
			if(vdd->multires_stat.prev_mode == MULTIRES_FHD)
				mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_FHD_TO_WQHD);
			else
				mdss_samsung_send_cmd(vdd->ctrl_dsi[DISPLAY_1], PANEL_MULTIRES_HD_TO_WQHD);
		}
		vdd->multires_stat.prev_mode = vdd->multires_stat.curr_mode;
	}
}

static void dsi_update_mdnie_data(void)
{
	/* Update mdnie command */
	mdnie_data.DSI0_COLOR_BLIND_MDNIE_1 = DSI0_COLOR_BLIND_MDNIE_1;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_1 = DSI0_RGB_SENSOR_MDNIE_1;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_2 = DSI0_RGB_SENSOR_MDNIE_2;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_3 = DSI0_RGB_SENSOR_MDNIE_3;
	mdnie_data.DSI0_TRANS_DIMMING_MDNIE = DSI0_RGB_SENSOR_MDNIE_1;
	mdnie_data.DSI0_UI_DYNAMIC_MDNIE_2 = DSI0_UI_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_UI_STANDARD_MDNIE_2 = DSI0_UI_STANDARD_MDNIE_2;
	mdnie_data.DSI0_UI_AUTO_MDNIE_2 = DSI0_UI_AUTO_MDNIE_2;
	mdnie_data.DSI0_VIDEO_DYNAMIC_MDNIE_2 = DSI0_VIDEO_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_VIDEO_STANDARD_MDNIE_2 = DSI0_VIDEO_STANDARD_MDNIE_2;
	mdnie_data.DSI0_VIDEO_AUTO_MDNIE_2 = DSI0_VIDEO_AUTO_MDNIE_2;
	mdnie_data.DSI0_CAMERA_MDNIE_2 = DSI0_CAMERA_MDNIE_2;
	mdnie_data.DSI0_CAMERA_AUTO_MDNIE_2 = DSI0_CAMERA_AUTO_MDNIE_2;
	mdnie_data.DSI0_GALLERY_DYNAMIC_MDNIE_2 = DSI0_GALLERY_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_GALLERY_STANDARD_MDNIE_2 = DSI0_GALLERY_STANDARD_MDNIE_2;
	mdnie_data.DSI0_GALLERY_AUTO_MDNIE_2 = DSI0_GALLERY_AUTO_MDNIE_2;
	mdnie_data.DSI0_VT_DYNAMIC_MDNIE_2 = DSI0_VT_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_VT_STANDARD_MDNIE_2 = DSI0_VT_STANDARD_MDNIE_2;
	mdnie_data.DSI0_VT_AUTO_MDNIE_2 = DSI0_VT_AUTO_MDNIE_2;
	mdnie_data.DSI0_BROWSER_DYNAMIC_MDNIE_2 = DSI0_BROWSER_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_BROWSER_STANDARD_MDNIE_2 = DSI0_BROWSER_STANDARD_MDNIE_2;
	mdnie_data.DSI0_BROWSER_AUTO_MDNIE_2 = DSI0_BROWSER_AUTO_MDNIE_2;
	mdnie_data.DSI0_EBOOK_DYNAMIC_MDNIE_2 = DSI0_EBOOK_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_EBOOK_STANDARD_MDNIE_2 = DSI0_EBOOK_STANDARD_MDNIE_2;
	mdnie_data.DSI0_EBOOK_AUTO_MDNIE_2 = DSI0_EBOOK_AUTO_MDNIE_2;
	mdnie_data.DSI0_TDMB_DYNAMIC_MDNIE_2 = DSI0_TDMB_DYNAMIC_MDNIE_2;
	mdnie_data.DSI0_TDMB_STANDARD_MDNIE_2 = DSI0_TDMB_STANDARD_MDNIE_2;
	mdnie_data.DSI0_TDMB_AUTO_MDNIE_2 = DSI0_TDMB_AUTO_MDNIE_2;

	mdnie_data.DSI0_BYPASS_MDNIE = DSI0_BYPASS_MDNIE;
	mdnie_data.DSI0_NEGATIVE_MDNIE = DSI0_NEGATIVE_MDNIE;
	mdnie_data.DSI0_COLOR_BLIND_MDNIE = DSI0_COLOR_BLIND_MDNIE;
	mdnie_data.DSI0_HBM_CE_MDNIE = DSI0_HBM_CE_MDNIE;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE = DSI0_RGB_SENSOR_MDNIE;
	mdnie_data.DSI0_UI_DYNAMIC_MDNIE = DSI0_UI_DYNAMIC_MDNIE;
	mdnie_data.DSI0_UI_STANDARD_MDNIE = DSI0_UI_STANDARD_MDNIE;
	mdnie_data.DSI0_UI_NATURAL_MDNIE = DSI0_UI_NATURAL_MDNIE;
	mdnie_data.DSI0_UI_MOVIE_MDNIE = DSI0_UI_MOVIE_MDNIE;
	mdnie_data.DSI0_UI_AUTO_MDNIE = DSI0_UI_AUTO_MDNIE;
	mdnie_data.DSI0_VIDEO_OUTDOOR_MDNIE = NULL;
	mdnie_data.DSI0_VIDEO_DYNAMIC_MDNIE = DSI0_VIDEO_DYNAMIC_MDNIE;
	mdnie_data.DSI0_VIDEO_STANDARD_MDNIE = DSI0_VIDEO_STANDARD_MDNIE;
	mdnie_data.DSI0_VIDEO_NATURAL_MDNIE = DSI0_VIDEO_NATURAL_MDNIE;
	mdnie_data.DSI0_VIDEO_MOVIE_MDNIE = DSI0_VIDEO_MOVIE_MDNIE;
	mdnie_data.DSI0_VIDEO_AUTO_MDNIE = DSI0_VIDEO_AUTO_MDNIE;
	mdnie_data.DSI0_VIDEO_WARM_OUTDOOR_MDNIE = DSI0_VIDEO_WARM_OUTDOOR_MDNIE;
	mdnie_data.DSI0_VIDEO_WARM_MDNIE = DSI0_VIDEO_WARM_MDNIE;
	mdnie_data.DSI0_VIDEO_COLD_OUTDOOR_MDNIE = DSI0_VIDEO_COLD_OUTDOOR_MDNIE;
	mdnie_data.DSI0_VIDEO_COLD_MDNIE = DSI0_VIDEO_COLD_MDNIE;
	mdnie_data.DSI0_CAMERA_OUTDOOR_MDNIE = DSI0_CAMERA_OUTDOOR_MDNIE;
	mdnie_data.DSI0_CAMERA_MDNIE = DSI0_CAMERA_MDNIE;
	mdnie_data.DSI0_CAMERA_AUTO_MDNIE = DSI0_CAMERA_AUTO_MDNIE;
	mdnie_data.DSI0_GALLERY_DYNAMIC_MDNIE = DSI0_GALLERY_DYNAMIC_MDNIE;
	mdnie_data.DSI0_GALLERY_STANDARD_MDNIE = DSI0_GALLERY_STANDARD_MDNIE;
	mdnie_data.DSI0_GALLERY_NATURAL_MDNIE = DSI0_GALLERY_NATURAL_MDNIE;
	mdnie_data.DSI0_GALLERY_MOVIE_MDNIE = DSI0_GALLERY_MOVIE_MDNIE;
	mdnie_data.DSI0_GALLERY_AUTO_MDNIE = DSI0_GALLERY_AUTO_MDNIE;
	mdnie_data.DSI0_VT_DYNAMIC_MDNIE = DSI0_VT_DYNAMIC_MDNIE;
	mdnie_data.DSI0_VT_STANDARD_MDNIE = DSI0_VT_STANDARD_MDNIE;
	mdnie_data.DSI0_VT_NATURAL_MDNIE = DSI0_VT_NATURAL_MDNIE;
	mdnie_data.DSI0_VT_MOVIE_MDNIE = DSI0_VT_MOVIE_MDNIE;
	mdnie_data.DSI0_VT_AUTO_MDNIE = DSI0_VT_AUTO_MDNIE;
	mdnie_data.DSI0_BROWSER_DYNAMIC_MDNIE = DSI0_BROWSER_DYNAMIC_MDNIE;
	mdnie_data.DSI0_BROWSER_STANDARD_MDNIE = DSI0_BROWSER_STANDARD_MDNIE;
	mdnie_data.DSI0_BROWSER_NATURAL_MDNIE = DSI0_BROWSER_NATURAL_MDNIE;
	mdnie_data.DSI0_BROWSER_MOVIE_MDNIE = DSI0_BROWSER_MOVIE_MDNIE;
	mdnie_data.DSI0_BROWSER_AUTO_MDNIE = DSI0_BROWSER_AUTO_MDNIE;
	mdnie_data.DSI0_EBOOK_DYNAMIC_MDNIE = DSI0_EBOOK_DYNAMIC_MDNIE;
	mdnie_data.DSI0_EBOOK_STANDARD_MDNIE = DSI0_EBOOK_STANDARD_MDNIE;
	mdnie_data.DSI0_EBOOK_NATURAL_MDNIE = DSI0_EBOOK_NATURAL_MDNIE;
	mdnie_data.DSI0_EBOOK_MOVIE_MDNIE = DSI0_EBOOK_MOVIE_MDNIE;
	mdnie_data.DSI0_EBOOK_AUTO_MDNIE = DSI0_EBOOK_AUTO_MDNIE;
	mdnie_data.DSI0_EMAIL_AUTO_MDNIE = DSI0_EMAIL_AUTO_MDNIE;
	mdnie_data.DSI0_GAME_LOW_MDNIE = DSI0_GAME_LOW_MDNIE;
	mdnie_data.DSI0_GAME_MID_MDNIE = DSI0_GAME_MID_MDNIE;
	mdnie_data.DSI0_GAME_HIGH_MDNIE = DSI0_GAME_HIGH_MDNIE;
	mdnie_data.DSI0_TDMB_DYNAMIC_MDNIE = DSI0_TDMB_DYNAMIC_MDNIE;
	mdnie_data.DSI0_TDMB_STANDARD_MDNIE = DSI0_TDMB_STANDARD_MDNIE;
	mdnie_data.DSI0_TDMB_NATURAL_MDNIE = DSI0_TDMB_NATURAL_MDNIE;
	mdnie_data.DSI0_TDMB_MOVIE_MDNIE = DSI0_TDMB_MOVIE_MDNIE;
	mdnie_data.DSI0_TDMB_AUTO_MDNIE = DSI0_TDMB_AUTO_MDNIE;
	mdnie_data.DSI0_GRAYSCALE_MDNIE = DSI0_GRAYSCALE_MDNIE;
	mdnie_data.DSI0_GRAYSCALE_NEGATIVE_MDNIE= DSI0_GRAYSCALE_NEGATIVE_MDNIE;
	mdnie_data.DSI0_CURTAIN = DSI0_SCREEN_CURTAIN_MDNIE;
	mdnie_data.DSI0_NIGHT_MODE_MDNIE = DSI0_NIGHT_MODE_MDNIE;
	mdnie_data.DSI0_NIGHT_MODE_MDNIE_SCR = DSI0_NIGHT_MODE_MDNIE_1;
	mdnie_data.DSI0_COLOR_BLIND_MDNIE_SCR = DSI0_COLOR_BLIND_MDNIE_1;
	mdnie_data.DSI0_RGB_SENSOR_MDNIE_SCR = DSI0_RGB_SENSOR_MDNIE_1;
	mdnie_data.DSI0_COLOR_LENS_MDNIE = DSI0_COLOR_LENS_MDNIE;
	mdnie_data.DSI0_COLOR_LENS_MDNIE_SCR = DSI0_COLOR_LENS_MDNIE_1;

	mdnie_data.mdnie_tune_value_dsi0 = mdnie_tune_value_dsi0;
	mdnie_data.mdnie_tune_value_dsi1 = mdnie_tune_value_dsi0;
	mdnie_data.hmt_color_temperature_tune_value_dsi0 = hmt_color_temperature_tune_value_dsi0;
	mdnie_data.hmt_color_temperature_tune_value_dsi1 = hmt_color_temperature_tune_value_dsi0;

	mdnie_data.hdr_tune_value_dsi0 = hdr_tune_value_dsi0;
	mdnie_data.hdr_tune_value_dsi1 = hdr_tune_value_dsi0;
	
	mdnie_data.light_notification_tune_value_dsi0 = light_notification_tune_value_dsi0;
	mdnie_data.light_notification_tune_value_dsi1 = light_notification_tune_value_dsi0;

	/* Update MDNIE data related with size, offset or index */
	mdnie_data.dsi0_bypass_mdnie_size = ARRAY_SIZE(DSI0_BYPASS_MDNIE);
	mdnie_data.mdnie_color_blinde_cmd_offset = MDNIE_COLOR_BLINDE_CMD_OFFSET;
	mdnie_data.mdnie_step_index[MDNIE_STEP1] = MDNIE_STEP1_INDEX;
	mdnie_data.mdnie_step_index[MDNIE_STEP2] = MDNIE_STEP2_INDEX;
	mdnie_data.mdnie_step_index[MDNIE_STEP3] = MDNIE_STEP3_INDEX;
	mdnie_data.address_scr_white[ADDRESS_SCR_WHITE_RED_OFFSET] = ADDRESS_SCR_WHITE_RED;
	mdnie_data.address_scr_white[ADDRESS_SCR_WHITE_GREEN_OFFSET] = ADDRESS_SCR_WHITE_GREEN;
	mdnie_data.address_scr_white[ADDRESS_SCR_WHITE_BLUE_OFFSET] = ADDRESS_SCR_WHITE_BLUE;
	mdnie_data.dsi0_rgb_sensor_mdnie_1_size = DSI0_RGB_SENSOR_MDNIE_1_SIZE;
	mdnie_data.dsi0_rgb_sensor_mdnie_2_size = DSI0_RGB_SENSOR_MDNIE_2_SIZE;
	mdnie_data.dsi0_rgb_sensor_mdnie_3_size = DSI0_RGB_SENSOR_MDNIE_3_SIZE;

	mdnie_data.dsi0_trans_dimming_data_index = MDNIE_TRANS_DIMMING_DATA_INDEX;

	mdnie_data.dsi0_adjust_ldu_table = adjust_ldu_data;
	mdnie_data.dsi1_adjust_ldu_table = adjust_ldu_data;
	mdnie_data.dsi0_max_adjust_ldu = 6;
	mdnie_data.dsi1_max_adjust_ldu = 6;
	mdnie_data.dsi0_night_mode_table = night_mode_data;
	mdnie_data.dsi1_night_mode_table = night_mode_data;
	mdnie_data.dsi0_max_night_mode_index = 11;
	mdnie_data.dsi1_max_night_mode_index = 11;
	mdnie_data.dsi0_color_lens_table = color_lens_data;
	mdnie_data.dsi1_color_lens_table = NULL;
	mdnie_data.dsi0_scr_step_index = MDNIE_STEP1_INDEX;
	mdnie_data.dsi1_scr_step_index = MDNIE_STEP1_INDEX;
	mdnie_data.dsi0_white_default_r = 0xff;
	mdnie_data.dsi0_white_default_g = 0xff;
	mdnie_data.dsi0_white_default_b = 0xff;
	mdnie_data.dsi0_white_rgb_enabled = 0;
	mdnie_data.dsi1_white_default_r = 0xff;
	mdnie_data.dsi1_white_default_g = 0xff;
	mdnie_data.dsi1_white_default_b = 0xff;
	mdnie_data.dsi1_white_rgb_enabled = 0;
}

static void  mdss_panel_init(struct samsung_display_driver_data *vdd)
{
	char resolution_default[] = "FHD";

	LCD_ERR("%s", vdd->panel_name);

	vdd->support_mdnie_lite = true;

	vdd->support_mdnie_trans_dimming = true;

	vdd->mdnie_tune_size1 = sizeof(DSI0_BYPASS_MDNIE_1);
	vdd->mdnie_tune_size2 = sizeof(DSI0_BYPASS_MDNIE_2);
	vdd->mdnie_tune_size3 = sizeof(DSI0_BYPASS_MDNIE_3);

	/* ON/OFF */
	vdd->panel_func.samsung_panel_on_pre = mdss_panel_on_pre;
	vdd->panel_func.samsung_panel_on_post = mdss_panel_on_post;
	vdd->panel_func.samsung_panel_off_post = samsung_panel_off_post;

	/* DDI RX */
	vdd->panel_func.samsung_panel_revision = mdss_panel_revision;
	vdd->panel_func.samsung_manufacture_date_read = mdss_manufacture_date_read;
	vdd->panel_func.samsung_ddi_id_read = mdss_ddi_id_read;

	vdd->panel_func.samsung_hbm_read = NULL;
	vdd->panel_func.samsung_mdnie_read = NULL;
	vdd->panel_func.samsung_smart_dimming_init = NULL;
	vdd->panel_func.samsung_smart_get_conf = NULL;

	vdd->panel_func.samsung_elvss_read = mdss_elvss_read;
	vdd->panel_func.samsung_hbm_read = mdss_hbm_read;
	vdd->panel_func.samsung_mdnie_read = mdss_mdnie_read;
	vdd->panel_func.samsung_smart_dimming_init = mdss_samart_dimming_init;
	vdd->panel_func.samsung_smart_get_conf = smart_get_conf_S6E3HA3_AMS567JA01;

	vdd->panel_func.samsung_cell_id_read = mdss_cell_id_read;

	/* Brightness */
	vdd->panel_func.samsung_brightness_hbm_off = NULL;
	vdd->panel_func.samsung_brightness_aid = mdss_aid;
	vdd->panel_func.samsung_brightness_acl_on = mdss_acl_on;
	vdd->panel_func.samsung_brightness_acl_percent = NULL;
	vdd->panel_func.samsung_brightness_acl_off = mdss_acl_off;
	vdd->panel_func.samsung_brightness_elvss = mdss_elvss;
	vdd->panel_func.samsung_brightness_elvss_temperature1 = mdss_elvss_temperature1;
	vdd->panel_func.samsung_brightness_elvss_temperature2 = NULL;
	vdd->panel_func.samsung_brightness_vint = mdss_vint;
	vdd->panel_func.samsung_brightness_irc = mdss_irc;
	vdd->panel_func.samsung_brightness_gamma = mdss_gamma;

	/* HBM */
	vdd->panel_func.samsung_hbm_gamma = mdss_hbm_gamma;
	vdd->panel_func.samsung_hbm_etc = mdss_hbm_etc;
	vdd->panel_func.samsung_hbm_irc = mdss_hbm_irc;
	vdd->panel_func.get_hbm_candela_value = get_hbm_candela_value;

	/* Event */
	vdd->panel_func.samsung_change_ldi_fps = mdss_samsung_change_ldi_fps;

	/* HMT */
	vdd->panel_func.samsung_brightness_gamma_hmt = mdss_gamma_hmt;
	vdd->panel_func.samsung_brightness_aid_hmt = mdss_aid_hmt;
	vdd->panel_func.samsung_brightness_elvss_hmt = mdss_elvss_hmt;
	vdd->panel_func.samsung_brightness_vint_hmt = mdss_vint_hmt;
	vdd->panel_func.samsung_smart_dimming_hmt_init = mdss_samart_dimming_init_hmt;
	vdd->panel_func.samsung_smart_get_conf_hmt = smart_get_conf_S6E3HA3_AMS567JA01_hmt;

	dsi_update_mdnie_data();

	/* Panel LPM */
	vdd->panel_func.samsung_get_panel_lpm_mode = mdss_get_panel_lpm_mode;

	vdd->manufacture_id_dsi[0] = PBA_ID;

	/* A3 line panel data parsing fn */
	vdd->panel_func.parsing_otherline_pdata = NULL;

	/* send recovery pck before sending image date (for ESD recovery) */
	vdd->send_esd_recovery = true;

	vdd->auto_brightness_level = 12;

	/* Enable panic on first pingpong timeout */
	vdd->debug_data->panic_on_pptimeout = true;

	/* Set IRC init value */
	vdd->irc_info.irc_enable_status = true;
	vdd->irc_info.irc_mode = IRC_LRU_MODE;

	/* COLOR WEAKNESS */
	vdd->panel_func.color_weakness_ccb_on_off = mdss_send_colorweakness_ccb_cmd;

	/* MULTI_RESOLUTION */
	vdd->panel_func.samsung_multires = mdss_panel_multires;
	memcpy(vdd->resolution_default, resolution_default, sizeof(resolution_default));

	/* ACL default ON */
	vdd->acl_status = 1;
}

static int __init samsung_panel_init(void)
{
	struct samsung_display_driver_data *vdd = samsung_get_vdd();
	char panel_string[] = "ss_dsi_panel_S6E3HA3_AMS567JA01_WQHD";

	vdd->panel_name = mdss_mdp_panel + 8;
	LCD_INFO("%s \n", vdd->panel_name);

	if (!strncmp(vdd->panel_name, panel_string, strlen(panel_string)))
		vdd->panel_func.samsung_panel_init = mdss_panel_init;

	return 0;
}

early_initcall(samsung_panel_init);
