/*
 * kernel/power/wakeup_reason.c
 *
 * Logs the reasons which caused the kernel to resume from
 * the suspend mode.
 *
 * Copyright (C) 2014 Google, Inc.
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/wakeup_reason.h>
#include <linux/kernel.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/kobject.h>
#include <linux/sysfs.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/notifier.h>
#include <linux/suspend.h>

#define MAX_WAKEUP_REASON_IRQS 32
static int irq_list[MAX_WAKEUP_REASON_IRQS];
static int irqcount;
static struct kobject *wakeup_reason;
static spinlock_t resume_reason_lock;

static ssize_t reason_show(struct kobject *kobj, struct kobj_attribute *attr,
		char *buf)
{
	int irq_no, buf_offset = 0;
	struct irq_desc *desc;
	spin_lock(&resume_reason_lock);
	for (irq_no = 0; irq_no < irqcount; irq_no++) {
		desc = irq_to_desc(irq_list[irq_no]);
		if (desc && desc->action && desc->action->name)
			buf_offset += sprintf(buf + buf_offset, "%d %s\n",
					irq_list[irq_no], desc->action->name);
		else
			buf_offset += sprintf(buf + buf_offset, "%d\n",
					irq_list[irq_no]);
	}
	spin_unlock(&resume_reason_lock);
	return buf_offset;
}

static struct kobj_attribute resume_reason = __ATTR(last_resume_reason, 0444,
		reason_show, NULL);

static struct attribute *attrs[] = {
	&resume_reason.attr,
	NULL,
};
static struct attribute_group attr_group = {
	.attrs = attrs,
};

/*
 * logs all the wake up reasons to the kernel
 * stores the irqs to expose them to the userspace via sysfs
 */
void log_wakeup_reason(int irq)
{
	struct irq_desc *desc;
	desc = irq_to_desc(irq);
	if (desc && desc->action && desc->action->name)
		printk(KERN_INFO "Resume caused by IRQ %d, %s\n", irq,
				desc->action->name);
	else
		printk(KERN_INFO "Resume caused by IRQ %d\n", irq);

	spin_lock(&resume_reason_lock);
	if (irqcount == MAX_WAKEUP_REASON_IRQS) {
		spin_unlock(&resume_reason_lock);
		printk(KERN_WARNING "Resume caused by more than %d IRQs\n",
				MAX_WAKEUP_REASON_IRQS);
		return;
	}

	irq_list[irqcount++] = irq;
	spin_unlock(&resume_reason_lock);
}

/* Detects a suspend and clears all the previous wake up reasons*/
static int wakeup_reason_pm_event(struct notifier_block *notifier,
		unsigned long pm_event, void *unused)
{
	switch (pm_event) {
	case PM_SUSPEND_PREPARE:
		spin_lock(&resume_reason_lock);
		irqcount = 0;
		spin_unlock(&resume_reason_lock);
		break;
	default:
		break;
	}
	return NOTIFY_DONE;
}

static struct notifier_block wakeup_reason_pm_notifier_block = {
	.notifier_call = wakeup_reason_pm_event,
};

/* Initializes the sysfs parameter
 * registers the pm_event notifier
 */
int __init wakeup_reason_init(void)
{
	int retval;
	spin_lock_init(&resume_reason_lock);
	retval = register_pm_notifier(&wakeup_reason_pm_notifier_block);
	if (retval)
		printk(KERN_WARNING "[%s] failed to register PM notifier %d\n",
				__func__, retval);

	wakeup_reason = kobject_create_and_add("wakeup_reasons", kernel_kobj);
	if (!wakeup_reason) {
		printk(KERN_WARNING "[%s] failed to create a sysfs kobject\n",
				__func__);
		return 1;
	}
	retval = sysfs_create_group(wakeup_reason, &attr_group);
	if (retval) {
		kobject_put(wakeup_reason);
		printk(KERN_WARNING "[%s] failed to create a sysfs group %d\n",
				__func__, retval);
	}
	return 0;
}

late_initcall(wakeup_reason_init);

#ifdef CONFIG_ARCH_MSM
#include <linux/debugfs.h>
#define NR_TOTAL_IRQS	1024

struct wakeup_reason_stats {
	unsigned int wakeup_count;
};
static struct wakeup_reason_stats wakeup_reason_stats[NR_TOTAL_IRQS] = {{0,},};

void update_wakeup_reason_stats(int irq)
{
	pr_info("IRQ %d [%ps]\n", irq, __builtin_return_address(0));
	wakeup_reason_stats[irq].wakeup_count++;
}

#ifdef CONFIG_DEBUG_FS
static int wakeup_reason_stats_show(struct seq_file *s, void *unused)
{
	int i;

	pr_info("nr_irqs: %d\n", nr_irqs);
	seq_puts(s, "irq\twakeup_count\tname\n");
	for (i = 0; i < nr_irqs; i++) {
		if (wakeup_reason_stats[i].wakeup_count > 0) {
			struct irq_desc *desc = irq_to_desc(i);
			const char *irq_name = NULL;

			if (desc && desc->action && desc->action->name)
				irq_name = desc->action->name;

			seq_printf(s, "%d\t%u\t%s\n", i,
					wakeup_reason_stats[i].wakeup_count, irq_name);
		}
	}

	return 0;
}

static int wakeup_reason_stats_open(struct inode *inode, struct file *file)
{
	return single_open(file, wakeup_reason_stats_show, NULL);
}

static const struct file_operations wakeup_reason_stats_ops = {
	.open           = wakeup_reason_stats_open,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = single_release,
};

static int __init wakeup_reason_debugfs_init(void)
{
	debugfs_create_file("wakeup_reason_stats", S_IFREG | S_IRUGO,
			NULL, NULL, &wakeup_reason_stats_ops);
	return 0;
}

late_initcall(wakeup_reason_debugfs_init);
#endif /* CONFIG_DEBUG_FS */
#endif /* CONFIG_ARCH_MSM */
