/* Copyright (c) 2012-2013, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#include <linux/debugfs.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/of.h>
#include <asm/uaccess.h>

#include <mach/msm_iomap.h>
#include "rpm_stats.h"

#define RPM_MASTERS_BUF_LEN 400

#define SNPRINTF(buf, size, format, ...) \
	do { \
		if (size > 0) { \
			int ret; \
			ret = snprintf(buf, size, format, ## __VA_ARGS__); \
			if (ret > size) { \
				buf += size; \
				size = 0; \
			} else { \
				buf += ret; \
				size -= ret; \
			} \
		} \
	} while (0)

#define GET_MASTER_NAME(a, prvdata) \
	((a >= prvdata->num_masters) ? "Invalid Master Name" : \
	 prvdata->master_names[a])

#define GET_FIELD(a) ((strnstr(#a, ".", 80) + 1))

struct msm_rpm_master_stats {
	uint32_t active_cores;
	uint32_t numshutdowns;
	uint64_t shutdown_req;
	uint64_t wakeup_ind;
	uint64_t bringup_req;
	uint64_t bringup_ack;
	uint32_t wakeup_reason; /* 0 = rude wakeup, 1 = scheduled wakeup */
	uint32_t last_sleep_transition_duration;
	uint32_t last_wake_transition_duration;
};

struct msm_rpm_master_stats_private_data {
	void __iomem *reg_base;
	u32 len;
	char **master_names;
	u32 num_masters;
	char buf[RPM_MASTERS_BUF_LEN];
	struct msm_rpm_master_stats_platform_data *platform_data;
};

int msm_rpm_master_stats_file_close(struct inode *inode,
		struct file *file)
{
	struct msm_rpm_master_stats_private_data *private = file->private_data;

	if (private->reg_base)
		iounmap(private->reg_base);
	kfree(file->private_data);

	return 0;
}

static int msm_rpm_master_copy_stats(
		struct msm_rpm_master_stats_private_data *prvdata)
{
	struct msm_rpm_master_stats record;
	struct msm_rpm_master_stats_platform_data *pdata;
	static int master_cnt;
	int count, j = 0;
	char *buf;
	static DEFINE_MUTEX(msm_rpm_master_stats_mutex);

	mutex_lock(&msm_rpm_master_stats_mutex);

	/* Iterate possible number of masters */
	if (master_cnt > prvdata->num_masters - 1) {
		master_cnt = 0;
		mutex_unlock(&msm_rpm_master_stats_mutex);
		return 0;
	}

	pdata = prvdata->platform_data;
	count = RPM_MASTERS_BUF_LEN;
	buf = prvdata->buf;

	if (prvdata->platform_data->version == 2) {
		SNPRINTF(buf, count, "%s\n",
				GET_MASTER_NAME(master_cnt, prvdata));

		record.shutdown_req = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, shutdown_req)));

		SNPRINTF(buf, count, "\t%s:0x%llX\n",
			GET_FIELD(record.shutdown_req),
			record.shutdown_req);

		record.wakeup_ind = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, wakeup_ind)));

		SNPRINTF(buf, count, "\t%s:0x%llX\n",
			GET_FIELD(record.wakeup_ind),
			record.wakeup_ind);

		record.bringup_req = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, bringup_req)));

		SNPRINTF(buf, count, "\t%s:0x%llX\n",
			GET_FIELD(record.bringup_req),
			record.bringup_req);

		record.bringup_ack = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, bringup_ack)));

		SNPRINTF(buf, count, "\t%s:0x%llX\n",
			GET_FIELD(record.bringup_ack),
			record.bringup_ack);

		record.last_sleep_transition_duration =
				readl_relaxed(prvdata->reg_base +
				(master_cnt * pdata->master_offset +
				offsetof(struct msm_rpm_master_stats,
				last_sleep_transition_duration)));

		SNPRINTF(buf, count, "\t%s:0x%x\n",
			GET_FIELD(record.last_sleep_transition_duration),
			record.last_sleep_transition_duration);

		record.last_wake_transition_duration =
				readl_relaxed(prvdata->reg_base +
				(master_cnt * pdata->master_offset +
				offsetof(struct msm_rpm_master_stats,
				last_wake_transition_duration)));

		SNPRINTF(buf, count, "\t%s:0x%x\n",
			GET_FIELD(record.last_wake_transition_duration),
			record.last_wake_transition_duration);

		record.wakeup_reason = readl_relaxed(prvdata->reg_base +
					(master_cnt * pdata->master_offset +
					offsetof(struct msm_rpm_master_stats,
					wakeup_reason)));

		SNPRINTF(buf, count, "\t%s:0x%x\n",
			GET_FIELD(record.wakeup_reason),
			record.wakeup_reason);

		record.numshutdowns = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			 offsetof(struct msm_rpm_master_stats, numshutdowns)));

		SNPRINTF(buf, count, "\t%s:0x%x\n",
			GET_FIELD(record.numshutdowns),
			record.numshutdowns);

		record.active_cores = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset) +
			offsetof(struct msm_rpm_master_stats, active_cores));

		SNPRINTF(buf, count, "\t%s:0x%x\n",
			GET_FIELD(record.active_cores),
			record.active_cores);
	} else {
		SNPRINTF(buf, count, "%s\n",
				GET_MASTER_NAME(master_cnt, prvdata));

		record.numshutdowns = readl_relaxed(prvdata->reg_base +
				(master_cnt * pdata->master_offset) + 0x0);

		SNPRINTF(buf, count, "\t%s:0x%0x\n",
			GET_FIELD(record.numshutdowns),
			record.numshutdowns);

		record.active_cores = readl_relaxed(prvdata->reg_base +
				(master_cnt * pdata->master_offset) + 0x4);

		SNPRINTF(buf, count, "\t%s:0x%0x\n",
			GET_FIELD(record.active_cores),
			record.active_cores);
	}

	j = find_first_bit((unsigned long *)&record.active_cores,
							BITS_PER_LONG);
	while (j < BITS_PER_LONG) {
		SNPRINTF(buf, count, "\t\tcore%d\n", j);
		j = find_next_bit((unsigned long *)&record.active_cores,
				BITS_PER_LONG, j + 1);
	}

	master_cnt++;
	mutex_unlock(&msm_rpm_master_stats_mutex);
	return RPM_MASTERS_BUF_LEN - count;
}

static int msm_rpm_master_stats_file_read(struct file *file, char __user *bufu,
				  size_t count, loff_t *ppos)
{
	struct msm_rpm_master_stats_private_data *prvdata;
	struct msm_rpm_master_stats_platform_data *pdata;

	prvdata = file->private_data;
	if (!prvdata)
		return -EINVAL;

	pdata = prvdata->platform_data;
	if (!pdata)
		return -EINVAL;

	if (!bufu || count == 0)
		return -EINVAL;

	if ((*ppos <= pdata->phys_size)) {
		prvdata->len = msm_rpm_master_copy_stats(prvdata);
		*ppos = 0;
	}

	return simple_read_from_buffer(bufu, count, ppos,
			prvdata->buf, prvdata->len);
}

static int msm_rpm_master_stats_file_open(struct inode *inode,
		struct file *file)
{
	struct msm_rpm_master_stats_private_data *prvdata;
	struct msm_rpm_master_stats_platform_data *pdata;

	pdata = inode->i_private;

	file->private_data =
		kzalloc(sizeof(struct msm_rpm_master_stats_private_data),
			GFP_KERNEL);

	if (!file->private_data)
		return -ENOMEM;
	prvdata = file->private_data;

	prvdata->reg_base = ioremap(pdata->phys_addr_base,
						pdata->phys_size);
	if (!prvdata->reg_base) {
		kfree(file->private_data);
		prvdata = NULL;
		pr_err("%s: ERROR could not ioremap start=%p, len=%u\n",
			__func__, (void *)pdata->phys_addr_base,
			pdata->phys_size);
		return -EBUSY;
	}

	prvdata->len = 0;
	prvdata->num_masters = pdata->num_masters;
	prvdata->master_names = pdata->masters;
	prvdata->platform_data = pdata;
	return 0;
}

static const struct file_operations msm_rpm_master_stats_fops = {
	.owner	  = THIS_MODULE,
	.open	  = msm_rpm_master_stats_file_open,
	.read	  = msm_rpm_master_stats_file_read,
	.release  = msm_rpm_master_stats_file_close,
	.llseek   = no_llseek,
};

#ifdef CONFIG_SEC_PM_DEBUG
static int is_init_done  = 0;
static int enable_master_info = 0;
static struct msm_rpm_master_stats_private_data *prvdata_global;

static int sec_rpm_master_stats_print(
		struct msm_rpm_master_stats_private_data *prvdata, int master_cnt)
{
	int  j = 0;
	struct msm_rpm_master_stats record;
	struct msm_rpm_master_stats_platform_data *pdata;
	static DEFINE_MUTEX(msm_rpm_master_stats_mutex);

	mutex_lock(&msm_rpm_master_stats_mutex);

	if (master_cnt > prvdata->num_masters - 1) {
		master_cnt = 0;
		mutex_unlock(&msm_rpm_master_stats_mutex);
		return 0;
	}

	pdata = prvdata->platform_data;
	if (prvdata->platform_data->version == 2) {
		record.shutdown_req = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, shutdown_req)));

		record.wakeup_ind = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, wakeup_ind)));

		record.bringup_req = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, bringup_req)));

		record.bringup_ack = readll_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, bringup_ack)));

		record.last_sleep_transition_duration = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, last_sleep_transition_duration)));

		record.last_wake_transition_duration = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, last_wake_transition_duration)));

		record.wakeup_reason = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			offsetof(struct msm_rpm_master_stats, wakeup_reason)));

		record.numshutdowns = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset +
			 offsetof(struct msm_rpm_master_stats, numshutdowns)));

		record.active_cores = readl_relaxed(prvdata->reg_base +
			(master_cnt * pdata->master_offset) +
			offsetof(struct msm_rpm_master_stats, active_cores));

		pr_info("%s:- Active_Cores(0x%x) "
			"Numshutdowns(0x%x) "
			"Last_Sleep_duration(0x%x) "
			"Last_Wake_duration(0x%x) "
			"Wakeup_reason(0x%x) "
			"Shutdown_req(0x%llX) "
			"Wakeup_ind(0x%llX) "
			"Bringup_req(0x%llX) "
			"Bringup_ack(0x%llX) ",
			GET_MASTER_NAME(master_cnt, prvdata),
			record.active_cores,
			record.numshutdowns,
			record.last_sleep_transition_duration,
			record.last_wake_transition_duration,
			record.wakeup_reason,
			record.shutdown_req,
			record.wakeup_ind,
			record.bringup_req,
			record.bringup_ack);
	} else {
		record.numshutdowns = readl_relaxed(prvdata->reg_base +
				(master_cnt * pdata->master_offset) + 0x0);

		record.active_cores = readl_relaxed(prvdata->reg_base +
				(master_cnt * pdata->master_offset) + 0x4);

		pr_info("%s Active_Cores(0x%0x) Numshutdowns(0x%0x) ",
			GET_MASTER_NAME(master_cnt, prvdata),
			record.active_cores, record.numshutdowns);
	}

	j = find_first_bit((unsigned long *)&record.active_cores, BITS_PER_LONG);
	while (j < BITS_PER_LONG) {
		printk(" Core%d Active", j);
		j = find_next_bit((unsigned long *)&record.active_cores, BITS_PER_LONG, j + 1);
	}

	mutex_unlock(&msm_rpm_master_stats_mutex);

	return 0;
}

int sec_print_masters_stats(void)
{
	if(enable_master_info == 1) {
		int i = 0;
		pr_info("---RPM_MASTERS_STATS---");
		for (i = 0; i < prvdata_global->num_masters; i++) {
			sec_rpm_master_stats_print(prvdata_global, i);
		}
		pr_info("");
	}
	return 0;
}

static ssize_t masters_info_enable_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	pr_info("SEC_MASTERSTATS : show called\n");
	return snprintf(buf, 4, "%d\n", enable_master_info);
}

static ssize_t masters_info_enable_store(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t size)
{
	pr_info("SEC_MASTERSTATS : store called\n");
	sscanf(buf, "%1d", &enable_master_info);

	if (is_init_done == 0 && enable_master_info == 1) {
		struct msm_rpm_master_stats_platform_data *pdata = dev_get_drvdata(dev);

		prvdata_global = kzalloc(sizeof(struct msm_rpm_master_stats_private_data), GFP_KERNEL);
		if (!prvdata_global) {
			pr_err("SEC_MASTERSTATS : ERROR could not allocate memory for private data\n");
			return -ENOMEM;
		}

		prvdata_global->reg_base = ioremap(pdata->phys_addr_base, pdata->phys_size);
		if (!prvdata_global->reg_base) {
			kfree(prvdata_global);
			prvdata_global = NULL;
			pr_err("%s: SEC_MASTERSTATS : ERROR could not ioremap start=%p, len=%u\n",
				__func__, (void *)pdata->phys_addr_base, pdata->phys_size);
			return -EBUSY;
		}

		prvdata_global->len = 0;
		prvdata_global->num_masters = pdata->num_masters;
		prvdata_global->master_names = pdata->masters;
		prvdata_global->platform_data = pdata;
		is_init_done = 1;
	}
	else if (is_init_done == 1 && enable_master_info == 0) {
		if (prvdata_global != NULL) {
			if (prvdata_global->reg_base)
				iounmap(prvdata_global->reg_base);
			kfree(prvdata_global);
		}
		is_init_done = 0;
	}

	return size;
}

static DEVICE_ATTR(masters_info_enable, 0664 , masters_info_enable_show, masters_info_enable_store);
#endif

static struct msm_rpm_master_stats_platform_data
			*msm_rpm_master_populate_pdata(struct device *dev)
{
	struct msm_rpm_master_stats_platform_data *pdata;
	struct device_node *node = dev->of_node;
	int rc = 0, i;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate memory for platform data\n");
		goto err;
	}

	rc = of_property_read_u32(node, "qcom,master-stats-version",
							&pdata->version);
	if (rc) {
		dev_err(dev, "master-stats-version missing rc=%d\n", rc);
		goto err;
	}

	rc = of_property_read_u32(node, "qcom,master-offset",
							&pdata->master_offset);
	if (rc) {
		dev_err(dev, "master-offset missing rc=%d\n", rc);
		goto err;
	}

	pdata->num_masters = of_property_count_strings(node, "qcom,masters");
	if (pdata->num_masters < 0) {
		dev_err(dev, "Failed to get number of masters =%d\n",
						pdata->num_masters);
		goto err;
	}

	pdata->masters = devm_kzalloc(dev, sizeof(char *) * pdata->num_masters,
								GFP_KERNEL);
	if (!pdata->masters) {
		dev_err(dev, "%s:Failed to allocated memory\n", __func__);
		goto err;
	}

	/*
	 * Read master names from DT
	 */
	for (i = 0; i < pdata->num_masters; i++) {
		const char *master_name;
		of_property_read_string_index(node, "qcom,masters",
							i, &master_name);
		pdata->masters[i] = devm_kzalloc(dev, sizeof(char) *
				strlen(master_name) + 1, GFP_KERNEL);
		if (!pdata->masters[i]) {
			dev_err(dev, "%s:Failed to get memory\n", __func__);
			goto err;
		}
		strlcpy(pdata->masters[i], master_name,
					strlen(master_name) + 1);
	}
	return pdata;
err:
	return NULL;
}

static  int __devinit msm_rpm_master_stats_probe(struct platform_device *pdev)
{
	struct dentry *dent;
	struct msm_rpm_master_stats_platform_data *pdata;
	struct resource *res = NULL;
#ifdef CONFIG_SEC_PM_DEBUG
	int ret = 0;
	struct class *sec_rpmstats;
	struct device *sec_masters_stats;
#endif

	if (!pdev)
		return -EINVAL;

	if (pdev->dev.of_node)
		pdata = msm_rpm_master_populate_pdata(&pdev->dev);
	else
		pdata = pdev->dev.platform_data;

	if (!pdata) {
		dev_err(&pdev->dev, "%s: Unable to get pdata\n", __func__);
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	if (!res) {
		dev_err(&pdev->dev,
			"%s: Failed to get IO resource from platform device",
			__func__);
		return -ENXIO;
	}

	pdata->phys_addr_base = res->start;
	pdata->phys_size = resource_size(res);

	dent = debugfs_create_file("rpm_master_stats", S_IRUGO, NULL,
					pdata, &msm_rpm_master_stats_fops);

	if (!dent) {
		dev_err(&pdev->dev, "%s: ERROR debugfs_create_file failed\n",
								__func__);
		return -ENOMEM;
	}

#ifdef CONFIG_SEC_PM_DEBUG
	sec_rpmstats = class_create(THIS_MODULE, "sec_rpmstats");
	if (IS_ERR(sec_rpmstats))
		pr_err("SEC_MASTERSTATS : Failed to create class(sec_rpm_stats)!\n");

	sec_masters_stats = device_create(sec_rpmstats, NULL, 0, NULL, "sec_masters_stats");
	if (IS_ERR(sec_masters_stats))
		pr_err("SEC_MASTERSTATS : Failed to create device(sec_masters_stats)!\n");

	ret = device_create_file(sec_masters_stats, &dev_attr_masters_info_enable);
	if (ret)
		pr_err("SEC_MASTERSTATS : Failed to create device file in sysfs entries(%s)!\n",
			dev_attr_masters_info_enable.attr.name);

	dev_set_drvdata(sec_masters_stats, pdata);

	pr_info("SEC_MASTERSTATS : Init done\n");
#endif
	platform_set_drvdata(pdev, dent);
	return 0;
}

static int __devexit msm_rpm_master_stats_remove(struct platform_device *pdev)
{
	struct dentry *dent;

	dent = platform_get_drvdata(pdev);
	debugfs_remove(dent);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static struct of_device_id rpm_master_table[] = {
	{.compatible = "qcom,rpm-master-stats"},
	{},
};

static struct platform_driver msm_rpm_master_stats_driver = {
	.probe	= msm_rpm_master_stats_probe,
	.remove = __devexit_p(msm_rpm_master_stats_remove),
	.driver = {
		.name = "msm_rpm_master_stats",
		.owner = THIS_MODULE,
		.of_match_table = rpm_master_table,
	},
};

static int __init msm_rpm_master_stats_init(void)
{
	return platform_driver_register(&msm_rpm_master_stats_driver);
}

static void __exit msm_rpm_master_stats_exit(void)
{
	platform_driver_unregister(&msm_rpm_master_stats_driver);
}

module_init(msm_rpm_master_stats_init);
module_exit(msm_rpm_master_stats_exit);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("MSM RPM Master Statistics driver");
MODULE_VERSION("1.0");
MODULE_ALIAS("platform:msm_master_stat_log");
