/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Communicator client code.
 *
 * The Initial Developer of the Original Code is
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Kohei TAKETA <k-tak@void.in> (Java port)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package org.mozilla.universalchardet.prober.distributionanalysis;

public class Big5DistributionAnalysis extends CharDistributionAnalysis {
    // //////////////////////////////////////////////////////////////
    // constants
    // //////////////////////////////////////////////////////////////
    public static final float BIG5_TYPICAL_DISTRIBUTION_RATIO = 0.75f;
    public static final int HIGHBYTE_BEGIN = 0xA4;
    public static final int HIGHBYTE_END = 0xFE;
    public static final int LOWBYTE_BEGIN_1 = 0x40;
    public static final int LOWBYTE_END_1 = 0x7E;
    public static final int LOWBYTE_BEGIN_2 = 0xA1;
    public static final int LOWBYTE_END_2 = 0xFE;

    // //////////////////////////////////////////////////////////////
    // methods
    // //////////////////////////////////////////////////////////////
    public Big5DistributionAnalysis() {
        super();
        this.charToFreqOrder = big5CharToFreqOrder;
        this.typicalDistributionRatio = BIG5_TYPICAL_DISTRIBUTION_RATIO;
    }

    protected int getOrder(final byte[] buf, int offset) {
        int highbyte = buf[offset] & 0xFF;
        if (highbyte >= HIGHBYTE_BEGIN) {
            int lowbyte = buf[offset + 1] & 0xFF;
            if (lowbyte >= LOWBYTE_BEGIN_2) {
                return (157 * (highbyte - HIGHBYTE_BEGIN) + lowbyte - LOWBYTE_BEGIN_2 + 63);
            } else {
                return (157 * (highbyte - HIGHBYTE_BEGIN) + lowbyte - LOWBYTE_BEGIN_1);
            }
        }

        return -1;
    }

    // //////////////////////////////////////////////////////////////
    // constants continued
    // //////////////////////////////////////////////////////////////
    private static final int[] big5CharToFreqOrder = new int[] {
            1,
            1801,
            1506,
            255,
            1431,
            198,
            9,
            82,
            6,
            5008,
            177,
            202,
            3681,
            1256,
            2821,
            110, // 16
            3814,
            33,
            3274,
            261,
            76,
            44,
            2114,
            16,
            2946,
            2187,
            1176,
            659,
            3971,
            26,
            3451,
            2653, // 32
            1198,
            3972,
            3350,
            4202,
            410,
            2215,
            302,
            590,
            361,
            1964,
            8,
            204,
            58,
            4510,
            5009,
            1932, // 48
            63,
            5010,
            5011,
            317,
            1614,
            75,
            222,
            159,
            4203,
            2417,
            1480,
            5012,
            3555,
            3091,
            224,
            2822, // 64
            3682,
            3,
            10,
            3973,
            1471,
            29,
            2787,
            1135,
            2866,
            1940,
            873,
            130,
            3275,
            1123,
            312,
            5013, // 80
            4511,
            2052,
            507,
            252,
            682,
            5014,
            142,
            1915,
            124,
            206,
            2947,
            34,
            3556,
            3204,
            64,
            604, // 96
            5015,
            2501,
            1977,
            1978,
            155,
            1991,
            645,
            641,
            1606,
            5016,
            3452,
            337,
            72,
            406,
            5017,
            80, // 112
            630,
            238,
            3205,
            1509,
            263,
            939,
            1092,
            2654,
            756,
            1440,
            1094,
            3453,
            449,
            69,
            2987,
            591, // 128
            179,
            2096,
            471,
            115,
            2035,
            1844,
            60,
            50,
            2988,
            134,
            806,
            1869,
            734,
            2036,
            3454,
            180, // 144
            995,
            1607,
            156,
            537,
            2907,
            688,
            5018,
            319,
            1305,
            779,
            2145,
            514,
            2379,
            298,
            4512,
            359, // 160
            2502,
            90,
            2716,
            1338,
            663,
            11,
            906,
            1099,
            2553,
            20,
            2441,
            182,
            532,
            1716,
            5019,
            732, // 176
            1376,
            4204,
            1311,
            1420,
            3206,
            25,
            2317,
            1056,
            113,
            399,
            382,
            1950,
            242,
            3455,
            2474,
            529, // 192
            3276,
            475,
            1447,
            3683,
            5020,
            117,
            21,
            656,
            810,
            1297,
            2300,
            2334,
            3557,
            5021,
            126,
            4205, // 208
            706,
            456,
            150,
            613,
            4513,
            71,
            1118,
            2037,
            4206,
            145,
            3092,
            85,
            835,
            486,
            2115,
            1246, // 224
            1426,
            428,
            727,
            1285,
            1015,
            800,
            106,
            623,
            303,
            1281,
            5022,
            2128,
            2359,
            347,
            3815,
            221, // 240
            3558,
            3135,
            5023,
            1956,
            1153,
            4207,
            83,
            296,
            1199,
            3093,
            192,
            624,
            93,
            5024,
            822,
            1898, // 256
            2823,
            3136,
            795,
            2065,
            991,
            1554,
            1542,
            1592,
            27,
            43,
            2867,
            859,
            139,
            1456,
            860,
            4514, // 272
            437,
            712,
            3974,
            164,
            2397,
            3137,
            695,
            211,
            3037,
            2097,
            195,
            3975,
            1608,
            3559,
            3560,
            3684, // 288
            3976,
            234,
            811,
            2989,
            2098,
            3977,
            2233,
            1441,
            3561,
            1615,
            2380,
            668,
            2077,
            1638,
            305,
            228, // 304
            1664,
            4515,
            467,
            415,
            5025,
            262,
            2099,
            1593,
            239,
            108,
            300,
            200,
            1033,
            512,
            1247,
            2078, // 320
            5026,
            5027,
            2176,
            3207,
            3685,
            2682,
            593,
            845,
            1062,
            3277,
            88,
            1723,
            2038,
            3978,
            1951,
            212, // 336
            266,
            152,
            149,
            468,
            1899,
            4208,
            4516,
            77,
            187,
            5028,
            3038,
            37,
            5,
            2990,
            5029,
            3979, // 352
            5030,
            5031,
            39,
            2524,
            4517,
            2908,
            3208,
            2079,
            55,
            148,
            74,
            4518,
            545,
            483,
            1474,
            1029, // 368
            1665,
            217,
            1870,
            1531,
            3138,
            1104,
            2655,
            4209,
            24,
            172,
            3562,
            900,
            3980,
            3563,
            3564,
            4519, // 384
            32,
            1408,
            2824,
            1312,
            329,
            487,
            2360,
            2251,
            2717,
            784,
            2683,
            4,
            3039,
            3351,
            1427,
            1789, // 400
            188,
            109,
            499,
            5032,
            3686,
            1717,
            1790,
            888,
            1217,
            3040,
            4520,
            5033,
            3565,
            5034,
            3352,
            1520, // 416
            3687,
            3981,
            196,
            1034,
            775,
            5035,
            5036,
            929,
            1816,
            249,
            439,
            38,
            5037,
            1063,
            5038,
            794, // 432
            3982,
            1435,
            2301,
            46,
            178,
            3278,
            2066,
            5039,
            2381,
            5040,
            214,
            1709,
            4521,
            804,
            35,
            707, // 448
            324,
            3688,
            1601,
            2554,
            140,
            459,
            4210,
            5041,
            5042,
            1365,
            839,
            272,
            978,
            2262,
            2580,
            3456, // 464
            2129,
            1363,
            3689,
            1423,
            697,
            100,
            3094,
            48,
            70,
            1231,
            495,
            3139,
            2196,
            5043,
            1294,
            5044, // 480
            2080,
            462,
            586,
            1042,
            3279,
            853,
            256,
            988,
            185,
            2382,
            3457,
            1698,
            434,
            1084,
            5045,
            3458, // 496
            314,
            2625,
            2788,
            4522,
            2335,
            2336,
            569,
            2285,
            637,
            1817,
            2525,
            757,
            1162,
            1879,
            1616,
            3459, // 512
            287,
            1577,
            2116,
            768,
            4523,
            1671,
            2868,
            3566,
            2526,
            1321,
            3816,
            909,
            2418,
            5046,
            4211,
            933, // 528
            3817,
            4212,
            2053,
            2361,
            1222,
            4524,
            765,
            2419,
            1322,
            786,
            4525,
            5047,
            1920,
            1462,
            1677,
            2909, // 544
            1699,
            5048,
            4526,
            1424,
            2442,
            3140,
            3690,
            2600,
            3353,
            1775,
            1941,
            3460,
            3983,
            4213,
            309,
            1369, // 560
            1130,
            2825,
            364,
            2234,
            1653,
            1299,
            3984,
            3567,
            3985,
            3986,
            2656,
            525,
            1085,
            3041,
            902,
            2001, // 576
            1475,
            964,
            4527,
            421,
            1845,
            1415,
            1057,
            2286,
            940,
            1364,
            3141,
            376,
            4528,
            4529,
            1381,
            7, // 592
            2527,
            983,
            2383,
            336,
            1710,
            2684,
            1846,
            321,
            3461,
            559,
            1131,
            3042,
            2752,
            1809,
            1132,
            1313, // 608
            265,
            1481,
            1858,
            5049,
            352,
            1203,
            2826,
            3280,
            167,
            1089,
            420,
            2827,
            776,
            792,
            1724,
            3568, // 624
            4214,
            2443,
            3281,
            5050,
            4215,
            5051,
            446,
            229,
            333,
            2753,
            901,
            3818,
            1200,
            1557,
            4530,
            2657, // 640
            1921,
            395,
            2754,
            2685,
            3819,
            4216,
            1836,
            125,
            916,
            3209,
            2626,
            4531,
            5052,
            5053,
            3820,
            5054, // 656
            5055,
            5056,
            4532,
            3142,
            3691,
            1133,
            2555,
            1757,
            3462,
            1510,
            2318,
            1409,
            3569,
            5057,
            2146,
            438, // 672
            2601,
            2910,
            2384,
            3354,
            1068,
            958,
            3043,
            461,
            311,
            2869,
            2686,
            4217,
            1916,
            3210,
            4218,
            1979, // 688
            383,
            750,
            2755,
            2627,
            4219,
            274,
            539,
            385,
            1278,
            1442,
            5058,
            1154,
            1965,
            384,
            561,
            210, // 704
            98,
            1295,
            2556,
            3570,
            5059,
            1711,
            2420,
            1482,
            3463,
            3987,
            2911,
            1257,
            129,
            5060,
            3821,
            642, // 720
            523,
            2789,
            2790,
            2658,
            5061,
            141,
            2235,
            1333,
            68,
            176,
            441,
            876,
            907,
            4220,
            603,
            2602, // 736
            710,
            171,
            3464,
            404,
            549,
            18,
            3143,
            2398,
            1410,
            3692,
            1666,
            5062,
            3571,
            4533,
            2912,
            4534, // 752
            5063,
            2991,
            368,
            5064,
            146,
            366,
            99,
            871,
            3693,
            1543,
            748,
            807,
            1586,
            1185,
            22,
            2263, // 768
            379,
            3822,
            3211,
            5065,
            3212,
            505,
            1942,
            2628,
            1992,
            1382,
            2319,
            5066,
            380,
            2362,
            218,
            702, // 784
            1818,
            1248,
            3465,
            3044,
            3572,
            3355,
            3282,
            5067,
            2992,
            3694,
            930,
            3283,
            3823,
            5068,
            59,
            5069, // 800
            585,
            601,
            4221,
            497,
            3466,
            1112,
            1314,
            4535,
            1802,
            5070,
            1223,
            1472,
            2177,
            5071,
            749,
            1837, // 816
            690,
            1900,
            3824,
            1773,
            3988,
            1476,
            429,
            1043,
            1791,
            2236,
            2117,
            917,
            4222,
            447,
            1086,
            1629, // 832
            5072,
            556,
            5073,
            5074,
            2021,
            1654,
            844,
            1090,
            105,
            550,
            966,
            1758,
            2828,
            1008,
            1783,
            686, // 848
            1095,
            5075,
            2287,
            793,
            1602,
            5076,
            3573,
            2603,
            4536,
            4223,
            2948,
            2302,
            4537,
            3825,
            980,
            2503, // 864
            544,
            353,
            527,
            4538,
            908,
            2687,
            2913,
            5077,
            381,
            2629,
            1943,
            1348,
            5078,
            1341,
            1252,
            560, // 880
            3095,
            5079,
            3467,
            2870,
            5080,
            2054,
            973,
            886,
            2081,
            143,
            4539,
            5081,
            5082,
            157,
            3989,
            496, // 896
            4224,
            57,
            840,
            540,
            2039,
            4540,
            4541,
            3468,
            2118,
            1445,
            970,
            2264,
            1748,
            1966,
            2082,
            4225, // 912
            3144,
            1234,
            1776,
            3284,
            2829,
            3695,
            773,
            1206,
            2130,
            1066,
            2040,
            1326,
            3990,
            1738,
            1725,
            4226, // 928
            279,
            3145,
            51,
            1544,
            2604,
            423,
            1578,
            2131,
            2067,
            173,
            4542,
            1880,
            5083,
            5084,
            1583,
            264, // 944
            610,
            3696,
            4543,
            2444,
            280,
            154,
            5085,
            5086,
            5087,
            1739,
            338,
            1282,
            3096,
            693,
            2871,
            1411, // 960
            1074,
            3826,
            2445,
            5088,
            4544,
            5089,
            5090,
            1240,
            952,
            2399,
            5091,
            2914,
            1538,
            2688,
            685,
            1483, // 976
            4227,
            2475,
            1436,
            953,
            4228,
            2055,
            4545,
            671,
            2400,
            79,
            4229,
            2446,
            3285,
            608,
            567,
            2689, // 992
            3469,
            4230,
            4231,
            1691,
            393,
            1261,
            1792,
            2401,
            5092,
            4546,
            5093,
            5094,
            5095,
            5096,
            1383,
            1672, // 1008
            3827,
            3213,
            1464,
            522,
            1119,
            661,
            1150,
            216,
            675,
            4547,
            3991,
            1432,
            3574,
            609,
            4548,
            2690, // 1024
            2402,
            5097,
            5098,
            5099,
            4232,
            3045,
            0,
            5100,
            2476,
            315,
            231,
            2447,
            301,
            3356,
            4549,
            2385, // 1040
            5101,
            233,
            4233,
            3697,
            1819,
            4550,
            4551,
            5102,
            96,
            1777,
            1315,
            2083,
            5103,
            257,
            5104,
            1810, // 1056
            3698,
            2718,
            1139,
            1820,
            4234,
            2022,
            1124,
            2164,
            2791,
            1778,
            2659,
            5105,
            3097,
            363,
            1655,
            3214, // 1072
            5106,
            2993,
            5107,
            5108,
            5109,
            3992,
            1567,
            3993,
            718,
            103,
            3215,
            849,
            1443,
            341,
            3357,
            2949, // 1088
            1484,
            5110,
            1712,
            127,
            67,
            339,
            4235,
            2403,
            679,
            1412,
            821,
            5111,
            5112,
            834,
            738,
            351, // 1104
            2994,
            2147,
            846,
            235,
            1497,
            1881,
            418,
            1993,
            3828,
            2719,
            186,
            1100,
            2148,
            2756,
            3575,
            1545, // 1120
            1355,
            2950,
            2872,
            1377,
            583,
            3994,
            4236,
            2581,
            2995,
            5113,
            1298,
            3699,
            1078,
            2557,
            3700,
            2363, // 1136
            78,
            3829,
            3830,
            267,
            1289,
            2100,
            2002,
            1594,
            4237,
            348,
            369,
            1274,
            2197,
            2178,
            1838,
            4552, // 1152
            1821,
            2830,
            3701,
            2757,
            2288,
            2003,
            4553,
            2951,
            2758,
            144,
            3358,
            882,
            4554,
            3995,
            2759,
            3470, // 1168
            4555,
            2915,
            5114,
            4238,
            1726,
            320,
            5115,
            3996,
            3046,
            788,
            2996,
            5116,
            2831,
            1774,
            1327,
            2873, // 1184
            3997,
            2832,
            5117,
            1306,
            4556,
            2004,
            1700,
            3831,
            3576,
            2364,
            2660,
            787,
            2023,
            506,
            824,
            3702, // 1200
            534,
            323,
            4557,
            1044,
            3359,
            2024,
            1901,
            946,
            3471,
            5118,
            1779,
            1500,
            1678,
            5119,
            1882,
            4558, // 1216
            165,
            243,
            4559,
            3703,
            2528,
            123,
            683,
            4239,
            764,
            4560,
            36,
            3998,
            1793,
            589,
            2916,
            816, // 1232
            626,
            1667,
            3047,
            2237,
            1639,
            1555,
            1622,
            3832,
            3999,
            5120,
            4000,
            2874,
            1370,
            1228,
            1933,
            891, // 1248
            2084,
            2917,
            304,
            4240,
            5121,
            292,
            2997,
            2720,
            3577,
            691,
            2101,
            4241,
            1115,
            4561,
            118,
            662, // 1264
            5122,
            611,
            1156,
            854,
            2386,
            1316,
            2875,
            2,
            386,
            515,
            2918,
            5123,
            5124,
            3286,
            868,
            2238, // 1280
            1486,
            855,
            2661,
            785,
            2216,
            3048,
            5125,
            1040,
            3216,
            3578,
            5126,
            3146,
            448,
            5127,
            1525,
            5128, // 1296
            2165,
            4562,
            5129,
            3833,
            5130,
            4242,
            2833,
            3579,
            3147,
            503,
            818,
            4001,
            3148,
            1568,
            814,
            676, // 1312
            1444,
            306,
            1749,
            5131,
            3834,
            1416,
            1030,
            197,
            1428,
            805,
            2834,
            1501,
            4563,
            5132,
            5133,
            5134, // 1328
            1994,
            5135,
            4564,
            5136,
            5137,
            2198,
            13,
            2792,
            3704,
            2998,
            3149,
            1229,
            1917,
            5138,
            3835,
            2132, // 1344
            5139,
            4243,
            4565,
            2404,
            3580,
            5140,
            2217,
            1511,
            1727,
            1120,
            5141,
            5142,
            646,
            3836,
            2448,
            307, // 1360
            5143,
            5144,
            1595,
            3217,
            5145,
            5146,
            5147,
            3705,
            1113,
            1356,
            4002,
            1465,
            2529,
            2530,
            5148,
            519, // 1376
            5149,
            128,
            2133,
            92,
            2289,
            1980,
            5150,
            4003,
            1512,
            342,
            3150,
            2199,
            5151,
            2793,
            2218,
            1981, // 1392
            3360,
            4244,
            290,
            1656,
            1317,
            789,
            827,
            2365,
            5152,
            3837,
            4566,
            562,
            581,
            4004,
            5153,
            401, // 1408
            4567,
            2252,
            94,
            4568,
            5154,
            1399,
            2794,
            5155,
            1463,
            2025,
            4569,
            3218,
            1944,
            5156,
            828,
            1105, // 1424
            4245,
            1262,
            1394,
            5157,
            4246,
            605,
            4570,
            5158,
            1784,
            2876,
            5159,
            2835,
            819,
            2102,
            578,
            2200, // 1440
            2952,
            5160,
            1502,
            436,
            3287,
            4247,
            3288,
            2836,
            4005,
            2919,
            3472,
            3473,
            5161,
            2721,
            2320,
            5162, // 1456
            5163,
            2337,
            2068,
            23,
            4571,
            193,
            826,
            3838,
            2103,
            699,
            1630,
            4248,
            3098,
            390,
            1794,
            1064, // 1472
            3581,
            5164,
            1579,
            3099,
            3100,
            1400,
            5165,
            4249,
            1839,
            1640,
            2877,
            5166,
            4572,
            4573,
            137,
            4250, // 1488
            598,
            3101,
            1967,
            780,
            104,
            974,
            2953,
            5167,
            278,
            899,
            253,
            402,
            572,
            504,
            493,
            1339, // 1504
            5168,
            4006,
            1275,
            4574,
            2582,
            2558,
            5169,
            3706,
            3049,
            3102,
            2253,
            565,
            1334,
            2722,
            863,
            41, // 1520
            5170,
            5171,
            4575,
            5172,
            1657,
            2338,
            19,
            463,
            2760,
            4251,
            606,
            5173,
            2999,
            3289,
            1087,
            2085, // 1536
            1323,
            2662,
            3000,
            5174,
            1631,
            1623,
            1750,
            4252,
            2691,
            5175,
            2878,
            791,
            2723,
            2663,
            2339,
            232, // 1552
            2421,
            5176,
            3001,
            1498,
            5177,
            2664,
            2630,
            755,
            1366,
            3707,
            3290,
            3151,
            2026,
            1609,
            119,
            1918, // 1568
            3474,
            862,
            1026,
            4253,
            5178,
            4007,
            3839,
            4576,
            4008,
            4577,
            2265,
            1952,
            2477,
            5179,
            1125,
            817, // 1584
            4254,
            4255,
            4009,
            1513,
            1766,
            2041,
            1487,
            4256,
            3050,
            3291,
            2837,
            3840,
            3152,
            5180,
            5181,
            1507, // 1600
            5182,
            2692,
            733,
            40,
            1632,
            1106,
            2879,
            345,
            4257,
            841,
            2531,
            230,
            4578,
            3002,
            1847,
            3292, // 1616
            3475,
            5183,
            1263,
            986,
            3476,
            5184,
            735,
            879,
            254,
            1137,
            857,
            622,
            1300,
            1180,
            1388,
            1562, // 1632
            4010,
            4011,
            2954,
            967,
            2761,
            2665,
            1349,
            592,
            2134,
            1692,
            3361,
            3003,
            1995,
            4258,
            1679,
            4012, // 1648
            1902,
            2188,
            5185,
            739,
            3708,
            2724,
            1296,
            1290,
            5186,
            4259,
            2201,
            2202,
            1922,
            1563,
            2605,
            2559, // 1664
            1871,
            2762,
            3004,
            5187,
            435,
            5188,
            343,
            1108,
            596,
            17,
            1751,
            4579,
            2239,
            3477,
            3709,
            5189, // 1680
            4580,
            294,
            3582,
            2955,
            1693,
            477,
            979,
            281,
            2042,
            3583,
            643,
            2043,
            3710,
            2631,
            2795,
            2266, // 1696
            1031,
            2340,
            2135,
            2303,
            3584,
            4581,
            367,
            1249,
            2560,
            5190,
            3585,
            5191,
            4582,
            1283,
            3362,
            2005, // 1712
            240,
            1762,
            3363,
            4583,
            4584,
            836,
            1069,
            3153,
            474,
            5192,
            2149,
            2532,
            268,
            3586,
            5193,
            3219, // 1728
            1521,
            1284,
            5194,
            1658,
            1546,
            4260,
            5195,
            3587,
            3588,
            5196,
            4261,
            3364,
            2693,
            1685,
            4262,
            961, // 1744
            1673,
            2632,
            190,
            2006,
            2203,
            3841,
            4585,
            4586,
            5197,
            570,
            2504,
            3711,
            1490,
            5198,
            4587,
            2633, // 1760
            3293,
            1957,
            4588,
            584,
            1514,
            396,
            1045,
            1945,
            5199,
            4589,
            1968,
            2449,
            5200,
            5201,
            4590,
            4013, // 1776
            619,
            5202,
            3154,
            3294,
            215,
            2007,
            2796,
            2561,
            3220,
            4591,
            3221,
            4592,
            763,
            4263,
            3842,
            4593, // 1792
            5203,
            5204,
            1958,
            1767,
            2956,
            3365,
            3712,
            1174,
            452,
            1477,
            4594,
            3366,
            3155,
            5205,
            2838,
            1253, // 1808
            2387,
            2189,
            1091,
            2290,
            4264,
            492,
            5206,
            638,
            1169,
            1825,
            2136,
            1752,
            4014,
            648,
            926,
            1021, // 1824
            1324,
            4595,
            520,
            4596,
            997,
            847,
            1007,
            892,
            4597,
            3843,
            2267,
            1872,
            3713,
            2405,
            1785,
            4598, // 1840
            1953,
            2957,
            3103,
            3222,
            1728,
            4265,
            2044,
            3714,
            4599,
            2008,
            1701,
            3156,
            1551,
            30,
            2268,
            4266, // 1856
            5207,
            2027,
            4600,
            3589,
            5208,
            501,
            5209,
            4267,
            594,
            3478,
            2166,
            1822,
            3590,
            3479,
            3591,
            3223, // 1872
            829,
            2839,
            4268,
            5210,
            1680,
            3157,
            1225,
            4269,
            5211,
            3295,
            4601,
            4270,
            3158,
            2341,
            5212,
            4602, // 1888
            4271,
            5213,
            4015,
            4016,
            5214,
            1848,
            2388,
            2606,
            3367,
            5215,
            4603,
            374,
            4017,
            652,
            4272,
            4273, // 1904
            375,
            1140,
            798,
            5216,
            5217,
            5218,
            2366,
            4604,
            2269,
            546,
            1659,
            138,
            3051,
            2450,
            4605,
            5219, // 1920
            2254,
            612,
            1849,
            910,
            796,
            3844,
            1740,
            1371,
            825,
            3845,
            3846,
            5220,
            2920,
            2562,
            5221,
            692, // 1936
            444,
            3052,
            2634,
            801,
            4606,
            4274,
            5222,
            1491,
            244,
            1053,
            3053,
            4275,
            4276,
            340,
            5223,
            4018, // 1952
            1041,
            3005,
            293,
            1168,
            87,
            1357,
            5224,
            1539,
            959,
            5225,
            2240,
            721,
            694,
            4277,
            3847,
            219, // 1968
            1478,
            644,
            1417,
            3368,
            2666,
            1413,
            1401,
            1335,
            1389,
            4019,
            5226,
            5227,
            3006,
            2367,
            3159,
            1826, // 1984
            730,
            1515,
            184,
            2840,
            66,
            4607,
            5228,
            1660,
            2958,
            246,
            3369,
            378,
            1457,
            226,
            3480,
            975, // 2000
            4020,
            2959,
            1264,
            3592,
            674,
            696,
            5229,
            163,
            5230,
            1141,
            2422,
            2167,
            713,
            3593,
            3370,
            4608, // 2016
            4021,
            5231,
            5232,
            1186,
            15,
            5233,
            1079,
            1070,
            5234,
            1522,
            3224,
            3594,
            276,
            1050,
            2725,
            758, // 2032
            1126,
            653,
            2960,
            3296,
            5235,
            2342,
            889,
            3595,
            4022,
            3104,
            3007,
            903,
            1250,
            4609,
            4023,
            3481, // 2048
            3596,
            1342,
            1681,
            1718,
            766,
            3297,
            286,
            89,
            2961,
            3715,
            5236,
            1713,
            5237,
            2607,
            3371,
            3008, // 2064
            5238,
            2962,
            2219,
            3225,
            2880,
            5239,
            4610,
            2505,
            2533,
            181,
            387,
            1075,
            4024,
            731,
            2190,
            3372, // 2080
            5240,
            3298,
            310,
            313,
            3482,
            2304,
            770,
            4278,
            54,
            3054,
            189,
            4611,
            3105,
            3848,
            4025,
            5241, // 2096
            1230,
            1617,
            1850,
            355,
            3597,
            4279,
            4612,
            3373,
            111,
            4280,
            3716,
            1350,
            3160,
            3483,
            3055,
            4281, // 2112
            2150,
            3299,
            3598,
            5242,
            2797,
            4026,
            4027,
            3009,
            722,
            2009,
            5243,
            1071,
            247,
            1207,
            2343,
            2478, // 2128
            1378,
            4613,
            2010,
            864,
            1437,
            1214,
            4614,
            373,
            3849,
            1142,
            2220,
            667,
            4615,
            442,
            2763,
            2563, // 2144
            3850,
            4028,
            1969,
            4282,
            3300,
            1840,
            837,
            170,
            1107,
            934,
            1336,
            1883,
            5244,
            5245,
            2119,
            4283, // 2160
            2841,
            743,
            1569,
            5246,
            4616,
            4284,
            582,
            2389,
            1418,
            3484,
            5247,
            1803,
            5248,
            357,
            1395,
            1729, // 2176
            3717,
            3301,
            2423,
            1564,
            2241,
            5249,
            3106,
            3851,
            1633,
            4617,
            1114,
            2086,
            4285,
            1532,
            5250,
            482, // 2192
            2451,
            4618,
            5251,
            5252,
            1492,
            833,
            1466,
            5253,
            2726,
            3599,
            1641,
            2842,
            5254,
            1526,
            1272,
            3718, // 2208
            4286,
            1686,
            1795,
            416,
            2564,
            1903,
            1954,
            1804,
            5255,
            3852,
            2798,
            3853,
            1159,
            2321,
            5256,
            2881, // 2224
            4619,
            1610,
            1584,
            3056,
            2424,
            2764,
            443,
            3302,
            1163,
            3161,
            5257,
            5258,
            4029,
            5259,
            4287,
            2506, // 2240
            3057,
            4620,
            4030,
            3162,
            2104,
            1647,
            3600,
            2011,
            1873,
            4288,
            5260,
            4289,
            431,
            3485,
            5261,
            250, // 2256
            97,
            81,
            4290,
            5262,
            1648,
            1851,
            1558,
            160,
            848,
            5263,
            866,
            740,
            1694,
            5264,
            2204,
            2843, // 2272
            3226,
            4291,
            4621,
            3719,
            1687,
            950,
            2479,
            426,
            469,
            3227,
            3720,
            3721,
            4031,
            5265,
            5266,
            1188, // 2288
            424,
            1996,
            861,
            3601,
            4292,
            3854,
            2205,
            2694,
            168,
            1235,
            3602,
            4293,
            5267,
            2087,
            1674,
            4622, // 2304
            3374,
            3303,
            220,
            2565,
            1009,
            5268,
            3855,
            670,
            3010,
            332,
            1208,
            717,
            5269,
            5270,
            3603,
            2452, // 2320
            4032,
            3375,
            5271,
            513,
            5272,
            1209,
            2882,
            3376,
            3163,
            4623,
            1080,
            5273,
            5274,
            5275,
            5276,
            2534, // 2336
            3722,
            3604,
            815,
            1587,
            4033,
            4034,
            5277,
            3605,
            3486,
            3856,
            1254,
            4624,
            1328,
            3058,
            1390,
            4035, // 2352
            1741,
            4036,
            3857,
            4037,
            5278,
            236,
            3858,
            2453,
            3304,
            5279,
            5280,
            3723,
            3859,
            1273,
            3860,
            4625, // 2368
            5281,
            308,
            5282,
            4626,
            245,
            4627,
            1852,
            2480,
            1307,
            2583,
            430,
            715,
            2137,
            2454,
            5283,
            270, // 2384
            199,
            2883,
            4038,
            5284,
            3606,
            2727,
            1753,
            761,
            1754,
            725,
            1661,
            1841,
            4628,
            3487,
            3724,
            5285, // 2400
            5286,
            587,
            14,
            3305,
            227,
            2608,
            326,
            480,
            2270,
            943,
            2765,
            3607,
            291,
            650,
            1884,
            5287, // 2416
            1702,
            1226,
            102,
            1547,
            62,
            3488,
            904,
            4629,
            3489,
            1164,
            4294,
            5288,
            5289,
            1224,
            1548,
            2766, // 2432
            391,
            498,
            1493,
            5290,
            1386,
            1419,
            5291,
            2056,
            1177,
            4630,
            813,
            880,
            1081,
            2368,
            566,
            1145, // 2448
            4631,
            2291,
            1001,
            1035,
            2566,
            2609,
            2242,
            394,
            1286,
            5292,
            5293,
            2069,
            5294,
            86,
            1494,
            1730, // 2464
            4039,
            491,
            1588,
            745,
            897,
            2963,
            843,
            3377,
            4040,
            2767,
            2884,
            3306,
            1768,
            998,
            2221,
            2070, // 2480
            397,
            1827,
            1195,
            1970,
            3725,
            3011,
            3378,
            284,
            5295,
            3861,
            2507,
            2138,
            2120,
            1904,
            5296,
            4041, // 2496
            2151,
            4042,
            4295,
            1036,
            3490,
            1905,
            114,
            2567,
            4296,
            209,
            1527,
            5297,
            5298,
            2964,
            2844,
            2635, // 2512
            2390,
            2728,
            3164,
            812,
            2568,
            5299,
            3307,
            5300,
            1559,
            737,
            1885,
            3726,
            1210,
            885,
            28,
            2695, // 2528
            3608,
            3862,
            5301,
            4297,
            1004,
            1780,
            4632,
            5302,
            346,
            1982,
            2222,
            2696,
            4633,
            3863,
            1742,
            797, // 2544
            1642,
            4043,
            1934,
            1072,
            1384,
            2152,
            896,
            4044,
            3308,
            3727,
            3228,
            2885,
            3609,
            5303,
            2569,
            1959, // 2560
            4634,
            2455,
            1786,
            5304,
            5305,
            5306,
            4045,
            4298,
            1005,
            1308,
            3728,
            4299,
            2729,
            4635,
            4636,
            1528, // 2576
            2610,
            161,
            1178,
            4300,
            1983,
            987,
            4637,
            1101,
            4301,
            631,
            4046,
            1157,
            3229,
            2425,
            1343,
            1241, // 2592
            1016,
            2243,
            2570,
            372,
            877,
            2344,
            2508,
            1160,
            555,
            1935,
            911,
            4047,
            5307,
            466,
            1170,
            169, // 2608
            1051,
            2921,
            2697,
            3729,
            2481,
            3012,
            1182,
            2012,
            2571,
            1251,
            2636,
            5308,
            992,
            2345,
            3491,
            1540, // 2624
            2730,
            1201,
            2071,
            2406,
            1997,
            2482,
            5309,
            4638,
            528,
            1923,
            2191,
            1503,
            1874,
            1570,
            2369,
            3379, // 2640
            3309,
            5310,
            557,
            1073,
            5311,
            1828,
            3492,
            2088,
            2271,
            3165,
            3059,
            3107,
            767,
            3108,
            2799,
            4639, // 2656
            1006,
            4302,
            4640,
            2346,
            1267,
            2179,
            3730,
            3230,
            778,
            4048,
            3231,
            2731,
            1597,
            2667,
            5312,
            4641, // 2672
            5313,
            3493,
            5314,
            5315,
            5316,
            3310,
            2698,
            1433,
            3311,
            131,
            95,
            1504,
            4049,
            723,
            4303,
            3166, // 2688
            1842,
            3610,
            2768,
            2192,
            4050,
            2028,
            2105,
            3731,
            5317,
            3013,
            4051,
            1218,
            5318,
            3380,
            3232,
            4052, // 2704
            4304,
            2584,
            248,
            1634,
            3864,
            912,
            5319,
            2845,
            3732,
            3060,
            3865,
            654,
            53,
            5320,
            3014,
            5321, // 2720
            1688,
            4642,
            777,
            3494,
            1032,
            4053,
            1425,
            5322,
            191,
            820,
            2121,
            2846,
            971,
            4643,
            931,
            3233, // 2736
            135,
            664,
            783,
            3866,
            1998,
            772,
            2922,
            1936,
            4054,
            3867,
            4644,
            2923,
            3234,
            282,
            2732,
            640, // 2752
            1372,
            3495,
            1127,
            922,
            325,
            3381,
            5323,
            5324,
            711,
            2045,
            5325,
            5326,
            4055,
            2223,
            2800,
            1937, // 2768
            4056,
            3382,
            2224,
            2255,
            3868,
            2305,
            5327,
            4645,
            3869,
            1258,
            3312,
            4057,
            3235,
            2139,
            2965,
            4058, // 2784
            4059,
            5328,
            2225,
            258,
            3236,
            4646,
            101,
            1227,
            5329,
            3313,
            1755,
            5330,
            1391,
            3314,
            5331,
            2924, // 2800
            2057,
            893,
            5332,
            5333,
            5334,
            1402,
            4305,
            2347,
            5335,
            5336,
            3237,
            3611,
            5337,
            5338,
            878,
            1325, // 2816
            1781,
            2801,
            4647,
            259,
            1385,
            2585,
            744,
            1183,
            2272,
            4648,
            5339,
            4060,
            2509,
            5340,
            684,
            1024, // 2832
            4306,
            5341,
            472,
            3612,
            3496,
            1165,
            3315,
            4061,
            4062,
            322,
            2153,
            881,
            455,
            1695,
            1152,
            1340, // 2848
            660,
            554,
            2154,
            4649,
            1058,
            4650,
            4307,
            830,
            1065,
            3383,
            4063,
            4651,
            1924,
            5342,
            1703,
            1919, // 2864
            5343,
            932,
            2273,
            122,
            5344,
            4652,
            947,
            677,
            5345,
            3870,
            2637,
            297,
            1906,
            1925,
            2274,
            4653, // 2880
            2322,
            3316,
            5346,
            5347,
            4308,
            5348,
            4309,
            84,
            4310,
            112,
            989,
            5349,
            547,
            1059,
            4064,
            701, // 2896
            3613,
            1019,
            5350,
            4311,
            5351,
            3497,
            942,
            639,
            457,
            2306,
            2456,
            993,
            2966,
            407,
            851,
            494, // 2912
            4654,
            3384,
            927,
            5352,
            1237,
            5353,
            2426,
            3385,
            573,
            4312,
            680,
            921,
            2925,
            1279,
            1875,
            285, // 2928
            790,
            1448,
            1984,
            719,
            2168,
            5354,
            5355,
            4655,
            4065,
            4066,
            1649,
            5356,
            1541,
            563,
            5357,
            1077, // 2944
            5358,
            3386,
            3061,
            3498,
            511,
            3015,
            4067,
            4068,
            3733,
            4069,
            1268,
            2572,
            3387,
            3238,
            4656,
            4657, // 2960
            5359,
            535,
            1048,
            1276,
            1189,
            2926,
            2029,
            3167,
            1438,
            1373,
            2847,
            2967,
            1134,
            2013,
            5360,
            4313, // 2976
            1238,
            2586,
            3109,
            1259,
            5361,
            700,
            5362,
            2968,
            3168,
            3734,
            4314,
            5363,
            4315,
            1146,
            1876,
            1907, // 2992
            4658,
            2611,
            4070,
            781,
            2427,
            132,
            1589,
            203,
            147,
            273,
            2802,
            2407,
            898,
            1787,
            2155,
            4071, // 3008
            4072,
            5364,
            3871,
            2803,
            5365,
            5366,
            4659,
            4660,
            5367,
            3239,
            5368,
            1635,
            3872,
            965,
            5369,
            1805, // 3024
            2699,
            1516,
            3614,
            1121,
            1082,
            1329,
            3317,
            4073,
            1449,
            3873,
            65,
            1128,
            2848,
            2927,
            2769,
            1590, // 3040
            3874,
            5370,
            5371,
            12,
            2668,
            45,
            976,
            2587,
            3169,
            4661,
            517,
            2535,
            1013,
            1037,
            3240,
            5372, // 3056
            3875,
            2849,
            5373,
            3876,
            5374,
            3499,
            5375,
            2612,
            614,
            1999,
            2323,
            3877,
            3110,
            2733,
            2638,
            5376, // 3072
            2588,
            4316,
            599,
            1269,
            5377,
            1811,
            3735,
            5378,
            2700,
            3111,
            759,
            1060,
            489,
            1806,
            3388,
            3318, // 3088
            1358,
            5379,
            5380,
            2391,
            1387,
            1215,
            2639,
            2256,
            490,
            5381,
            5382,
            4317,
            1759,
            2392,
            2348,
            5383, // 3104
            4662,
            3878,
            1908,
            4074,
            2640,
            1807,
            3241,
            4663,
            3500,
            3319,
            2770,
            2349,
            874,
            5384,
            5385,
            3501, // 3120
            3736,
            1859,
            91,
            2928,
            3737,
            3062,
            3879,
            4664,
            5386,
            3170,
            4075,
            2669,
            5387,
            3502,
            1202,
            1403, // 3136
            3880,
            2969,
            2536,
            1517,
            2510,
            4665,
            3503,
            2511,
            5388,
            4666,
            5389,
            2701,
            1886,
            1495,
            1731,
            4076, // 3152
            2370,
            4667,
            5390,
            2030,
            5391,
            5392,
            4077,
            2702,
            1216,
            237,
            2589,
            4318,
            2324,
            4078,
            3881,
            4668, // 3168
            4669,
            2703,
            3615,
            3504,
            445,
            4670,
            5393,
            5394,
            5395,
            5396,
            2771,
            61,
            4079,
            3738,
            1823,
            4080, // 3184
            5397,
            687,
            2046,
            935,
            925,
            405,
            2670,
            703,
            1096,
            1860,
            2734,
            4671,
            4081,
            1877,
            1367,
            2704, // 3200
            3389,
            918,
            2106,
            1782,
            2483,
            334,
            3320,
            1611,
            1093,
            4672,
            564,
            3171,
            3505,
            3739,
            3390,
            945, // 3216
            2641,
            2058,
            4673,
            5398,
            1926,
            872,
            4319,
            5399,
            3506,
            2705,
            3112,
            349,
            4320,
            3740,
            4082,
            4674, // 3232
            3882,
            4321,
            3741,
            2156,
            4083,
            4675,
            4676,
            4322,
            4677,
            2408,
            2047,
            782,
            4084,
            400,
            251,
            4323, // 3248
            1624,
            5400,
            5401,
            277,
            3742,
            299,
            1265,
            476,
            1191,
            3883,
            2122,
            4324,
            4325,
            1109,
            205,
            5402, // 3264
            2590,
            1000,
            2157,
            3616,
            1861,
            5403,
            5404,
            5405,
            4678,
            5406,
            4679,
            2573,
            107,
            2484,
            2158,
            4085, // 3280
            3507,
            3172,
            5407,
            1533,
            541,
            1301,
            158,
            753,
            4326,
            2886,
            3617,
            5408,
            1696,
            370,
            1088,
            4327, // 3296
            4680,
            3618,
            579,
            327,
            440,
            162,
            2244,
            269,
            1938,
            1374,
            3508,
            968,
            3063,
            56,
            1396,
            3113, // 3312
            2107,
            3321,
            3391,
            5409,
            1927,
            2159,
            4681,
            3016,
            5410,
            3619,
            5411,
            5412,
            3743,
            4682,
            2485,
            5413, // 3328
            2804,
            5414,
            1650,
            4683,
            5415,
            2613,
            5416,
            5417,
            4086,
            2671,
            3392,
            1149,
            3393,
            4087,
            3884,
            4088, // 3344
            5418,
            1076,
            49,
            5419,
            951,
            3242,
            3322,
            3323,
            450,
            2850,
            920,
            5420,
            1812,
            2805,
            2371,
            4328, // 3360
            1909,
            1138,
            2372,
            3885,
            3509,
            5421,
            3243,
            4684,
            1910,
            1147,
            1518,
            2428,
            4685,
            3886,
            5422,
            4686, // 3376
            2393,
            2614,
            260,
            1796,
            3244,
            5423,
            5424,
            3887,
            3324,
            708,
            5425,
            3620,
            1704,
            5426,
            3621,
            1351, // 3392
            1618,
            3394,
            3017,
            1887,
            944,
            4329,
            3395,
            4330,
            3064,
            3396,
            4331,
            5427,
            3744,
            422,
            413,
            1714, // 3408
            3325,
            500,
            2059,
            2350,
            4332,
            2486,
            5428,
            1344,
            1911,
            954,
            5429,
            1668,
            5430,
            5431,
            4089,
            2409, // 3424
            4333,
            3622,
            3888,
            4334,
            5432,
            2307,
            1318,
            2512,
            3114,
            133,
            3115,
            2887,
            4687,
            629,
            31,
            2851, // 3440
            2706,
            3889,
            4688,
            850,
            949,
            4689,
            4090,
            2970,
            1732,
            2089,
            4335,
            1496,
            1853,
            5433,
            4091,
            620, // 3456
            3245,
            981,
            1242,
            3745,
            3397,
            1619,
            3746,
            1643,
            3326,
            2140,
            2457,
            1971,
            1719,
            3510,
            2169,
            5434, // 3472
            3246,
            5435,
            5436,
            3398,
            1829,
            5437,
            1277,
            4690,
            1565,
            2048,
            5438,
            1636,
            3623,
            3116,
            5439,
            869, // 3488
            2852,
            655,
            3890,
            3891,
            3117,
            4092,
            3018,
            3892,
            1310,
            3624,
            4691,
            5440,
            5441,
            5442,
            1733,
            558, // 3504
            4692,
            3747,
            335,
            1549,
            3065,
            1756,
            4336,
            3748,
            1946,
            3511,
            1830,
            1291,
            1192,
            470,
            2735,
            2108, // 3520
            2806,
            913,
            1054,
            4093,
            5443,
            1027,
            5444,
            3066,
            4094,
            4693,
            982,
            2672,
            3399,
            3173,
            3512,
            3247, // 3536
            3248,
            1947,
            2807,
            5445,
            571,
            4694,
            5446,
            1831,
            5447,
            3625,
            2591,
            1523,
            2429,
            5448,
            2090,
            984, // 3552
            4695,
            3749,
            1960,
            5449,
            3750,
            852,
            923,
            2808,
            3513,
            3751,
            969,
            1519,
            999,
            2049,
            2325,
            1705, // 3568
            5450,
            3118,
            615,
            1662,
            151,
            597,
            4095,
            2410,
            2326,
            1049,
            275,
            4696,
            3752,
            4337,
            568,
            3753, // 3584
            3626,
            2487,
            4338,
            3754,
            5451,
            2430,
            2275,
            409,
            3249,
            5452,
            1566,
            2888,
            3514,
            1002,
            769,
            2853, // 3600
            194,
            2091,
            3174,
            3755,
            2226,
            3327,
            4339,
            628,
            1505,
            5453,
            5454,
            1763,
            2180,
            3019,
            4096,
            521, // 3616
            1161,
            2592,
            1788,
            2206,
            2411,
            4697,
            4097,
            1625,
            4340,
            4341,
            412,
            42,
            3119,
            464,
            5455,
            2642, // 3632
            4698,
            3400,
            1760,
            1571,
            2889,
            3515,
            2537,
            1219,
            2207,
            3893,
            2643,
            2141,
            2373,
            4699,
            4700,
            3328, // 3648
            1651,
            3401,
            3627,
            5456,
            5457,
            3628,
            2488,
            3516,
            5458,
            3756,
            5459,
            5460,
            2276,
            2092,
            460,
            5461, // 3664
            4701,
            5462,
            3020,
            962,
            588,
            3629,
            289,
            3250,
            2644,
            1116,
            52,
            5463,
            3067,
            1797,
            5464,
            5465, // 3680
            5466,
            1467,
            5467,
            1598,
            1143,
            3757,
            4342,
            1985,
            1734,
            1067,
            4702,
            1280,
            3402,
            465,
            4703,
            1572, // 3696
            510,
            5468,
            1928,
            2245,
            1813,
            1644,
            3630,
            5469,
            4704,
            3758,
            5470,
            5471,
            2673,
            1573,
            1534,
            5472, // 3712
            5473,
            536,
            1808,
            1761,
            3517,
            3894,
            3175,
            2645,
            5474,
            5475,
            5476,
            4705,
            3518,
            2929,
            1912,
            2809, // 3728
            5477,
            3329,
            1122,
            377,
            3251,
            5478,
            360,
            5479,
            5480,
            4343,
            1529,
            551,
            5481,
            2060,
            3759,
            1769, // 3744
            2431,
            5482,
            2930,
            4344,
            3330,
            3120,
            2327,
            2109,
            2031,
            4706,
            1404,
            136,
            1468,
            1479,
            672,
            1171, // 3760
            3252,
            2308,
            271,
            3176,
            5483,
            2772,
            5484,
            2050,
            678,
            2736,
            865,
            1948,
            4707,
            5485,
            2014,
            4098, // 3776
            2971,
            5486,
            2737,
            2227,
            1397,
            3068,
            3760,
            4708,
            4709,
            1735,
            2931,
            3403,
            3631,
            5487,
            3895,
            509, // 3792
            2854,
            2458,
            2890,
            3896,
            5488,
            5489,
            3177,
            3178,
            4710,
            4345,
            2538,
            4711,
            2309,
            1166,
            1010,
            552, // 3808
            681,
            1888,
            5490,
            5491,
            2972,
            2973,
            4099,
            1287,
            1596,
            1862,
            3179,
            358,
            453,
            736,
            175,
            478, // 3824
            1117,
            905,
            1167,
            1097,
            5492,
            1854,
            1530,
            5493,
            1706,
            5494,
            2181,
            3519,
            2292,
            3761,
            3520,
            3632, // 3840
            4346,
            2093,
            4347,
            5495,
            3404,
            1193,
            2489,
            4348,
            1458,
            2193,
            2208,
            1863,
            1889,
            1421,
            3331,
            2932, // 3856
            3069,
            2182,
            3521,
            595,
            2123,
            5496,
            4100,
            5497,
            5498,
            4349,
            1707,
            2646,
            223,
            3762,
            1359,
            751, // 3872
            3121,
            183,
            3522,
            5499,
            2810,
            3021,
            419,
            2374,
            633,
            704,
            3897,
            2394,
            241,
            5500,
            5501,
            5502, // 3888
            838,
            3022,
            3763,
            2277,
            2773,
            2459,
            3898,
            1939,
            2051,
            4101,
            1309,
            3122,
            2246,
            1181,
            5503,
            1136, // 3904
            2209,
            3899,
            2375,
            1446,
            4350,
            2310,
            4712,
            5504,
            5505,
            4351,
            1055,
            2615,
            484,
            3764,
            5506,
            4102, // 3920
            625,
            4352,
            2278,
            3405,
            1499,
            4353,
            4103,
            5507,
            4104,
            4354,
            3253,
            2279,
            2280,
            3523,
            5508,
            5509, // 3936
            2774,
            808,
            2616,
            3765,
            3406,
            4105,
            4355,
            3123,
            2539,
            526,
            3407,
            3900,
            4356,
            955,
            5510,
            1620, // 3952
            4357,
            2647,
            2432,
            5511,
            1429,
            3766,
            1669,
            1832,
            994,
            928,
            5512,
            3633,
            1260,
            5513,
            5514,
            5515, // 3968
            1949,
            2293,
            741,
            2933,
            1626,
            4358,
            2738,
            2460,
            867,
            1184,
            362,
            3408,
            1392,
            5516,
            5517,
            4106, // 3984
            4359,
            1770,
            1736,
            3254,
            2934,
            4713,
            4714,
            1929,
            2707,
            1459,
            1158,
            5518,
            3070,
            3409,
            2891,
            1292, // 4000
            1930,
            2513,
            2855,
            3767,
            1986,
            1187,
            2072,
            2015,
            2617,
            4360,
            5519,
            2574,
            2514,
            2170,
            3768,
            2490, // 4016
            3332,
            5520,
            3769,
            4715,
            5521,
            5522,
            666,
            1003,
            3023,
            1022,
            3634,
            4361,
            5523,
            4716,
            1814,
            2257, // 4032
            574,
            3901,
            1603,
            295,
            1535,
            705,
            3902,
            4362,
            283,
            858,
            417,
            5524,
            5525,
            3255,
            4717,
            4718, // 4048
            3071,
            1220,
            1890,
            1046,
            2281,
            2461,
            4107,
            1393,
            1599,
            689,
            2575,
            388,
            4363,
            5526,
            2491,
            802, // 4064
            5527,
            2811,
            3903,
            2061,
            1405,
            2258,
            5528,
            4719,
            3904,
            2110,
            1052,
            1345,
            3256,
            1585,
            5529,
            809, // 4080
            5530,
            5531,
            5532,
            575,
            2739,
            3524,
            956,
            1552,
            1469,
            1144,
            2328,
            5533,
            2329,
            1560,
            2462,
            3635, // 4096
            3257,
            4108,
            616,
            2210,
            4364,
            3180,
            2183,
            2294,
            5534,
            1833,
            5535,
            3525,
            4720,
            5536,
            1319,
            3770, // 4112
            3771,
            1211,
            3636,
            1023,
            3258,
            1293,
            2812,
            5537,
            5538,
            5539,
            3905,
            607,
            2311,
            3906,
            762,
            2892, // 4128
            1439,
            4365,
            1360,
            4721,
            1485,
            3072,
            5540,
            4722,
            1038,
            4366,
            1450,
            2062,
            2648,
            4367,
            1379,
            4723, // 4144
            2593,
            5541,
            5542,
            4368,
            1352,
            1414,
            2330,
            2935,
            1172,
            5543,
            5544,
            3907,
            3908,
            4724,
            1798,
            1451, // 4160
            5545,
            5546,
            5547,
            5548,
            2936,
            4109,
            4110,
            2492,
            2351,
            411,
            4111,
            4112,
            3637,
            3333,
            3124,
            4725, // 4176
            1561,
            2674,
            1452,
            4113,
            1375,
            5549,
            5550,
            47,
            2974,
            316,
            5551,
            1406,
            1591,
            2937,
            3181,
            5552, // 4192
            1025,
            2142,
            3125,
            3182,
            354,
            2740,
            884,
            2228,
            4369,
            2412,
            508,
            3772,
            726,
            3638,
            996,
            2433, // 4208
            3639,
            729,
            5553,
            392,
            2194,
            1453,
            4114,
            4726,
            3773,
            5554,
            5555,
            2463,
            3640,
            2618,
            1675,
            2813, // 4224
            919,
            2352,
            2975,
            2353,
            1270,
            4727,
            4115,
            73,
            5556,
            5557,
            647,
            5558,
            3259,
            2856,
            2259,
            1550, // 4240
            1346,
            3024,
            5559,
            1332,
            883,
            3526,
            5560,
            5561,
            5562,
            5563,
            3334,
            2775,
            5564,
            1212,
            831,
            1347, // 4256
            4370,
            4728,
            2331,
            3909,
            1864,
            3073,
            720,
            3910,
            4729,
            4730,
            3911,
            5565,
            4371,
            5566,
            5567,
            4731, // 4272
            5568,
            5569,
            1799,
            4732,
            3774,
            2619,
            4733,
            3641,
            1645,
            2376,
            4734,
            5570,
            2938,
            669,
            2211,
            2675, // 4288
            2434,
            5571,
            2893,
            5572,
            5573,
            1028,
            3260,
            5574,
            4372,
            2413,
            5575,
            2260,
            1353,
            5576,
            5577,
            4735, // 4304
            3183,
            518,
            5578,
            4116,
            5579,
            4373,
            1961,
            5580,
            2143,
            4374,
            5581,
            5582,
            3025,
            2354,
            2355,
            3912, // 4320
            516,
            1834,
            1454,
            4117,
            2708,
            4375,
            4736,
            2229,
            2620,
            1972,
            1129,
            3642,
            5583,
            2776,
            5584,
            2976, // 4336
            1422,
            577,
            1470,
            3026,
            1524,
            3410,
            5585,
            5586,
            432,
            4376,
            3074,
            3527,
            5587,
            2594,
            1455,
            2515, // 4352
            2230,
            1973,
            1175,
            5588,
            1020,
            2741,
            4118,
            3528,
            4737,
            5589,
            2742,
            5590,
            1743,
            1361,
            3075,
            3529, // 4368
            2649,
            4119,
            4377,
            4738,
            2295,
            895,
            924,
            4378,
            2171,
            331,
            2247,
            3076,
            166,
            1627,
            3077,
            1098, // 4384
            5591,
            1232,
            2894,
            2231,
            3411,
            4739,
            657,
            403,
            1196,
            2377,
            542,
            3775,
            3412,
            1600,
            4379,
            3530, // 4400
            5592,
            4740,
            2777,
            3261,
            576,
            530,
            1362,
            4741,
            4742,
            2540,
            2676,
            3776,
            4120,
            5593,
            842,
            3913, // 4416
            5594,
            2814,
            2032,
            1014,
            4121,
            213,
            2709,
            3413,
            665,
            621,
            4380,
            5595,
            3777,
            2939,
            2435,
            5596, // 4432
            2436,
            3335,
            3643,
            3414,
            4743,
            4381,
            2541,
            4382,
            4744,
            3644,
            1682,
            4383,
            3531,
            1380,
            5597,
            724, // 4448
            2282,
            600,
            1670,
            5598,
            1337,
            1233,
            4745,
            3126,
            2248,
            5599,
            1621,
            4746,
            5600,
            651,
            4384,
            5601, // 4464
            1612,
            4385,
            2621,
            5602,
            2857,
            5603,
            2743,
            2312,
            3078,
            5604,
            716,
            2464,
            3079,
            174,
            1255,
            2710, // 4480
            4122,
            3645,
            548,
            1320,
            1398,
            728,
            4123,
            1574,
            5605,
            1891,
            1197,
            3080,
            4124,
            5606,
            3081,
            3082, // 4496
            3778,
            3646,
            3779,
            747,
            5607,
            635,
            4386,
            4747,
            5608,
            5609,
            5610,
            4387,
            5611,
            5612,
            4748,
            5613, // 4512
            3415,
            4749,
            2437,
            451,
            5614,
            3780,
            2542,
            2073,
            4388,
            2744,
            4389,
            4125,
            5615,
            1764,
            4750,
            5616, // 4528
            4390,
            350,
            4751,
            2283,
            2395,
            2493,
            5617,
            4391,
            4126,
            2249,
            1434,
            4127,
            488,
            4752,
            458,
            4392, // 4544
            4128,
            3781,
            771,
            1330,
            2396,
            3914,
            2576,
            3184,
            2160,
            2414,
            1553,
            2677,
            3185,
            4393,
            5618,
            2494, // 4560
            2895,
            2622,
            1720,
            2711,
            4394,
            3416,
            4753,
            5619,
            2543,
            4395,
            5620,
            3262,
            4396,
            2778,
            5621,
            2016, // 4576
            2745,
            5622,
            1155,
            1017,
            3782,
            3915,
            5623,
            3336,
            2313,
            201,
            1865,
            4397,
            1430,
            5624,
            4129,
            5625, // 4592
            5626,
            5627,
            5628,
            5629,
            4398,
            1604,
            5630,
            414,
            1866,
            371,
            2595,
            4754,
            4755,
            3532,
            2017,
            3127, // 4608
            4756,
            1708,
            960,
            4399,
            887,
            389,
            2172,
            1536,
            1663,
            1721,
            5631,
            2232,
            4130,
            2356,
            2940,
            1580, // 4624
            5632,
            5633,
            1744,
            4757,
            2544,
            4758,
            4759,
            5634,
            4760,
            5635,
            2074,
            5636,
            4761,
            3647,
            3417,
            2896, // 4640
            4400,
            5637,
            4401,
            2650,
            3418,
            2815,
            673,
            2712,
            2465,
            709,
            3533,
            4131,
            3648,
            4402,
            5638,
            1148, // 4656
            502,
            634,
            5639,
            5640,
            1204,
            4762,
            3649,
            1575,
            4763,
            2623,
            3783,
            5641,
            3784,
            3128,
            948,
            3263, // 4672
            121,
            1745,
            3916,
            1110,
            5642,
            4403,
            3083,
            2516,
            3027,
            4132,
            3785,
            1151,
            1771,
            3917,
            1488,
            4133, // 4688
            1987,
            5643,
            2438,
            3534,
            5644,
            5645,
            2094,
            5646,
            4404,
            3918,
            1213,
            1407,
            2816,
            531,
            2746,
            2545, // 4704
            3264,
            1011,
            1537,
            4764,
            2779,
            4405,
            3129,
            1061,
            5647,
            3786,
            3787,
            1867,
            2897,
            5648,
            2018,
            120, // 4720
            4406,
            4407,
            2063,
            3650,
            3265,
            2314,
            3919,
            2678,
            3419,
            1955,
            4765,
            4134,
            5649,
            3535,
            1047,
            2713, // 4736
            1266,
            5650,
            1368,
            4766,
            2858,
            649,
            3420,
            3920,
            2546,
            2747,
            1102,
            2859,
            2679,
            5651,
            5652,
            2000, // 4752
            5653,
            1111,
            3651,
            2977,
            5654,
            2495,
            3921,
            3652,
            2817,
            1855,
            3421,
            3788,
            5655,
            5656,
            3422,
            2415, // 4768
            2898,
            3337,
            3266,
            3653,
            5657,
            2577,
            5658,
            3654,
            2818,
            4135,
            1460,
            856,
            5659,
            3655,
            5660,
            2899, // 4784
            2978,
            5661,
            2900,
            3922,
            5662,
            4408,
            632,
            2517,
            875,
            3923,
            1697,
            3924,
            2296,
            5663,
            5664,
            4767, // 4800
            3028,
            1239,
            580,
            4768,
            4409,
            5665,
            914,
            936,
            2075,
            1190,
            4136,
            1039,
            2124,
            5666,
            5667,
            5668, // 4816
            5669,
            3423,
            1473,
            5670,
            1354,
            4410,
            3925,
            4769,
            2173,
            3084,
            4137,
            915,
            3338,
            4411,
            4412,
            3339, // 4832
            1605,
            1835,
            5671,
            2748,
            398,
            3656,
            4413,
            3926,
            4138,
            328,
            1913,
            2860,
            4139,
            3927,
            1331,
            4414, // 4848
            3029,
            937,
            4415,
            5672,
            3657,
            4140,
            4141,
            3424,
            2161,
            4770,
            3425,
            524,
            742,
            538,
            3085,
            1012, // 4864
            5673,
            5674,
            3928,
            2466,
            5675,
            658,
            1103,
            225,
            3929,
            5676,
            5677,
            4771,
            5678,
            4772,
            5679,
            3267, // 4880
            1243,
            5680,
            4142,
            963,
            2250,
            4773,
            5681,
            2714,
            3658,
            3186,
            5682,
            5683,
            2596,
            2332,
            5684,
            4774, // 4896
            5685,
            5686,
            5687,
            3536,
            957,
            3426,
            2547,
            2033,
            1931,
            2941,
            2467,
            870,
            2019,
            3659,
            1746,
            2780, // 4912
            2781,
            2439,
            2468,
            5688,
            3930,
            5689,
            3789,
            3130,
            3790,
            3537,
            3427,
            3791,
            5690,
            1179,
            3086,
            5691, // 4928
            3187,
            2378,
            4416,
            3792,
            2548,
            3188,
            3131,
            2749,
            4143,
            5692,
            3428,
            1556,
            2549,
            2297,
            977,
            2901, // 4944
            2034,
            4144,
            1205,
            3429,
            5693,
            1765,
            3430,
            3189,
            2125,
            1271,
            714,
            1689,
            4775,
            3538,
            5694,
            2333, // 4960
            3931,
            533,
            4417,
            3660,
            2184,
            617,
            5695,
            2469,
            3340,
            3539,
            2315,
            5696,
            5697,
            3190,
            5698,
            5699, // 4976
            3932,
            1988,
            618,
            427,
            2651,
            3540,
            3431,
            5700,
            5701,
            1244,
            1690,
            5702,
            2819,
            4418,
            4776,
            5703, // 4992
            3541,
            4777,
            5704,
            2284,
            1576,
            473,
            3661,
            4419,
            3432,
            972,
            5705,
            3662,
            5706,
            3087,
            5707,
            5708, // 5008
            4778,
            4779,
            5709,
            3793,
            4145,
            4146,
            5710,
            153,
            4780,
            356,
            5711,
            1892,
            2902,
            4420,
            2144,
            408, // 5024
            803,
            2357,
            5712,
            3933,
            5713,
            4421,
            1646,
            2578,
            2518,
            4781,
            4782,
            3934,
            5714,
            3935,
            4422,
            5715, // 5040
            2416,
            3433,
            752,
            5716,
            5717,
            1962,
            3341,
            2979,
            5718,
            746,
            3030,
            2470,
            4783,
            4423,
            3794,
            698, // 5056
            4784,
            1893,
            4424,
            3663,
            2550,
            4785,
            3664,
            3936,
            5719,
            3191,
            3434,
            5720,
            1824,
            1302,
            4147,
            2715, // 5072
            3937,
            1974,
            4425,
            5721,
            4426,
            3192,
            823,
            1303,
            1288,
            1236,
            2861,
            3542,
            4148,
            3435,
            774,
            3938, // 5088
            5722,
            1581,
            4786,
            1304,
            2862,
            3939,
            4787,
            5723,
            2440,
            2162,
            1083,
            3268,
            4427,
            4149,
            4428,
            344, // 5104
            1173,
            288,
            2316,
            454,
            1683,
            5724,
            5725,
            1461,
            4788,
            4150,
            2597,
            5726,
            5727,
            4789,
            985,
            894, // 5120
            5728,
            3436,
            3193,
            5729,
            1914,
            2942,
            3795,
            1989,
            5730,
            2111,
            1975,
            5731,
            4151,
            5732,
            2579,
            1194, // 5136
            425,
            5733,
            4790,
            3194,
            1245,
            3796,
            4429,
            5734,
            5735,
            2863,
            5736,
            636,
            4791,
            1856,
            3940,
            760, // 5152
            1800, 5737,
            4430,
            2212,
            1508,
            4792,
            4152,
            1894,
            1684,
            2298,
            5738,
            5739,
            4793,
            4431,
            4432,
            2213, // 5168
            479, 5740, 5741,
            832,
            5742,
            4153,
            2496,
            5743,
            2980,
            2497,
            3797,
            990,
            3132,
            627,
            1815,
            2652, // 5184
            4433, 1582, 4434, 2126,
            2112,
            3543,
            4794,
            5744,
            799,
            4435,
            3195,
            5745,
            4795,
            2113,
            1737,
            3031, // 5200
            1018, 543, 754, 4436, 3342,
            1676,
            4796,
            4797,
            4154,
            4798,
            1489,
            5746,
            3544,
            5747,
            2624,
            2903, // 5216
            4155, 5748, 5749, 2981, 5750, 5751,
            5752,
            5753,
            3196,
            4799,
            4800,
            2185,
            1722,
            5754,
            3269,
            3270, // 5232
            1843, 3665, 1715, 481, 365, 1976, 1857,
            5755,
            5756,
            1963,
            2498,
            4801,
            5757,
            2127,
            3666,
            3271, // 5248
            433, 1895, 2064, 2076, 5758, 602, 2750, 5759,
            5760,
            5761,
            5762,
            5763,
            3032,
            1628,
            3437,
            5764, // 5264
            3197, 4802, 4156, 2904, 4803, 2519, 5765, 2551, 2782,
            5766,
            5767,
            5768,
            3343,
            4804,
            2905,
            5769, // 5280
            4805, 5770, 2864, 4806, 4807, 1221, 2982, 4157, 2520, 5771,
            5772,
            5773,
            1868,
            1990,
            5774,
            5775, // 5296
            5776, 1896, 5777, 5778, 4808, 1897, 4158, 318, 5779, 2095, 4159,
            4437,
            5780,
            5781,
            485,
            5782, // 5312
            938, 3941, 553, 2680, 116, 5783, 3942, 3667, 5784, 3545, 2681, 2783,
            3438,
            3344,
            2820,
            5785, // 5328
            3668, 2943, 4160, 1747, 2944, 2983, 5786, 5787, 207, 5788, 4809, 5789, 4810,
            2521,
            5790,
            3033, // 5344
            890, 3669, 3943, 5791, 1878, 3798, 3439, 5792, 2186, 2358, 3440, 1652, 5793, 5794,
            5795,
            941, // 5360
            2299, 208, 3546, 4161, 2020, 330, 4438, 3944, 2906, 2499, 3799, 4439, 4811, 5796, 5797,
            5798, // 5376 //last 512

    /***************************************************************************************
     * Everything below is of no interest for detection purpose *
     * 2522,1613,4812,
     * 5799,3345,3945,2523,5800,4162,5801,1637,4163,2471,4813,3946,5802, // 5392
     * 2500
     * ,3034,3800,5803,5804,2195,4814,5805,2163,5806,5807,5808,5809,5810,5811
     * ,5812, // 5408
     * 5813,5814,5815,5816,5817,5818,5819,5820,5821,5822,5823,5824
     * ,5825,5826,5827,5828, // 5424
     * 5829,5830,5831,5832,5833,5834,5835,5836,5837
     * ,5838,5839,5840,5841,5842,5843,5844, // 5440
     * 5845,5846,5847,5848,5849,5850
     * ,5851,5852,5853,5854,5855,5856,5857,5858,5859,5860, // 5456
     * 5861,5862,5863
     * ,5864,5865,5866,5867,5868,5869,5870,5871,5872,5873,5874,5875,5876, //
     * 5472
     * 5877,5878,5879,5880,5881,5882,5883,5884,5885,5886,5887,5888,5889,5890
     * ,5891,5892, // 5488
     * 5893,5894,5895,5896,5897,5898,5899,5900,5901,5902,5903
     * ,5904,5905,5906,5907,5908, // 5504
     * 5909,5910,5911,5912,5913,5914,5915,5916
     * ,5917,5918,5919,5920,5921,5922,5923,5924, // 5520
     * 5925,5926,5927,5928,5929
     * ,5930,5931,5932,5933,5934,5935,5936,5937,5938,5939,5940, // 5536
     * 5941,5942
     * ,5943,5944,5945,5946,5947,5948,5949,5950,5951,5952,5953,5954,5955,5956,
     * // 5552
     * 5957,5958,5959,5960,5961,5962,5963,5964,5965,5966,5967,5968,5969,5970
     * ,5971,5972, // 5568
     * 5973,5974,5975,5976,5977,5978,5979,5980,5981,5982,5983
     * ,5984,5985,5986,5987,5988, // 5584
     * 5989,5990,5991,5992,5993,5994,5995,5996
     * ,5997,5998,5999,6000,6001,6002,6003,6004, // 5600
     * 6005,6006,6007,6008,6009
     * ,6010,6011,6012,6013,6014,6015,6016,6017,6018,6019,6020, // 5616
     * 6021,6022
     * ,6023,6024,6025,6026,6027,6028,6029,6030,6031,6032,6033,6034,6035,6036,
     * // 5632
     * 6037,6038,6039,6040,6041,6042,6043,6044,6045,6046,6047,6048,6049,6050
     * ,6051,6052, // 5648
     * 6053,6054,6055,6056,6057,6058,6059,6060,6061,6062,6063
     * ,6064,6065,6066,6067,6068, // 5664
     * 6069,6070,6071,6072,6073,6074,6075,6076
     * ,6077,6078,6079,6080,6081,6082,6083,6084, // 5680
     * 6085,6086,6087,6088,6089
     * ,6090,6091,6092,6093,6094,6095,6096,6097,6098,6099,6100, // 5696
     * 6101,6102
     * ,6103,6104,6105,6106,6107,6108,6109,6110,6111,6112,6113,6114,6115,6116,
     * // 5712
     * 6117,6118,6119,6120,6121,6122,6123,6124,6125,6126,6127,6128,6129,6130
     * ,6131,6132, // 5728
     * 6133,6134,6135,6136,6137,6138,6139,6140,6141,6142,6143
     * ,6144,6145,6146,6147,6148, // 5744
     * 6149,6150,6151,6152,6153,6154,6155,6156
     * ,6157,6158,6159,6160,6161,6162,6163,6164, // 5760
     * 6165,6166,6167,6168,6169
     * ,6170,6171,6172,6173,6174,6175,6176,6177,6178,6179,6180, // 5776
     * 6181,6182
     * ,6183,6184,6185,6186,6187,6188,6189,6190,6191,6192,6193,6194,6195,6196,
     * // 5792
     * 6197,6198,6199,6200,6201,6202,6203,6204,6205,6206,6207,6208,6209,6210
     * ,6211,6212, // 5808
     * 6213,6214,6215,6216,6217,6218,6219,6220,6221,6222,6223
     * ,3670,6224,6225,6226,6227, // 5824
     * 6228,6229,6230,6231,6232,6233,6234,6235
     * ,6236,6237,6238,6239,6240,6241,6242,6243, // 5840
     * 6244,6245,6246,6247,6248
     * ,6249,6250,6251,6252,6253,6254,6255,6256,6257,6258,6259, // 5856
     * 6260,6261
     * ,6262,6263,6264,6265,6266,6267,6268,6269,6270,6271,6272,6273,6274,6275,
     * // 5872
     * 6276,6277,6278,6279,6280,6281,6282,6283,6284,6285,4815,6286,6287,6288
     * ,6289,6290, // 5888
     * 6291,6292,4816,6293,6294,6295,6296,6297,6298,6299,6300
     * ,6301,6302,6303,6304,6305, // 5904
     * 6306,6307,6308,6309,6310,6311,4817,4818
     * ,6312,6313,6314,6315,6316,6317,6318,4819, // 5920
     * 6319,6320,6321,6322,6323
     * ,6324,6325,6326,6327,6328,6329,6330,6331,6332,6333,6334, // 5936
     * 6335,6336
     * ,6337,4820,6338,6339,6340,6341,6342,6343,6344,6345,6346,6347,6348,6349,
     * // 5952
     * 6350,6351,6352,6353,6354,6355,6356,6357,6358,6359,6360,6361,6362,6363
     * ,6364,6365, // 5968
     * 6366,6367,6368,6369,6370,6371,6372,6373,6374,6375,6376
     * ,6377,6378,6379,6380,6381, // 5984
     * 6382,6383,6384,6385,6386,6387,6388,6389
     * ,6390,6391,6392,6393,6394,6395,6396,6397, // 6000
     * 6398,6399,6400,6401,6402
     * ,6403,6404,6405,6406,6407,6408,6409,6410,3441,6411,6412, // 6016
     * 6413,6414
     * ,6415,6416,6417,6418,6419,6420,6421,6422,6423,6424,6425,4440,6426,6427,
     * // 6032
     * 6428,6429,6430,6431,6432,6433,6434,6435,6436,6437,6438,6439,6440,6441
     * ,6442,6443, // 6048
     * 6444,6445,6446,6447,6448,6449,6450,6451,6452,6453,6454
     * ,4821,6455,6456,6457,6458, // 6064
     * 6459,6460,6461,6462,6463,6464,6465,6466
     * ,6467,6468,6469,6470,6471,6472,6473,6474, // 6080
     * 6475,6476,6477,3947,3948
     * ,6478,6479,6480,6481,3272,4441,6482,6483,6484,6485,4442, // 6096
     * 6486,6487
     * ,6488,6489,6490,6491,6492,6493,6494,6495,6496,4822,6497,6498,6499,6500,
     * // 6112
     * 6501,6502,6503,6504,6505,6506,6507,6508,6509,6510,6511,6512,6513,6514
     * ,6515,6516, // 6128
     * 6517,6518,6519,6520,6521,6522,6523,6524,6525,6526,6527
     * ,6528,6529,6530,6531,6532, // 6144
     * 6533,6534,6535,6536,6537,6538,6539,6540
     * ,6541,6542,6543,6544,6545,6546,6547,6548, // 6160
     * 6549,6550,6551,6552,6553
     * ,6554,6555,6556,2784,6557,4823,6558,6559,6560,6561,6562, // 6176
     * 6563,6564
     * ,6565,6566,6567,6568,6569,3949,6570,6571,6572,4824,6573,6574,6575,6576,
     * // 6192
     * 6577,6578,6579,6580,6581,6582,6583,4825,6584,6585,6586,3950,2785,6587
     * ,6588,6589, // 6208
     * 6590,6591,6592,6593,6594,6595,6596,6597,6598,6599,6600
     * ,6601,6602,6603,6604,6605, // 6224
     * 6606,6607,6608,6609,6610,6611,6612,4826
     * ,6613,6614,6615,4827,6616,6617,6618,6619, // 6240
     * 6620,6621,6622,6623,6624
     * ,6625,4164,6626,6627,6628,6629,6630,6631,6632,6633,6634, // 6256
     * 3547,6635
     * ,4828,6636,6637,6638,6639,6640,6641,6642,3951,2984,6643,6644,6645,6646,
     * // 6272
     * 6647,6648,6649,4165,6650,4829,6651,6652,4830,6653,6654,6655,6656,6657
     * ,6658,6659, // 6288
     * 6660,6661,6662,4831,6663,6664,6665,6666,6667,6668,6669
     * ,6670,6671,4166,6672,4832, // 6304
     * 3952,6673,6674,6675,6676,4833,6677,6678
     * ,6679,4167,6680,6681,6682,3198,6683,6684, // 6320
     * 6685,6686,6687,6688,6689
     * ,6690,6691,6692,6693,6694,6695,6696,6697,4834,6698,6699, // 6336
     * 6700,6701
     * ,6702,6703,6704,6705,6706,6707,6708,6709,6710,6711,6712,6713,6714,6715,
     * // 6352
     * 6716,6717,6718,6719,6720,6721,6722,6723,6724,6725,6726,6727,6728,6729
     * ,6730,6731, // 6368
     * 6732,6733,6734,4443,6735,6736,6737,6738,6739,6740,6741
     * ,6742,6743,6744,6745,4444, // 6384
     * 6746,6747,6748,6749,6750,6751,6752,6753
     * ,6754,6755,6756,6757,6758,6759,6760,6761, // 6400
     * 6762,6763,6764,6765,6766
     * ,6767,6768,6769,6770,6771,6772,6773,6774,6775,6776,6777, // 6416
     * 6778,6779
     * ,6780,6781,4168,6782,6783,3442,6784,6785,6786,6787,6788,6789,6790,6791,
     * // 6432
     * 4169,6792,6793,6794,6795,6796,6797,6798,6799,6800,6801,6802,6803,6804
     * ,6805,6806, // 6448
     * 6807,6808,6809,6810,6811,4835,6812,6813,6814,4445,6815
     * ,6816,4446,6817,6818,6819, // 6464
     * 6820,6821,6822,6823,6824,6825,6826,6827
     * ,6828,6829,6830,6831,6832,6833,6834,6835, // 6480
     * 3548,6836,6837,6838,6839
     * ,6840,6841,6842,6843,6844,6845,6846,4836,6847,6848,6849, // 6496
     * 6850,6851
     * ,6852,6853,6854,3953,6855,6856,6857,6858,6859,6860,6861,6862,6863,6864,
     * // 6512
     * 6865,6866,6867,6868,6869,6870,6871,6872,6873,6874,6875,6876,6877,3199
     * ,6878,6879, // 6528
     * 6880,6881,6882,4447,6883,6884,6885,6886,6887,6888,6889
     * ,6890,6891,6892,6893,6894, // 6544
     * 6895,6896,6897,6898,6899,6900,6901,6902
     * ,6903,6904,4170,6905,6906,6907,6908,6909, // 6560
     * 6910,6911,6912,6913,6914
     * ,6915,6916,6917,6918,6919,6920,6921,6922,6923,6924,6925, // 6576
     * 6926,6927
     * ,4837,6928,6929,6930,6931,6932,6933,6934,6935,6936,3346,6937,6938,4838,
     * // 6592
     * 6939,6940,6941,4448,6942,6943,6944,6945,6946,4449,6947,6948,6949,6950
     * ,6951,6952, // 6608
     * 6953,6954,6955,6956,6957,6958,6959,6960,6961,6962,6963
     * ,6964,6965,6966,6967,6968, // 6624
     * 6969,6970,6971,6972,6973,6974,6975,6976
     * ,6977,6978,6979,6980,6981,6982,6983,6984, // 6640
     * 6985,6986,6987,6988,6989
     * ,6990,6991,6992,6993,6994,3671,6995,6996,6997,6998,4839, // 6656
     * 6999,7000
     * ,7001,7002,3549,7003,7004,7005,7006,7007,7008,7009,7010,7011,7012,7013,
     * // 6672
     * 7014,7015,7016,7017,7018,7019,7020,7021,7022,7023,7024,7025,7026,7027
     * ,7028,7029, // 6688
     * 7030,4840,7031,7032,7033,7034,7035,7036,7037,7038,4841
     * ,7039,7040,7041,7042,7043, // 6704
     * 7044,7045,7046,7047,7048,7049,7050,7051
     * ,7052,7053,7054,7055,7056,7057,7058,7059, // 6720
     * 7060,7061,7062,7063,7064
     * ,7065,7066,7067,7068,7069,7070,2985,7071,7072,7073,7074, // 6736
     * 7075,7076
     * ,7077,7078,7079,7080,4842,7081,7082,7083,7084,7085,7086,7087,7088,7089,
     * // 6752
     * 7090,7091,7092,7093,7094,7095,7096,7097,7098,7099,7100,7101,7102,7103
     * ,7104,7105, // 6768
     * 7106,7107,7108,7109,7110,7111,7112,7113,7114,7115,7116
     * ,7117,7118,4450,7119,7120, // 6784
     * 7121,7122,7123,7124,7125,7126,7127,7128
     * ,7129,7130,7131,7132,7133,7134,7135,7136, // 6800
     * 7137,7138,7139,7140,7141
     * ,7142,7143,4843,7144,7145,7146,7147,7148,7149,7150,7151, // 6816
     * 7152,7153
     * ,7154,7155,7156,7157,7158,7159,7160,7161,7162,7163,7164,7165,7166,7167,
     * // 6832
     * 7168,7169,7170,7171,7172,7173,7174,7175,7176,7177,7178,7179,7180,7181
     * ,7182,7183, // 6848
     * 7184,7185,7186,7187,7188,4171,4172,7189,7190,7191,7192
     * ,7193,7194,7195,7196,7197, // 6864
     * 7198,7199,7200,7201,7202,7203,7204,7205
     * ,7206,7207,7208,7209,7210,7211,7212,7213, // 6880
     * 7214,7215,7216,7217,7218
     * ,7219,7220,7221,7222,7223,7224,7225,7226,7227,7228,7229, // 6896
     * 7230,7231
     * ,7232,7233,7234,7235,7236,7237,7238,7239,7240,7241,7242,7243,7244,7245,
     * // 6912
     * 7246,7247,7248,7249,7250,7251,7252,7253,7254,7255,7256,7257,7258,7259
     * ,7260,7261, // 6928
     * 7262,7263,7264,7265,7266,7267,7268,7269,7270,7271,7272
     * ,7273,7274,7275,7276,7277, // 6944
     * 7278,7279,7280,7281,7282,7283,7284,7285
     * ,7286,7287,7288,7289,7290,7291,7292,7293, // 6960
     * 7294,7295,7296,4844,7297
     * ,7298,7299,7300,7301,7302,7303,7304,7305,7306,7307,7308, // 6976
     * 7309,7310
     * ,7311,7312,7313,7314,7315,7316,4451,7317,7318,7319,7320,7321,7322,7323,
     * // 6992
     * 7324,7325,7326,7327,7328,7329,7330,7331,7332,7333,7334,7335,7336,7337
     * ,7338,7339, // 7008
     * 7340,7341,7342,7343,7344,7345,7346,7347,7348,7349,7350
     * ,7351,7352,7353,4173,7354, // 7024
     * 7355,4845,7356,7357,7358,7359,7360,7361
     * ,7362,7363,7364,7365,7366,7367,7368,7369, // 7040
     * 7370,7371,7372,7373,7374
     * ,7375,7376,7377,7378,7379,7380,7381,7382,7383,7384,7385, // 7056
     * 7386,7387
     * ,7388,4846,7389,7390,7391,7392,7393,7394,7395,7396,7397,7398,7399,7400,
     * // 7072
     * 7401,7402,7403,7404,7405,3672,7406,7407,7408,7409,7410,7411,7412,7413
     * ,7414,7415, // 7088
     * 7416,7417,7418,7419,7420,7421,7422,7423,7424,7425,7426
     * ,7427,7428,7429,7430,7431, // 7104
     * 7432,7433,7434,7435,7436,7437,7438,7439
     * ,7440,7441,7442,7443,7444,7445,7446,7447, // 7120
     * 7448,7449,7450,7451,7452
     * ,7453,4452,7454,3200,7455,7456,7457,7458,7459,7460,7461, // 7136
     * 7462,7463
     * ,7464,7465,7466,7467,7468,7469,7470,7471,7472,7473,7474,4847,7475,7476,
     * // 7152
     * 7477,3133,7478,7479,7480,7481,7482,7483,7484,7485,7486,7487,7488,7489
     * ,7490,7491, // 7168
     * 7492,7493,7494,7495,7496,7497,7498,7499,7500,7501,7502
     * ,3347,7503,7504,7505,7506, // 7184
     * 7507,7508,7509,7510,7511,7512,7513,7514
     * ,7515,7516,7517,7518,7519,7520,7521,4848, // 7200
     * 7522,7523,7524,7525,7526
     * ,7527,7528,7529,7530,7531,7532,7533,7534,7535,7536,7537, // 7216
     * 7538,7539
     * ,7540,7541,7542,7543,7544,7545,7546,7547,7548,7549,3801,4849,7550,7551,
     * // 7232
     * 7552,7553,7554,7555,7556,7557,7558,7559,7560,7561,7562,7563,7564,7565
     * ,7566,7567, // 7248
     * 7568,7569,3035,7570,7571,7572,7573,7574,7575,7576,7577
     * ,7578,7579,7580,7581,7582, // 7264
     * 7583,7584,7585,7586,7587,7588,7589,7590
     * ,7591,7592,7593,7594,7595,7596,7597,7598, // 7280
     * 7599,7600,7601,7602,7603
     * ,7604,7605,7606,7607,7608,7609,7610,7611,7612,7613,7614, // 7296
     * 7615,7616
     * ,4850,7617,7618,3802,7619,7620,7621,7622,7623,7624,7625,7626,7627,7628,
     * // 7312
     * 7629,7630,7631,7632,4851,7633,7634,7635,7636,7637,7638,7639,7640,7641
     * ,7642,7643, // 7328
     * 7644,7645,7646,7647,7648,7649,7650,7651,7652,7653,7654
     * ,7655,7656,7657,7658,7659, // 7344
     * 7660,7661,7662,7663,7664,7665,7666,7667
     * ,7668,7669,7670,4453,7671,7672,7673,7674, // 7360
     * 7675,7676,7677,7678,7679
     * ,7680,7681,7682,7683,7684,7685,7686,7687,7688,7689,7690, // 7376
     * 7691,7692
     * ,7693,7694,7695,7696,7697,3443,7698,7699,7700,7701,7702,4454,7703,7704,
     * // 7392
     * 7705,7706,7707,7708,7709,7710,7711,7712,7713,2472,7714,7715,7716,7717
     * ,7718,7719, // 7408
     * 7720,7721,7722,7723,7724,7725,7726,7727,7728,7729,7730
     * ,7731,3954,7732,7733,7734, // 7424
     * 7735,7736,7737,7738,7739,7740,7741,7742
     * ,7743,7744,7745,7746,7747,7748,7749,7750, // 7440
     * 3134,7751,7752,4852,7753
     * ,7754,7755,4853,7756,7757,7758,7759,7760,4174,7761,7762, // 7456
     * 7763,7764
     * ,7765,7766,7767,7768,7769,7770,7771,7772,7773,7774,7775,7776,7777,7778,
     * // 7472
     * 7779,7780,7781,7782,7783,7784,7785,7786,7787,7788,7789,7790,7791,7792
     * ,7793,7794, // 7488
     * 7795,7796,7797,7798,7799,7800,7801,7802,7803,7804,7805
     * ,4854,7806,7807,7808,7809, // 7504
     * 7810,7811,7812,7813,7814,7815,7816,7817
     * ,7818,7819,7820,7821,7822,7823,7824,7825, // 7520
     * 4855,7826,7827,7828,7829
     * ,7830,7831,7832,7833,7834,7835,7836,7837,7838,7839,7840, // 7536
     * 7841,7842
     * ,7843,7844,7845,7846,7847,3955,7848,7849,7850,7851,7852,7853,7854,7855,
     * // 7552
     * 7856,7857,7858,7859,7860,3444,7861,7862,7863,7864,7865,7866,7867,7868
     * ,7869,7870, // 7568
     * 7871,7872,7873,7874,7875,7876,7877,7878,7879,7880,7881
     * ,7882,7883,7884,7885,7886, // 7584
     * 7887,7888,7889,7890,7891,4175,7892,7893
     * ,7894,7895,7896,4856,4857,7897,7898,7899, // 7600
     * 7900,2598,7901,7902,7903
     * ,7904,7905,7906,7907,7908,4455,7909,7910,7911,7912,7913, // 7616
     * 7914,3201
     * ,7915,7916,7917,7918,7919,7920,7921,4858,7922,7923,7924,7925,7926,7927,
     * // 7632
     * 7928,7929,7930,7931,7932,7933,7934,7935,7936,7937,7938,7939,7940,7941
     * ,7942,7943, // 7648
     * 7944,7945,7946,7947,7948,7949,7950,7951,7952,7953,7954
     * ,7955,7956,7957,7958,7959, // 7664
     * 7960,7961,7962,7963,7964,7965,7966,7967
     * ,7968,7969,7970,7971,7972,7973,7974,7975, // 7680
     * 7976,7977,7978,7979,7980
     * ,7981,4859,7982,7983,7984,7985,7986,7987,7988,7989,7990, // 7696
     * 7991,7992
     * ,7993,7994,7995,7996,4860,7997,7998,7999,8000,8001,8002,8003,8004,8005,
     * // 7712
     * 8006,8007,8008,8009,8010,8011,8012,8013,8014,8015,8016,4176,8017,8018
     * ,8019,8020, // 7728
     * 8021,8022,8023,4861,8024,8025,8026,8027,8028,8029,8030
     * ,8031,8032,8033,8034,8035, // 7744
     * 8036,4862,4456,8037,8038,8039,8040,4863
     * ,8041,8042,8043,8044,8045,8046,8047,8048, // 7760
     * 8049,8050,8051,8052,8053
     * ,8054,8055,8056,8057,8058,8059,8060,8061,8062,8063,8064, // 7776
     * 8065,8066
     * ,8067,8068,8069,8070,8071,8072,8073,8074,8075,8076,8077,8078,8079,8080,
     * // 7792
     * 8081,8082,8083,8084,8085,8086,8087,8088,8089,8090,8091,8092,8093,8094
     * ,8095,8096, // 7808
     * 8097,8098,8099,4864,4177,8100,8101,8102,8103,8104,8105
     * ,8106,8107,8108,8109,8110, // 7824
     * 8111,8112,8113,8114,8115,8116,8117,8118
     * ,8119,8120,4178,8121,8122,8123,8124,8125, // 7840
     * 8126,8127,8128,8129,8130
     * ,8131,8132,8133,8134,8135,8136,8137,8138,8139,8140,8141, // 7856
     * 8142,8143
     * ,8144,8145,4865,4866,8146,8147,8148,8149,8150,8151,8152,8153,8154,8155,
     * // 7872
     * 8156,8157,8158,8159,8160,8161,8162,8163,8164,8165,4179,8166,8167,8168
     * ,8169,8170, // 7888
     * 8171,8172,8173,8174,8175,8176,8177,8178,8179,8180,8181
     * ,4457,8182,8183,8184,8185, // 7904
     * 8186,8187,8188,8189,8190,8191,8192,8193
     * ,8194,8195,8196,8197,8198,8199,8200,8201, // 7920
     * 8202,8203,8204,8205,8206
     * ,8207,8208,8209,8210,8211,8212,8213,8214,8215,8216,8217, // 7936
     * 8218,8219
     * ,8220,8221,8222,8223,8224,8225,8226,8227,8228,8229,8230,8231,8232,8233,
     * // 7952
     * 8234,8235,8236,8237,8238,8239,8240,8241,8242,8243,8244,8245,8246,8247
     * ,8248,8249, // 7968
     * 8250,8251,8252,8253,8254,8255,8256,3445,8257,8258,8259
     * ,8260,8261,8262,4458,8263, // 7984
     * 8264,8265,8266,8267,8268,8269,8270,8271
     * ,8272,4459,8273,8274,8275,8276,3550,8277, // 8000
     * 8278,8279,8280,8281,8282
     * ,8283,8284,8285,8286,8287,8288,8289,4460,8290,8291,8292, // 8016
     * 8293,8294
     * ,8295,8296,8297,8298,8299,8300,8301,8302,8303,8304,8305,8306,8307,4867,
     * // 8032
     * 8308,8309,8310,8311,8312,3551,8313,8314,8315,8316,8317,8318,8319,8320
     * ,8321,8322, // 8048
     * 8323,8324,8325,8326,4868,8327,8328,8329,8330,8331,8332
     * ,8333,8334,8335,8336,8337, // 8064
     * 8338,8339,8340,8341,8342,8343,8344,8345
     * ,8346,8347,8348,8349,8350,8351,8352,8353, // 8080
     * 8354,8355,8356,8357,8358
     * ,8359,8360,8361,8362,8363,4869,4461,8364,8365,8366,8367, // 8096
     * 8368,8369
     * ,8370,4870,8371,8372,8373,8374,8375,8376,8377,8378,8379,8380,8381,8382,
     * // 8112
     * 8383,8384,8385,8386,8387,8388,8389,8390,8391,8392,8393,8394,8395,8396
     * ,8397,8398, // 8128
     * 8399,8400,8401,8402,8403,8404,8405,8406,8407,8408,8409
     * ,8410,4871,8411,8412,8413, // 8144
     * 8414,8415,8416,8417,8418,8419,8420,8421
     * ,8422,4462,8423,8424,8425,8426,8427,8428, // 8160
     * 8429,8430,8431,8432,8433
     * ,2986,8434,8435,8436,8437,8438,8439,8440,8441,8442,8443, // 8176
     * 8444,8445
     * ,8446,8447,8448,8449,8450,8451,8452,8453,8454,8455,8456,8457,8458,8459,
     * // 8192
     * 8460,8461,8462,8463,8464,8465,8466,8467,8468,8469,8470,8471,8472,8473
     * ,8474,8475, // 8208
     * 8476,8477,8478,4180,8479,8480,8481,8482,8483,8484,8485
     * ,8486,8487,8488,8489,8490, // 8224
     * 8491,8492,8493,8494,8495,8496,8497,8498
     * ,8499,8500,8501,8502,8503,8504,8505,8506, // 8240
     * 8507,8508,8509,8510,8511
     * ,8512,8513,8514,8515,8516,8517,8518,8519,8520,8521,8522, // 8256
     * 8523,8524
     * ,8525,8526,8527,8528,8529,8530,8531,8532,8533,8534,8535,8536,8537,8538,
     * // 8272
     * 8539,8540,8541,8542,8543,8544,8545,8546,8547,8548,8549,8550,8551,8552
     * ,8553,8554, // 8288
     * 8555,8556,8557,8558,8559,8560,8561,8562,8563,8564,4872
     * ,8565,8566,8567,8568,8569, // 8304
     * 8570,8571,8572,8573,4873,8574,8575,8576
     * ,8577,8578,8579,8580,8581,8582,8583,8584, // 8320
     * 8585,8586,8587,8588,8589
     * ,8590,8591,8592,8593,8594,8595,8596,8597,8598,8599,8600, // 8336
     * 8601,8602
     * ,8603,8604,8605,3803,8606,8607,8608,8609,8610,8611,8612,8613,4874,3804,
     * // 8352
     * 8614,8615,8616,8617,8618,8619,8620,8621,3956,8622,8623,8624,8625,8626
     * ,8627,8628, // 8368
     * 8629,8630,8631,8632,8633,8634,8635,8636,8637,8638,2865
     * ,8639,8640,8641,8642,8643, // 8384
     * 8644,8645,8646,8647,8648,8649,8650,8651
     * ,8652,8653,8654,8655,8656,4463,8657,8658, // 8400
     * 8659,4875,4876,8660,8661
     * ,8662,8663,8664,8665,8666,8667,8668,8669,8670,8671,8672, // 8416
     * 8673,8674
     * ,8675,8676,8677,8678,8679,8680,8681,4464,8682,8683,8684,8685,8686,8687,
     * // 8432
     * 8688,8689,8690,8691,8692,8693,8694,8695,8696,8697,8698,8699,8700,8701
     * ,8702,8703, // 8448
     * 8704,8705,8706,8707,8708,8709,2261,8710,8711,8712,8713
     * ,8714,8715,8716,8717,8718, // 8464
     * 8719,8720,8721,8722,8723,8724,8725,8726
     * ,8727,8728,8729,8730,8731,8732,8733,4181, // 8480
     * 8734,8735,8736,8737,8738
     * ,8739,8740,8741,8742,8743,8744,8745,8746,8747,8748,8749, // 8496
     * 8750,8751
     * ,8752,8753,8754,8755,8756,8757,8758,8759,8760,8761,8762,8763,4877,8764,
     * // 8512
     * 8765,8766,8767,8768,8769,8770,8771,8772,8773,8774,8775,8776,8777,8778
     * ,8779,8780, // 8528
     * 8781,8782,8783,8784,8785,8786,8787,8788,4878,8789,4879
     * ,8790,8791,8792,4880,8793, // 8544
     * 8794,8795,8796,8797,8798,8799,8800,8801
     * ,4881,8802,8803,8804,8805,8806,8807,8808, // 8560
     * 8809,8810,8811,8812,8813
     * ,8814,8815,3957,8816,8817,8818,8819,8820,8821,8822,8823, // 8576
     * 8824,8825
     * ,8826,8827,8828,8829,8830,8831,8832,8833,8834,8835,8836,8837,8838,8839,
     * // 8592
     * 8840,8841,8842,8843,8844,8845,8846,8847,4882,8848,8849,8850,8851,8852
     * ,8853,8854, // 8608
     * 8855,8856,8857,8858,8859,8860,8861,8862,8863,8864,8865
     * ,8866,8867,8868,8869,8870, // 8624
     * 8871,8872,8873,8874,8875,8876,8877,8878
     * ,8879,8880,8881,8882,8883,8884,3202,8885, // 8640
     * 8886,8887,8888,8889,8890
     * ,8891,8892,8893,8894,8895,8896,8897,8898,8899,8900,8901, // 8656
     * 8902,8903
     * ,8904,8905,8906,8907,8908,8909,8910,8911,8912,8913,8914,8915,8916,8917,
     * // 8672
     * 8918,8919,8920,8921,8922,8923,8924,4465,8925,8926,8927,8928,8929,8930
     * ,8931,8932, // 8688
     * 4883,8933,8934,8935,8936,8937,8938,8939,8940,8941,8942
     * ,8943,2214,8944,8945,8946, // 8704
     * 8947,8948,8949,8950,8951,8952,8953,8954
     * ,8955,8956,8957,8958,8959,8960,8961,8962, // 8720
     * 8963,8964,8965,4884,8966
     * ,8967,8968,8969,8970,8971,8972,8973,8974,8975,8976,8977, // 8736
     * 8978,8979
     * ,8980,8981,8982,8983,8984,8985,8986,8987,8988,8989,8990,8991,8992,4885,
     * // 8752
     * 8993,8994,8995,8996,8997,8998,8999,9000,9001,9002,9003,9004,9005,9006
     * ,9007,9008, // 8768
     * 9009,9010,9011,9012,9013,9014,9015,9016,9017,9018,9019
     * ,9020,9021,4182,9022,9023, // 8784
     * 9024,9025,9026,9027,9028,9029,9030,9031
     * ,9032,9033,9034,9035,9036,9037,9038,9039, // 8800
     * 9040,9041,9042,9043,9044
     * ,9045,9046,9047,9048,9049,9050,9051,9052,9053,9054,9055, // 8816
     * 9056,9057
     * ,9058,9059,9060,9061,9062,9063,4886,9064,9065,9066,9067,9068,9069,4887,
     * // 8832
     * 9070,9071,9072,9073,9074,9075,9076,9077,9078,9079,9080,9081,9082,9083
     * ,9084,9085, // 8848
     * 9086,9087,9088,9089,9090,9091,9092,9093,9094,9095,9096
     * ,9097,9098,9099,9100,9101, // 8864
     * 9102,9103,9104,9105,9106,9107,9108,9109
     * ,9110,9111,9112,9113,9114,9115,9116,9117, // 8880
     * 9118,9119,9120,9121,9122
     * ,9123,9124,9125,9126,9127,9128,9129,9130,9131,9132,9133, // 8896
     * 9134,9135
     * ,9136,9137,9138,9139,9140,9141,3958,9142,9143,9144,9145,9146,9147,9148,
     * // 8912
     * 9149,9150,9151,4888,9152,9153,9154,9155,9156,9157,9158,9159,9160,9161
     * ,9162,9163, // 8928
     * 9164,9165,9166,9167,9168,9169,9170,9171,9172,9173,9174
     * ,9175,4889,9176,9177,9178, // 8944
     * 9179,9180,9181,9182,9183,9184,9185,9186
     * ,9187,9188,9189,9190,9191,9192,9193,9194, // 8960
     * 9195,9196,9197,9198,9199
     * ,9200,9201,9202,9203,4890,9204,9205,9206,9207,9208,9209, // 8976
     * 9210,9211
     * ,9212,9213,9214,9215,9216,9217,9218,9219,9220,9221,9222,4466,9223,9224,
     * // 8992
     * 9225,9226,9227,9228,9229,9230,9231,9232,9233,9234,9235,9236,9237,9238
     * ,9239,9240, // 9008
     * 9241,9242,9243,9244,9245,4891,9246,9247,9248,9249,9250
     * ,9251,9252,9253,9254,9255, // 9024
     * 9256,9257,4892,9258,9259,9260,9261,4893
     * ,4894,9262,9263,9264,9265,9266,9267,9268, // 9040
     * 9269,9270,9271,9272,9273
     * ,4467,9274,9275,9276,9277,9278,9279,9280,9281,9282,9283, // 9056
     * 9284,9285
     * ,3673,9286,9287,9288,9289,9290,9291,9292,9293,9294,9295,9296,9297,9298,
     * // 9072
     * 9299,9300,9301,9302,9303,9304,9305,9306,9307,9308,9309,9310,9311,9312
     * ,9313,9314, // 9088
     * 9315,9316,9317,9318,9319,9320,9321,9322,4895,9323,9324
     * ,9325,9326,9327,9328,9329, // 9104
     * 9330,9331,9332,9333,9334,9335,9336,9337
     * ,9338,9339,9340,9341,9342,9343,9344,9345, // 9120
     * 9346,9347,4468,9348,9349
     * ,9350,9351,9352,9353,9354,9355,9356,9357,9358,9359,9360, // 9136
     * 9361,9362
     * ,9363,9364,9365,9366,9367,9368,9369,9370,9371,9372,9373,4896,9374,4469,
     * // 9152
     * 9375,9376,9377,9378,9379,4897,9380,9381,9382,9383,9384,9385,9386,9387
     * ,9388,9389, // 9168
     * 9390,9391,9392,9393,9394,9395,9396,9397,9398,9399,9400
     * ,9401,9402,9403,9404,9405, // 9184
     * 9406,4470,9407,2751,9408,9409,3674,3552
     * ,9410,9411,9412,9413,9414,9415,9416,9417, // 9200
     * 9418,9419,9420,9421,4898
     * ,9422,9423,9424,9425,9426,9427,9428,9429,3959,9430,9431, // 9216
     * 9432,9433
     * ,9434,9435,9436,4471,9437,9438,9439,9440,9441,9442,9443,9444,9445,9446,
     * // 9232
     * 9447,9448,9449,9450,3348,9451,9452,9453,9454,9455,9456,9457,9458,9459
     * ,9460,9461, // 9248
     * 9462,9463,9464,9465,9466,9467,9468,9469,9470,9471,9472
     * ,4899,9473,9474,9475,9476, // 9264
     * 9477,4900,9478,9479,9480,9481,9482,9483
     * ,9484,9485,9486,9487,9488,3349,9489,9490, // 9280
     * 9491,9492,9493,9494,9495
     * ,9496,9497,9498,9499,9500,9501,9502,9503,9504,9505,9506, // 9296
     * 9507,9508
     * ,9509,9510,9511,9512,9513,9514,9515,9516,9517,9518,9519,9520,4901,9521,
     * // 9312
     * 9522,9523,9524,9525,9526,4902,9527,9528,9529,9530,9531,9532,9533,9534
     * ,9535,9536, // 9328
     * 9537,9538,9539,9540,9541,9542,9543,9544,9545,9546,9547
     * ,9548,9549,9550,9551,9552, // 9344
     * 9553,9554,9555,9556,9557,9558,9559,9560
     * ,9561,9562,9563,9564,9565,9566,9567,9568, // 9360
     * 9569,9570,9571,9572,9573
     * ,9574,9575,9576,9577,9578,9579,9580,9581,9582,9583,9584, // 9376
     * 3805,9585
     * ,9586,9587,9588,9589,9590,9591,9592,9593,9594,9595,9596,9597,9598,9599,
     * // 9392
     * 9600,9601,9602,4903,9603,9604,9605,9606,9607,4904,9608,9609,9610,9611
     * ,9612,9613, // 9408
     * 9614,4905,9615,9616,9617,9618,9619,9620,9621,9622,9623
     * ,9624,9625,9626,9627,9628, // 9424
     * 9629,9630,9631,9632,4906,9633,9634,9635
     * ,9636,9637,9638,9639,9640,9641,9642,9643, // 9440
     * 4907,9644,9645,9646,9647
     * ,9648,9649,9650,9651,9652,9653,9654,9655,9656,9657,9658, // 9456
     * 9659,9660
     * ,9661,9662,9663,9664,9665,9666,9667,9668,9669,9670,9671,9672,4183,9673,
     * // 9472
     * 9674,9675,9676,9677,4908,9678,9679,9680,9681,4909,9682,9683,9684,9685
     * ,9686,9687, // 9488
     * 9688,9689,9690,4910,9691,9692,9693,3675,9694,9695,9696
     * ,2945,9697,9698,9699,9700, // 9504
     * 9701,9702,9703,9704,9705,4911,9706,9707
     * ,9708,9709,9710,9711,9712,9713,9714,9715, // 9520
     * 9716,9717,9718,9719,9720
     * ,9721,9722,9723,9724,9725,9726,9727,9728,9729,9730,9731, // 9536
     * 9732,9733
     * ,9734,9735,4912,9736,9737,9738,9739,9740,4913,9741,9742,9743,9744,9745,
     * // 9552
     * 9746,9747,9748,9749,9750,9751,9752,9753,9754,9755,9756,9757,9758,4914
     * ,9759,9760, // 9568
     * 9761,9762,9763,9764,9765,9766,9767,9768,9769,9770,9771
     * ,9772,9773,9774,9775,9776, // 9584
     * 9777,9778,9779,9780,9781,9782,4915,9783
     * ,9784,9785,9786,9787,9788,9789,9790,9791, // 9600
     * 9792,9793,4916,9794,9795
     * ,9796,9797,9798,9799,9800,9801,9802,9803,9804,9805,9806, // 9616
     * 9807,9808
     * ,9809,9810,9811,9812,9813,9814,9815,9816,9817,9818,9819,9820,9821,9822,
     * // 9632
     * 9823,9824,9825,9826,9827,9828,9829,9830,9831,9832,9833,9834,9835,9836
     * ,9837,9838, // 9648
     * 9839,9840,9841,9842,9843,9844,9845,9846,9847,9848,9849
     * ,9850,9851,9852,9853,9854, // 9664
     * 9855,9856,9857,9858,9859,9860,9861,9862
     * ,9863,9864,9865,9866,9867,9868,4917,9869, // 9680
     * 9870,9871,9872,9873,9874
     * ,9875,9876,9877,9878,9879,9880,9881,9882,9883,9884,9885, // 9696
     * 9886,9887
     * ,9888,9889,9890,9891,9892,4472,9893,9894,9895,9896,9897,3806,9898,9899,
     * // 9712
     * 9900,9901,9902,9903,9904,9905,9906,9907,9908,9909,9910,9911,9912,9913
     * ,9914,4918, // 9728
     * 9915,9916,9917,4919,9918,9919,9920,9921,4184,9922,9923
     * ,9924,9925,9926,9927,9928, // 9744
     * 9929,9930,9931,9932,9933,9934,9935,9936
     * ,9937,9938,9939,9940,9941,9942,9943,9944, // 9760
     * 9945,9946,4920,9947,9948
     * ,9949,9950,9951,9952,9953,9954,9955,4185,9956,9957,9958, // 9776
     * 9959,9960
     * ,9961,9962,9963,9964,9965,4921,9966,9967,9968,4473,9969,9970,9971,9972,
     * // 9792
     * 9973,9974,9975,9976,9977,4474,9978,9979,9980,9981,9982,9983,9984,9985
     * ,9986,9987, // 9808
     * 9988,9989,9990,9991,9992,9993,9994,9995,9996,9997,9998
     * ,9999,10000,10001,10002,10003, // 9824
     * 10004,10005,10006,10007,10008,10009
     * ,10010,10011,10012,10013,10014,10015,10016,10017,10018,10019, // 9840
     * 10020
     * ,10021,4922,10022,4923,10023,10024,10025,10026,10027,10028,10029,10030
     * ,10031,10032,10033, // 9856
     * 10034,10035,10036,10037,10038,10039,10040,10041
     * ,10042,10043,10044,10045,10046,10047,10048,4924, // 9872
     * 10049,10050,10051
     * ,10052,10053,10054,10055,10056,10057,10058,10059,10060,10061
     * ,10062,10063,10064, // 9888
     * 10065,10066,10067,10068,10069,10070,10071,10072
     * ,10073,10074,10075,10076,10077,10078,10079,10080, // 9904
     * 10081,10082,10083
     * ,10084,10085,10086,10087,4475,10088,10089,10090,10091,10092
     * ,10093,10094,10095, // 9920
     * 10096,10097,4476,10098,10099,10100,10101,10102
     * ,10103,10104,10105,10106,10107,10108,10109,10110, // 9936
     * 10111,2174,10112
     * ,10113,10114,10115,10116,10117,10118,10119,10120,10121,10122
     * ,10123,10124,10125, // 9952
     * 10126,10127,10128,10129,10130,10131,10132,10133
     * ,10134,10135,10136,10137,10138,10139,10140,3807, // 9968
     * 4186,4925,10141,10142
     * ,10143,10144,10145,10146,10147,4477,4187,10148,10149,10150,10151,10152,
     * // 9984
     * 10153,4188,10154,10155,10156,10157,10158,10159,10160,10161,4926,10162
     * ,10163,10164,10165,10166, //10000
     * 10167,10168,10169,10170,10171,10172,10173
     * ,10174,10175,10176,10177,10178,10179,10180,10181,10182, //10016
     * 10183,10184
     * ,10185,10186,10187,10188,10189,10190,10191,10192,3203,10193,10194
     * ,10195,10196,10197, //10032
     * 10198,10199,10200,4478,10201,10202,10203,10204
     * ,4479,10205,10206,10207,10208,10209,10210,10211, //10048
     * 10212,10213,10214
     * ,10215,10216,10217,10218,10219,10220,10221,10222,10223,10224
     * ,10225,10226,10227, //10064
     * 10228,10229,10230,10231,10232,10233,10234,4927
     * ,10235,10236,10237,10238,10239,10240,10241,10242, //10080
     * 10243,10244,10245
     * ,10246,10247,10248,10249,10250,10251,10252,10253,10254,10255
     * ,10256,10257,10258, //10096
     * 10259,10260,10261,10262,10263,10264,10265,10266
     * ,10267,10268,10269,10270,10271,10272,10273,4480, //10112
     * 4928,4929,10274,10275
     * ,10276,10277,10278,10279,10280,10281,10282,10283,10284,10285,10286,10287,
     * //10128
     * 10288,10289,10290,10291,10292,10293,10294,10295,10296,10297,10298,
     * 10299,10300,10301,10302,10303, //10144
     * 10304,10305,10306,10307,10308,10309
     * ,10310,10311,10312,10313,10314,10315,10316,10317,10318,10319, //10160
     * 10320
     * ,10321,10322,10323,10324,10325,10326,10327,10328,10329,10330,10331,10332
     * ,10333,10334,4930, //10176
     * 10335,10336,10337,10338,10339,10340,10341,10342
     * ,4931,10343,10344,10345,10346,10347,10348,10349, //10192
     * 10350,10351,10352
     * ,10353,10354,10355,3088,10356,2786,10357,10358,10359,10360
     * ,4189,10361,10362, //10208
     * 10363,10364,10365,10366,10367,10368,10369,10370
     * ,10371,10372,10373,10374,10375,4932,10376,10377, //10224
     * 10378,10379,10380
     * ,10381,10382,10383,10384,10385,10386,10387,10388,10389,10390
     * ,10391,10392,4933, //10240
     * 10393,10394,10395,4934,10396,10397,10398,10399,
     * 10400,10401,10402,10403,10404,10405,10406,10407, //10256
     * 10408,10409,10410
     * ,10411,10412,3446,10413,10414,10415,10416,10417,10418,10419
     * ,10420,10421,10422, //10272
     * 10423,4935,10424,10425,10426,10427,10428,10429
     * ,10430,4936,10431,10432,10433,10434,10435,10436, //10288
     * 10437,10438,10439
     * ,10440,10441,10442,10443,4937,10444,10445,10446,10447,4481
     * ,10448,10449,10450, //10304
     * 10451,10452,10453,10454,10455,10456,10457,10458
     * ,10459,10460,10461,10462,10463,10464,10465,10466, //10320
     * 10467,10468,10469
     * ,10470,10471,10472,10473,10474,10475,10476,10477,10478,10479
     * ,10480,10481,10482, //10336
     * 10483,10484,10485,10486,10487,10488,10489,10490
     * ,10491,10492,10493,10494,10495,10496,10497,10498, //10352
     * 10499,10500,10501
     * ,10502,10503,10504,10505,4938,10506,10507,10508,10509,10510
     * ,2552,10511,10512, //10368
     * 10513,10514,10515,10516,3447,10517,10518,10519,
     * 10520,10521,10522,10523,10524,10525,10526,10527, //10384
     * 10528,10529,10530
     * ,10531,10532,10533,10534,10535,10536,10537,10538,10539,10540
     * ,10541,10542,10543, //10400
     * 4482,10544,4939,10545,10546,10547,10548,10549,
     * 10550,10551,10552,10553,10554,10555,10556,10557, //10416
     * 10558,10559,10560
     * ,10561,10562,10563,10564,10565,10566,10567,3676,4483,10568
     * ,10569,10570,10571, //10432
     * 10572,3448,10573,10574,10575,10576,10577,10578
     * ,10579,10580,10581,10582,10583,10584,10585,10586, //10448
     * 10587,10588,10589
     * ,10590,10591,10592,10593,10594,10595,10596,10597,10598,10599
     * ,10600,10601,10602, //10464
     * 10603,10604,10605,10606,10607,10608,10609,10610
     * ,10611,10612,10613,10614,10615,10616,10617,10618, //10480
     * 10619,10620,10621
     * ,10622,10623,10624,10625,10626,10627,4484,10628,10629,10630
     * ,10631,10632,4940, //10496
     * 10633,10634,10635,10636,10637,10638,10639,10640
     * ,10641,10642,10643,10644,10645,10646,10647,10648, //10512
     * 10649,10650,10651
     * ,10652,10653,10654,10655,10656,4941,10657,10658,10659,2599
     * ,10660,10661,10662, //10528
     * 10663,10664,10665,10666,3089,10667,10668,10669
     * ,10670,10671,10672,10673,10674,10675,10676,10677, //10544
     * 10678,10679,10680
     * ,4942,10681,10682,10683,10684,10685,10686,10687,10688,10689
     * ,10690,10691,10692, //10560
     * 10693,10694,10695,10696,10697,4485,10698,10699
     * ,10700,10701,10702,10703,10704,4943,10705,3677, //10576
     * 10706,10707,10708,
     * 10709,10710,10711,10712,4944,10713,10714,10715,10716,10717
     * ,10718,10719,10720, //10592
     * 10721,10722,10723,10724,10725,10726,10727,10728
     * ,4945,10729,10730,10731,10732,10733,10734,10735, //10608
     * 10736,10737,10738
     * ,10739,10740,10741,10742,10743,10744,10745,10746,10747,10748
     * ,10749,10750,10751, //10624
     * 10752,10753,10754,10755,10756,10757,10758,10759
     * ,10760,10761,4946,10762,10763,10764,10765,10766, //10640
     * 10767,4947,4948,10768
     * ,10769,10770,10771,10772,10773,10774,10775,10776,10777,10778,10779,10780,
     * //10656
     * 10781,10782,10783,10784,10785,10786,10787,10788,10789,10790,10791,
     * 10792,10793,10794,10795,10796, //10672
     * 10797,10798,10799,10800,10801,10802
     * ,10803,10804,10805,10806,10807,10808,10809,10810,10811,10812, //10688
     * 10813
     * ,10814,10815,10816,10817,10818,10819,10820,10821,10822,10823,10824,10825
     * ,10826,10827,10828, //10704
     * 10829,10830,10831,10832,10833,10834,10835,10836
     * ,10837,10838,10839,10840,10841,10842,10843,10844, //10720
     * 10845,10846,10847
     * ,10848,10849,10850,10851,10852,10853,10854,10855,10856,10857
     * ,10858,10859,10860, //10736
     * 10861,10862,10863,10864,10865,10866,10867,10868
     * ,10869,10870,10871,10872,10873,10874,10875,10876, //10752
     * 10877,10878,4486
     * ,10879,10880,10881,10882,10883,10884,10885,4949,10886,10887
     * ,10888,10889,10890, //10768
     * 10891,10892,10893,10894,10895,10896,10897,10898
     * ,10899,10900,10901,10902,10903,10904,10905,10906, //10784
     * 10907,10908,10909
     * ,10910,10911,10912,10913,10914,10915,10916,10917,10918,10919
     * ,4487,10920,10921, //10800
     * 10922,10923,10924,10925,10926,10927,10928,10929
     * ,10930,10931,10932,4950,10933,10934,10935,10936, //10816
     * 10937,10938,10939
     * ,10940,10941,10942,10943,10944,10945,10946,10947,10948,10949
     * ,4488,10950,10951, //10832
     * 10952,10953,10954,10955,10956,10957,10958,10959
     * ,4190,10960,10961,10962,10963,10964,10965,10966, //10848
     * 10967,10968,10969
     * ,10970,10971,10972,10973,10974,10975,10976,10977,10978,10979
     * ,10980,10981,10982, //10864
     * 10983,10984,10985,10986,10987,10988,10989,10990
     * ,10991,10992,10993,10994,10995,10996,10997,10998, //10880
     * 10999,11000,11001
     * ,11002,11003,11004,11005,11006,3960,11007,11008,11009,11010
     * ,11011,11012,11013, //10896
     * 11014,11015,11016,11017,11018,11019,11020,11021
     * ,11022,11023,11024,11025,11026,11027,11028,11029, //10912
     * 11030,11031,11032
     * ,4951,11033,11034,11035,11036,11037,11038,11039,11040,11041
     * ,11042,11043,11044, //10928
     * 11045,11046,11047,4489,11048,11049,11050,11051
     * ,4952,11052,11053,11054,11055,11056,11057,11058, //10944
     * 4953,11059,11060,
     * 11061,11062,11063,11064,11065,11066,11067,11068,11069,11070
     * ,11071,4954,11072, //10960
     * 11073,11074,11075,11076,11077,11078,11079,11080
     * ,11081,11082,11083,11084,11085,11086,11087,11088, //10976
     * 11089,11090,11091
     * ,11092,11093,11094,11095,11096,11097,11098,11099,11100,11101
     * ,11102,11103,11104, //10992
     * 11105,11106,11107,11108,11109,11110,11111,11112
     * ,11113,11114,11115,3808,11116,11117,11118,11119, //11008
     * 11120,11121,11122
     * ,11123,11124,11125,11126,11127,11128,11129,11130,11131,11132
     * ,11133,11134,4955, //11024
     * 11135,11136,11137,11138,11139,11140,11141,11142
     * ,11143,11144,11145,11146,11147,11148,11149,11150, //11040
     * 11151,11152,11153
     * ,11154,11155,11156,11157,11158,11159,11160,11161,4956,11162
     * ,11163,11164,11165, //11056
     * 11166,11167,11168,11169,11170,11171,11172,11173
     * ,11174,11175,11176,11177,11178,11179,11180,4957, //11072
     * 11181,11182,11183
     * ,11184,11185,11186,4958,11187,11188,11189,11190,11191,11192
     * ,11193,11194,11195, //11088
     * 11196,11197,11198,11199,11200,3678,11201,11202
     * ,11203,11204,11205,11206,4191,11207,11208,11209, //11104
     * 11210,11211,11212
     * ,11213,11214,11215,11216,11217,11218,11219,11220,11221,11222
     * ,11223,11224,11225, //11120
     * 11226,11227,11228,11229,11230,11231,11232,11233
     * ,11234,11235,11236,11237,11238,11239,11240,11241, //11136
     * 11242,11243,11244
     * ,11245,11246,11247,11248,11249,11250,11251,4959,11252,11253
     * ,11254,11255,11256, //11152
     * 11257,11258,11259,11260,11261,11262,11263,11264
     * ,11265,11266,11267,11268,11269,11270,11271,11272, //11168
     * 11273,11274,11275
     * ,11276,11277,11278,11279,11280,11281,11282,11283,11284,11285
     * ,11286,11287,11288, //11184
     * 11289,11290,11291,11292,11293,11294,11295,11296
     * ,11297,11298,11299,11300,11301,11302,11303,11304, //11200
     * 11305,11306,11307
     * ,11308,11309,11310,11311,11312,11313,11314,3679,11315,11316
     * ,11317,11318,4490, //11216
     * 11319,11320,11321,11322,11323,11324,11325,11326
     * ,11327,11328,11329,11330,11331,11332,11333,11334, //11232
     * 11335,11336,11337
     * ,11338,11339,11340,11341,11342,11343,11344,11345,11346,11347
     * ,4960,11348,11349, //11248
     * 11350,11351,11352,11353,11354,11355,11356,11357
     * ,11358,11359,11360,11361,11362,11363,11364,11365, //11264
     * 11366,11367,11368
     * ,11369,11370,11371,11372,11373,11374,11375,11376,11377,3961
     * ,4961,11378,11379, //11280
     * 11380,11381,11382,11383,11384,11385,11386,11387
     * ,11388,11389,11390,11391,11392,11393,11394,11395, //11296
     * 11396,11397,4192
     * ,11398,11399,11400,11401,11402,11403,11404,11405,11406,11407
     * ,11408,11409,11410, //11312
     * 11411,4962,11412,11413,11414,11415,11416,11417
     * ,11418,11419,11420,11421,11422,11423,11424,11425, //11328
     * 11426,11427,11428
     * ,11429,11430,11431,11432,11433,11434,11435,11436,11437,11438
     * ,11439,11440,11441, //11344
     * 11442,11443,11444,11445,11446,11447,11448,11449
     * ,11450,11451,11452,11453,11454,11455,11456,11457, //11360
     * 11458,11459,11460
     * ,11461,11462,11463,11464,11465,11466,11467,11468,11469,4963
     * ,11470,11471,4491, //11376
     * 11472,11473,11474,11475,4964,11476,11477,11478,
     * 11479,11480,11481,11482,11483,11484,11485,11486, //11392
     * 11487,11488,11489
     * ,11490,11491,11492,4965,11493,11494,11495,11496,11497,11498
     * ,11499,11500,11501, //11408
     * 11502,11503,11504,11505,11506,11507,11508,11509
     * ,11510,11511,11512,11513,11514,11515,11516,11517, //11424
     * 11518,11519,11520
     * ,11521,11522,11523,11524,11525,11526,11527,11528,11529,3962
     * ,11530,11531,11532, //11440
     * 11533,11534,11535,11536,11537,11538,11539,11540
     * ,11541,11542,11543,11544,11545,11546,11547,11548, //11456
     * 11549,11550,11551
     * ,11552,11553,11554,11555,11556,11557,11558,11559,11560,11561
     * ,11562,11563,11564, //11472
     * 4193,4194,11565,11566,11567,11568,11569,11570,
     * 11571,11572,11573,11574,11575,11576,11577,11578, //11488
     * 11579,11580,11581
     * ,11582,11583,11584,11585,11586,11587,11588,11589,11590,11591
     * ,4966,4195,11592, //11504
     * 11593,11594,11595,11596,11597,11598,11599,11600,
     * 11601,11602,11603,11604,3090,11605,11606,11607, //11520
     * 11608,11609,11610,
     * 4967,11611,11612,11613,11614,11615,11616,11617,11618,11619
     * ,11620,11621,11622, //11536
     * 11623,11624,11625,11626,11627,11628,11629,11630
     * ,11631,11632,11633,11634,11635,11636,11637,11638, //11552
     * 11639,11640,11641
     * ,11642,11643,11644,11645,11646,11647,11648,11649,11650,11651
     * ,11652,11653,11654, //11568
     * 11655,11656,11657,11658,11659,11660,11661,11662
     * ,11663,11664,11665,11666,11667,11668,11669,11670, //11584
     * 11671,11672,11673
     * ,11674,4968,11675,11676,11677,11678,11679,11680,11681,11682
     * ,11683,11684,11685, //11600
     * 11686,11687,11688,11689,11690,11691,11692,11693
     * ,3809,11694,11695,11696,11697,11698,11699,11700, //11616
     * 11701,11702,11703
     * ,11704,11705,11706,11707,11708,11709,11710,11711,11712,11713
     * ,11714,11715,11716, //11632
     * 11717,11718,3553,11719,11720,11721,11722,11723
     * ,11724,11725,11726,11727,11728,11729,11730,4969, //11648
     * 11731,11732,11733
     * ,11734,11735,11736,11737,11738,11739,11740,4492,11741,11742
     * ,11743,11744,11745, //11664
     * 11746,11747,11748,11749,11750,11751,11752,4970
     * ,11753,11754,11755,11756,11757,11758,11759,11760, //11680
     * 11761,11762,11763
     * ,11764,11765,11766,11767,11768,11769,11770,11771,11772,11773
     * ,11774,11775,11776, //11696
     * 11777,11778,11779,11780,11781,11782,11783,11784
     * ,11785,11786,11787,11788,11789,11790,4971,11791, //11712
     * 11792,11793,11794
     * ,11795,11796,11797,4972,11798,11799,11800,11801,11802,11803
     * ,11804,11805,11806, //11728
     * 11807,11808,11809,11810,4973,11811,11812,11813
     * ,11814,11815,11816,11817,11818,11819,11820,11821, //11744
     * 11822,11823,11824
     * ,11825,11826,11827,11828,11829,11830,11831,11832,11833,11834
     * ,3680,3810,11835, //11760
     * 11836,4974,11837,11838,11839,11840,11841,11842,11843
     * ,11844,11845,11846,11847,11848,11849,11850, //11776
     * 11851,11852,11853,11854
     * ,11855,11856,11857,11858,11859,11860,11861,11862,11863,11864,11865,11866,
     * //11792
     * 11867,11868,11869,11870,11871,11872,11873,11874,11875,11876,11877,
     * 11878,11879,11880,11881,11882, //11808
     * 11883,11884,4493,11885,11886,11887,
     * 11888,11889,11890,11891,11892,11893,11894,11895,11896,11897, //11824
     * 11898
     * ,11899,11900,11901,11902,11903,11904,11905,11906,11907,11908,11909,11910
     * ,11911,11912,11913, //11840
     * 11914,11915,4975,11916,11917,11918,11919,11920
     * ,11921,11922,11923,11924,11925,11926,11927,11928, //11856
     * 11929,11930,11931
     * ,11932,11933,11934,11935,11936,11937,11938,11939,11940,11941
     * ,11942,11943,11944, //11872
     * 11945,11946,11947,11948,11949,4976,11950,11951
     * ,11952,11953,11954,11955,11956,11957,11958,11959, //11888
     * 11960,11961,11962
     * ,11963,11964,11965,11966,11967,11968,11969,11970,11971,11972
     * ,11973,11974,11975, //11904
     * 11976,11977,11978,11979,11980,11981,11982,11983
     * ,11984,11985,11986,11987,4196,11988,11989,11990, //11920
     * 11991,11992,4977,
     * 11993,11994,11995,11996,11997,11998,11999,12000,12001,12002
     * ,12003,12004,12005, //11936
     * 12006,12007,12008,12009,12010,12011,12012,12013
     * ,12014,12015,12016,12017,12018,12019,12020,12021, //11952
     * 12022,12023,12024
     * ,12025,12026,12027,12028,12029,12030,12031,12032,12033,12034
     * ,12035,12036,12037, //11968
     * 12038,12039,12040,12041,12042,12043,12044,12045
     * ,12046,12047,12048,12049,12050,12051,12052,12053, //11984
     * 12054,12055,12056
     * ,12057,12058,12059,12060,12061,4978,12062,12063,12064,12065
     * ,12066,12067,12068, //12000
     * 12069,12070,12071,12072,12073,12074,12075,12076
     * ,12077,12078,12079,12080,12081,12082,12083,12084, //12016
     * 12085,12086,12087
     * ,12088,12089,12090,12091,12092,12093,12094,12095,12096,12097
     * ,12098,12099,12100, //12032
     * 12101,12102,12103,12104,12105,12106,12107,12108
     * ,12109,12110,12111,12112,12113,12114,12115,12116, //12048
     * 12117,12118,12119
     * ,12120,12121,12122,12123,4979,12124,12125,12126,12127,12128
     * ,4197,12129,12130, //12064
     * 12131,12132,12133,12134,12135,12136,12137,12138
     * ,12139,12140,12141,12142,12143,12144,12145,12146, //12080
     * 12147,12148,12149
     * ,12150,12151,12152,12153,12154,4980,12155,12156,12157,12158
     * ,12159,12160,4494, //12096
     * 12161,12162,12163,12164,3811,12165,12166,12167,
     * 12168,12169,4495,12170,12171,4496,12172,12173, //12112
     * 12174,12175,12176,3812
     * ,12177,12178,12179,12180,12181,12182,12183,12184,12185,12186,12187,12188,
     * //12128
     * 12189,12190,12191,12192,12193,12194,12195,12196,12197,12198,12199,
     * 12200,12201,12202,12203,12204, //12144
     * 12205,12206,12207,12208,12209,12210
     * ,12211,12212,12213,12214,12215,12216,12217,12218,12219,12220, //12160
     * 12221
     * ,4981,12222,12223,12224,12225,12226,12227,12228,12229,12230,12231,12232
     * ,12233,12234,12235, //12176
     * 4982,12236,12237,12238,12239,12240,12241,12242
     * ,12243,12244,12245,4983,12246,12247,12248,12249, //12192
     * 4984,12250,12251,
     * 12252,12253,12254,12255,12256,12257,12258,12259,12260,12261
     * ,12262,12263,12264, //12208
     * 4985,12265,4497,12266,12267,12268,12269,12270,
     * 12271,12272,12273,12274,12275,12276,12277,12278, //12224
     * 12279,12280,12281
     * ,12282,12283,12284,12285,12286,12287,4986,12288,12289,12290
     * ,12291,12292,12293, //12240
     * 12294,12295,12296,2473,12297,12298,12299,12300
     * ,12301,12302,12303,12304,12305,12306,12307,12308, //12256
     * 12309,12310,12311
     * ,12312,12313,12314,12315,12316,12317,12318,12319,3963,12320
     * ,12321,12322,12323, //12272
     * 12324,12325,12326,12327,12328,12329,12330,12331
     * ,12332,4987,12333,12334,12335,12336,12337,12338, //12288
     * 12339,12340,12341
     * ,12342,12343,12344,12345,12346,12347,12348,12349,12350,12351
     * ,12352,12353,12354, //12304
     * 12355,12356,12357,12358,12359,3964,12360,12361
     * ,12362,12363,12364,12365,12366,12367,12368,12369, //12320
     * 12370,3965,12371
     * ,12372,12373,12374,12375,12376,12377,12378,12379,12380,12381
     * ,12382,12383,12384, //12336
     * 12385,12386,12387,12388,12389,12390,12391,12392
     * ,12393,12394,12395,12396,12397,12398,12399,12400, //12352
     * 12401,12402,12403
     * ,12404,12405,12406,12407,12408,4988,12409,12410,12411,12412
     * ,12413,12414,12415, //12368
     * 12416,12417,12418,12419,12420,12421,12422,12423
     * ,12424,12425,12426,12427,12428,12429,12430,12431, //12384
     * 12432,12433,12434
     * ,12435,12436,12437,12438,3554,12439,12440,12441,12442,12443
     * ,12444,12445,12446, //12400
     * 12447,12448,12449,12450,12451,12452,12453,12454
     * ,12455,12456,12457,12458,12459,12460,12461,12462, //12416
     * 12463,12464,4989
     * ,12465,12466,12467,12468,12469,12470,12471,12472,12473,12474
     * ,12475,12476,12477, //12432
     * 12478,12479,12480,4990,12481,12482,12483,12484
     * ,12485,12486,12487,12488,12489,4498,12490,12491, //12448
     * 12492,12493,12494
     * ,12495,12496,12497,12498,12499,12500,12501,12502,12503,12504
     * ,12505,12506,12507, //12464
     * 12508,12509,12510,12511,12512,12513,12514,12515
     * ,12516,12517,12518,12519,12520,12521,12522,12523, //12480
     * 12524,12525,12526
     * ,12527,12528,12529,12530,12531,12532,12533,12534,12535,12536
     * ,12537,12538,12539, //12496
     * 12540,12541,12542,12543,12544,12545,12546,12547
     * ,12548,12549,12550,12551,4991,12552,12553,12554, //12512
     * 12555,12556,12557
     * ,12558,12559,12560,12561,12562,12563,12564,12565,12566,12567
     * ,12568,12569,12570, //12528
     * 12571,12572,12573,12574,12575,12576,12577,12578
     * ,3036,12579,12580,12581,12582,12583,3966,12584, //12544
     * 12585,12586,12587,
     * 12588,12589,12590,12591,12592,12593,12594,12595,12596,12597
     * ,12598,12599,12600, //12560
     * 12601,12602,12603,12604,12605,12606,12607,12608
     * ,12609,12610,12611,12612,12613,12614,12615,12616, //12576
     * 12617,12618,12619
     * ,12620,12621,12622,12623,12624,12625,12626,12627,12628,12629
     * ,12630,12631,12632, //12592
     * 12633,12634,12635,12636,12637,12638,12639,12640
     * ,12641,12642,12643,12644,12645,12646,4499,12647, //12608
     * 12648,12649,12650
     * ,12651,12652,12653,12654,12655,12656,12657,12658,12659,12660
     * ,12661,12662,12663, //12624
     * 12664,12665,12666,12667,12668,12669,12670,12671
     * ,12672,12673,12674,12675,12676,12677,12678,12679, //12640
     * 12680,12681,12682
     * ,12683,12684,12685,12686,12687,12688,12689,12690,12691,12692
     * ,12693,12694,12695, //12656
     * 12696,12697,12698,4992,12699,12700,12701,12702
     * ,12703,12704,12705,12706,12707,12708,12709,12710, //12672
     * 12711,12712,12713
     * ,12714,12715,12716,12717,12718,12719,12720,12721,12722,12723
     * ,12724,12725,12726, //12688
     * 12727,12728,12729,12730,12731,12732,12733,12734
     * ,12735,12736,12737,12738,12739,12740,12741,12742, //12704
     * 12743,12744,12745
     * ,12746,12747,12748,12749,12750,12751,12752,12753,12754,12755
     * ,12756,12757,12758, //12720
     * 12759,12760,12761,12762,12763,12764,12765,12766
     * ,12767,12768,12769,12770,12771,12772,12773,12774, //12736
     * 12775,12776,12777
     * ,12778,4993,2175,12779,12780,12781,12782,12783,12784,12785
     * ,12786,4500,12787, //12752
     * 12788,12789,12790,12791,12792,12793,12794,12795
     * ,12796,12797,12798,12799,12800,12801,12802,12803, //12768
     * 12804,12805,12806
     * ,12807,12808,12809,12810,12811,12812,12813,12814,12815,12816
     * ,12817,12818,12819, //12784
     * 12820,12821,12822,12823,12824,12825,12826,4198
     * ,3967,12827,12828,12829,12830,12831,12832,12833, //12800
     * 12834,12835,12836
     * ,12837,12838,12839,12840,12841,12842,12843,12844,12845,12846
     * ,12847,12848,12849, //12816
     * 12850,12851,12852,12853,12854,12855,12856,12857
     * ,12858,12859,12860,12861,4199,12862,12863,12864, //12832
     * 12865,12866,12867
     * ,12868,12869,12870,12871,12872,12873,12874,12875,12876,12877
     * ,12878,12879,12880, //12848
     * 12881,12882,12883,12884,12885,12886,12887,4501
     * ,12888,12889,12890,12891,12892,12893,12894,12895, //12864
     * 12896,12897,12898
     * ,12899,12900,12901,12902,12903,12904,12905,12906,12907,12908
     * ,12909,12910,12911, //12880
     * 12912,4994,12913,12914,12915,12916,12917,12918
     * ,12919,12920,12921,12922,12923,12924,12925,12926, //12896
     * 12927,12928,12929
     * ,12930,12931,12932,12933,12934,12935,12936,12937,12938,12939
     * ,12940,12941,12942, //12912
     * 12943,12944,12945,12946,12947,12948,12949,12950
     * ,12951,12952,12953,12954,12955,12956,1772,12957, //12928
     * 12958,12959,12960
     * ,12961,12962,12963,12964,12965,12966,12967,12968,12969,12970
     * ,12971,12972,12973, //12944
     * 12974,12975,12976,12977,12978,12979,12980,12981
     * ,12982,12983,12984,12985,12986,12987,12988,12989, //12960
     * 12990,12991,12992
     * ,12993,12994,12995,12996,12997,4502,12998,4503,12999,13000
     * ,13001,13002,13003, //12976
     * 4504,13004,13005,13006,13007,13008,13009,13010
     * ,13011,13012,13013,13014,13015,13016,13017,13018, //12992
     * 13019,13020,13021
     * ,13022,13023,13024,13025,13026,13027,13028,13029,3449,13030
     * ,13031,13032,13033, //13008
     * 13034,13035,13036,13037,13038,13039,13040,13041
     * ,13042,13043,13044,13045,13046,13047,13048,13049, //13024
     * 13050,13051,13052
     * ,13053,13054,13055,13056,13057,13058,13059,13060,13061,13062
     * ,13063,13064,13065, //13040
     * 13066,13067,13068,13069,13070,13071,13072,13073
     * ,13074,13075,13076,13077,13078,13079,13080,13081, //13056
     * 13082,13083,13084
     * ,13085,13086,13087,13088,13089,13090,13091,13092,13093,13094
     * ,13095,13096,13097, //13072
     * 13098,13099,13100,13101,13102,13103,13104,13105
     * ,13106,13107,13108,13109,13110,13111,13112,13113, //13088
     * 13114,13115,13116
     * ,13117,13118,3968,13119,4995,13120,13121,13122,13123,13124
     * ,13125,13126,13127, //13104
     * 4505,13128,13129,13130,13131,13132,13133,13134
     * ,4996,4506,13135,13136,13137,13138,13139,4997, //13120
     * 13140,13141,13142,13143
     * ,13144,13145,13146,13147,13148,13149,13150,13151,13152,13153,13154,13155,
     * //13136
     * 13156,13157,13158,13159,4998,13160,13161,13162,13163,13164,13165,13166
     * ,13167,13168,13169,13170, //13152
     * 13171,13172,13173,13174,13175,13176,4999
     * ,13177,13178,13179,13180,13181,13182,13183,13184,13185, //13168
     * 13186,13187
     * ,13188,13189,13190,13191,13192,13193,13194,13195,13196,13197,13198
     * ,13199,13200,13201, //13184
     * 13202,13203,13204,13205,13206,5000,13207,13208
     * ,13209,13210,13211,13212,13213,13214,13215,13216, //13200
     * 13217,13218,13219
     * ,13220,13221,13222,13223,13224,13225,13226,13227,4200,5001
     * ,13228,13229,13230, //13216
     * 13231,13232,13233,13234,13235,13236,13237,13238
     * ,13239,13240,3969,13241,13242,13243,13244,3970, //13232
     * 13245,13246,13247,
     * 13248,13249,13250,13251,13252,13253,13254,13255,13256,13257
     * ,13258,13259,13260, //13248
     * 13261,13262,13263,13264,13265,13266,13267,13268
     * ,3450,13269,13270,13271,13272,13273,13274,13275, //13264
     * 13276,5002,13277,
     * 13278,13279,13280,13281,13282,13283,13284,13285,13286,13287
     * ,13288,13289,13290, //13280
     * 13291,13292,13293,13294,13295,13296,13297,13298
     * ,13299,13300,13301,13302,3813,13303,13304,13305, //13296
     * 13306,13307,13308
     * ,13309,13310,13311,13312,13313,13314,13315,13316,13317,13318
     * ,13319,13320,13321, //13312
     * 13322,13323,13324,13325,13326,13327,13328,4507
     * ,13329,13330,13331,13332,13333,13334,13335,13336, //13328
     * 13337,13338,13339
     * ,13340,13341,5003,13342,13343,13344,13345,13346,13347,13348
     * ,13349,13350,13351, //13344
     * 13352,13353,13354,13355,13356,13357,13358,13359
     * ,13360,13361,13362,13363,13364,13365,13366,13367, //13360
     * 5004,13368,13369
     * ,13370,13371,13372,13373,13374,13375,13376,13377,13378,13379
     * ,13380,13381,13382, //13376
     * 13383,13384,13385,13386,13387,13388,13389,13390
     * ,13391,13392,13393,13394,13395,13396,13397,13398, //13392
     * 13399,13400,13401
     * ,13402,13403,13404,13405,13406,13407,13408,13409,13410,13411
     * ,13412,13413,13414, //13408
     * 13415,13416,13417,13418,13419,13420,13421,13422
     * ,13423,13424,13425,13426,13427,13428,13429,13430, //13424
     * 13431,13432,4508
     * ,13433,13434,13435,4201,13436,13437,13438,13439,13440,13441
     * ,13442,13443,13444, //13440
     * 13445,13446,13447,13448,13449,13450,13451,13452
     * ,13453,13454,13455,13456,13457,5005,13458,13459, //13456
     * 13460,13461,13462
     * ,13463,13464,13465,13466,13467,13468,13469,13470,4509,13471
     * ,13472,13473,13474, //13472
     * 13475,13476,13477,13478,13479,13480,13481,13482
     * ,13483,13484,13485,13486,13487,13488,13489,13490, //13488
     * 13491,13492,13493
     * ,13494,13495,13496,13497,13498,13499,13500,13501,13502,13503
     * ,13504,13505,13506, //13504
     * 13507,13508,13509,13510,13511,13512,13513,13514
     * ,13515,13516,13517,13518,13519,13520,13521,13522, //13520
     * 13523,13524,13525
     * ,13526,13527,13528,13529,13530,13531,13532,13533,13534,13535
     * ,13536,13537,13538, //13536
     * 13539,13540,13541,13542,13543,13544,13545,13546
     * ,13547,13548,13549,13550,13551,13552,13553,13554, //13552
     * 13555,13556,13557
     * ,13558,13559,13560,13561,13562,13563,13564,13565,13566,13567
     * ,13568,13569,13570, //13568
     * 13571,13572,13573,13574,13575,13576,13577,13578
     * ,13579,13580,13581,13582,13583,13584,13585,13586, //13584
     * 13587,13588,13589
     * ,13590,13591,13592,13593,13594,13595,13596,13597,13598,13599
     * ,13600,13601,13602, //13600
     * 13603,13604,13605,13606,13607,13608,13609,13610
     * ,13611,13612,13613,13614,13615,13616,13617,13618, //13616
     * 13619,13620,13621
     * ,13622,13623,13624,13625,13626,13627,13628,13629,13630,13631
     * ,13632,13633,13634, //13632
     * 13635,13636,13637,13638,13639,13640,13641,13642
     * ,5006,13643,13644,13645,13646,13647,13648,13649, //13648
     * 13650,13651,5007,
     * 13652,13653,13654,13655,13656,13657,13658,13659,13660,13661
     * ,13662,13663,13664, //13664
     * 13665,13666,13667,13668,13669,13670,13671,13672
     * ,13673,13674,13675,13676,13677,13678,13679,13680, //13680
     * 13681,13682,13683
     * ,13684,13685,13686,13687,13688,13689,13690,13691,13692,13693
     * ,13694,13695,13696, //13696
     * 13697,13698,13699,13700,13701,13702,13703,13704
     * ,13705,13706,13707,13708,13709,13710,13711,13712, //13712
     * 13713,13714,13715
     * ,13716,13717,13718,13719,13720,13721,13722,13723,13724,13725
     * ,13726,13727,13728, //13728
     * 13729,13730,13731,13732,13733,13734,13735,13736
     * ,13737,13738,13739,13740,13741,13742,13743,13744, //13744
     * 13745,13746,13747
     * ,13748,13749,13750,13751,13752,13753,13754,13755,13756,13757
     * ,13758,13759,13760, //13760
     * 13761,13762,13763,13764,13765,13766,13767,13768
     * ,13769,13770,13771,13772,13773,13774,3273,13775, //13776
     * 13776,13777,13778
     * ,13779,13780,13781,13782,13783,13784,13785,13786,13787,13788
     * ,13789,13790,13791, //13792
     * 13792,13793,13794,13795,13796,13797,13798,13799
     * ,13800,13801,13802,13803,13804,13805,13806,13807, //13808
     * 13808,13809,13810
     * ,13811,13812,13813,13814,13815,13816,13817,13818,13819,13820
     * ,13821,13822,13823, //13824
     * 13824,13825,13826,13827,13828,13829,13830,13831
     * ,13832,13833,13834,13835,13836,13837,13838,13839, //13840
     * 13840,13841,13842
     * ,13843,13844,13845,13846,13847,13848,13849,13850,13851,13852
     * ,13853,13854,13855, //13856
     * 13856,13857,13858,13859,13860,13861,13862,13863
     * ,13864,13865,13866,13867,13868,13869,13870,13871, //13872
     * 13872,13873,13874
     * ,13875,13876,13877,13878,13879,13880,13881,13882,13883,13884
     * ,13885,13886,13887, //13888
     * 13888,13889,13890,13891,13892,13893,13894,13895
     * ,13896,13897,13898,13899,13900,13901,13902,13903, //13904
     * 13904,13905,13906
     * ,13907,13908,13909,13910,13911,13912,13913,13914,13915,13916
     * ,13917,13918,13919, //13920
     * 13920,13921,13922,13923,13924,13925,13926,13927
     * ,13928,13929,13930,13931,13932,13933,13934,13935, //13936
     * 13936,13937,13938
     * ,13939,13940,13941,13942,13943,13944,13945,13946,13947,13948
     * ,13949,13950,13951, //13952
     * 13952,13953,13954,13955,13956,13957,13958,13959
     * ,13960,13961,13962,13963,13964,13965,13966,13967, //13968
     * 13968,13969,13970,13971,13972, //13973
     ****************************************************************************************/
    };
}
