/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*

                Motion-JPEG SW decoder for stagefright plugin

GENERAL DESCRIPTION
  This module provides Motion-JPEG video decoder for stagefright plugin.


Copyright(c) 2014 by LG Electronics. All Rights Reserved.
*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*/

/*===========================================================================

                      EDIT HISTORY FOR FILE

  This section contains comments describing changes made to this file.
  Notice that changes are listed in reverse chronological order.

when         who         what, where, why
--------     -------    -------------------------------------------------------
01/12/2015   KHLEE      Modified API location(avoiding memory leak)
12/31/2014   KHLEE      Added LGE Copyright & Updated NOTICE file
xx/xx/2013   KHLee      Created file

============================================================================*/

//#define LOG_NDEBUG 0
#define LOG_TAG "SoftMJPG"
#include <utils/Log.h>

#include <media/stagefright/foundation/ADebug.h>
//#include <media/stagefright/MediaDefs.h>
#include "LgeMediaDefs.h"
#include <HardwareAPI.h>
#include <setjmp.h>
#include <stdio.h>

#include "SoftMJPG.h"

#ifdef LGE_MJPEG_FRAME_SKIP
#define MAX_SPECOUT_WIDTH 1280
#define MAX_SPECOUT_HEIGHT 720
#define MAX_SPECOUT_FRAMERATE 30
#endif

#ifdef LGE_MJPEG_USE_NEON_REGISTER
#include <arm_neon.h>
#endif

extern "C" {
    #include "jpeglib.h"
    #include "jerror.h"
}

namespace android {

template<class T>
static void InitOMXParams(T *params) {
    params->nSize = sizeof(T);                      /**< Size of the structure in bytes */
    params->nVersion.s.nVersionMajor = 1;           /**< OMX specification version information */
    params->nVersion.s.nVersionMinor = 0;
    params->nVersion.s.nRevision = 0;
    params->nVersion.s.nStep = 0;
}

void codec_jpeg_error_exit(j_common_ptr cinfo) {
    struct codec_jpeg_error_mgr *c_err = (struct codec_jpeg_error_mgr *)cinfo->err;
    longjmp(c_err->longjmp_buffer, 0);
}

SoftMJPG::SoftMJPG(
        const char *name,
        const OMX_CALLBACKTYPE *callbacks,
        OMX_PTR appData,
        OMX_COMPONENTTYPE **component)
    : SimpleSoftOMXComponent(name, callbacks, appData, component),
      mSignalledError(false),
#ifdef LGE_MJPEG_FRAME_SKIP
      mIsDownScale(false),
      mIsSpecOut(false),
      mFrameRate(30),
#endif
      mSkipRate(-1),
      mWidth(1920),
      mHeight(1080),
      mCropLeft(0),
      mCropTop(0),
      mCropWidth(mWidth),
      mCropHeight(mHeight),
      inBuffInfoBackup(new uint8_t[sizeof(BufferInfo) * sizeof(uint8_t)]),
      outBuffInfoBackup(new uint8_t[sizeof(BufferInfo) * sizeof(uint8_t)]),
      inBuffInfoBackupThread(new uint8_t[sizeof(BufferInfo) * sizeof(uint8_t)]),
      outBuffInfoBackupThread(new uint8_t[sizeof(BufferInfo) * sizeof(uint8_t)]),
      mThreadKill(false), // Thread Kill or not
      mThreadWork(false), // Thread can work or not - only enabled after main thread called FILLBUFFERDONE
      mIsThreadCreated(false), // Thread is created or not
      mThreadDecoding(false), // Thread is working or not
      mIsEOSSignaled(false), // EOS is reached or not
      mOutputPortSettingsChange(NONE)
{
    initPorts();
    initHandle();

    if (!mIsThreadCreated) { // Just in port setting changed case
        createThread(threadStrat, this);
    }
}

SoftMJPG::~SoftMJPG() {
    ALOGD("Free memory : %p", this);
    threadKill();
    mIsThreadCreated = false;

    if (inBuffInfoBackup) delete [] inBuffInfoBackup;
    if (outBuffInfoBackup) delete [] outBuffInfoBackup;
    if (inBuffInfoBackupThread) delete [] inBuffInfoBackupThread;
    if (outBuffInfoBackupThread) delete [] outBuffInfoBackupThread;
}

int SoftMJPG::threadStrat(void* self) {
    ALOGD("Create Thread : %p", self);
    SoftMJPG* selfPtr = reinterpret_cast<SoftMJPG*>(self);

    while (!selfPtr->mThreadKill) {
        while (!selfPtr->mThreadWork && !selfPtr->mThreadKill) {
            usleep(5000);
        }
        {
            Mutex::Autolock _l(selfPtr->mLockThread);
            selfPtr->mConditionThread.signal();
        }
        selfPtr->onJPEGDecodingThread();
    }

    selfPtr->mIsThreadCreated = true;
    return 0;
}

void SoftMJPG::threadWork() {
    Mutex::Autolock _l(mLockThread);

    mThreadWork = true;
    mConditionThread.wait(mLockThread);
}

void SoftMJPG::threadKill() {
    ALOGD("Start threadKill : %p", this);
    Mutex::Autolock _l(mLockProcess);

    mThreadWork = false;
    mThreadKill = true;
    mConditionThread.signal();

    while(1) {
        usleep(1000); // For thread end safety
        if(mIsThreadCreated) {
            usleep(1000); // For thread end safety
            break;
        }
    }
    ALOGD("End threadKill : %p", this);
}

void SoftMJPG::initHandle() {
    cinfo.err = jpeg_std_error(&c_err.jerr);
    c_err.jerr.error_exit = codec_jpeg_error_exit;
    dinfo.err = jpeg_std_error(&c_err.jerr);
    c_err.jerr.error_exit = codec_jpeg_error_exit;
}

void SoftMJPG::initPorts() {
    OMX_PARAM_PORTDEFINITIONTYPE def;
    InitOMXParams(&def);

    // Input Port Initialization
    def.nPortIndex = kInputPortIndex;               /**< Port number the structure applies to */
    def.eDir = OMX_DirInput;                        /**< Direction (input or output) of this port */
    def.nBufferCountMin = kNumInputBuffers;         /**< The minimum number of buffers this port requires */
    def.nBufferCountActual = kNumInputBuffers;  /**< The actual number of buffers allocated on this port */
    def.nBufferSize = 1 * 1024 * 1024;              /**< Size, in bytes, for buffers to be used for this channel */
    def.bEnabled = OMX_TRUE;                        /**< Ports default to enabled and are enabled/disabled by
                                                    OMX_CommandPortEnable/OMX_CommandPortDisable.
                                                    When disabled a port is unpopulated. A disabled port
                                                    is not populated with buffers on a transition to IDLE. */
    def.bPopulated = OMX_FALSE;                     /**< Port is populated with all of its buffers as indicated by
                                                    nBufferCountActual. A disabled port is always unpopulated.
                                                    An enabled port is populated on a transition to OMX_StateIdle
                                                    and unpopulated on a transition to loaded. */
    def.eDomain = OMX_PortDomainVideo;              /**< Domain of the port. Determines the contents of metadata below. */
    def.bBuffersContiguous = OMX_FALSE;
    def.nBufferAlignment = 1;

    def.format.video.cMIMEType = const_cast<char *>(LGE_MEDIA_MIMETYPE_VIDEO_MJPG);

    def.format.video.pNativeRender = NULL;
    def.format.video.nFrameWidth = mWidth;
    def.format.video.nFrameHeight = mHeight;
    def.format.video.nStride = def.format.video.nFrameWidth;
    def.format.video.nSliceHeight = def.format.video.nFrameHeight;
    def.format.video.nBitrate = 0;
    def.format.video.xFramerate = 30;
    def.format.video.bFlagErrorConcealment = OMX_FALSE;

    def.format.video.eCompressionFormat = OMX_VIDEO_CodingMJPEG;

    def.format.video.eColorFormat = OMX_COLOR_FormatUnused;
    def.format.video.pNativeWindow = NULL;

    addPort(def);

    // Output Port Initialization
    def.nPortIndex = kOutputPortIndex;
    def.eDir = OMX_DirOutput;
    def.nBufferCountMin = kNumOutputBuffers;
    def.nBufferCountActual = kNumOutputBuffers;
    def.bEnabled = OMX_TRUE;
    def.bPopulated = OMX_FALSE;
    def.eDomain = OMX_PortDomainVideo;
    def.bBuffersContiguous = OMX_FALSE;
    def.nBufferAlignment = 2;

    def.format.video.cMIMEType = const_cast<char *>(LGE_MEDIA_MIMETYPE_VIDEO_RAW);
    def.format.video.pNativeRender = NULL;
    def.format.video.nFrameWidth = mWidth;
    def.format.video.nFrameHeight = mHeight;
    def.format.video.nStride = def.format.video.nFrameWidth;
    def.format.video.nSliceHeight = def.format.video.nFrameHeight;
    def.format.video.nBitrate = 0;
    def.format.video.xFramerate = 0;
    def.format.video.bFlagErrorConcealment = OMX_FALSE;
    def.format.video.eCompressionFormat = OMX_VIDEO_CodingUnused;
    def.format.video.eColorFormat = OMX_COLOR_FormatYUV420Planar; //OMX_QCOM_COLOR_FormatYVU420SemiPlanar / OMX_COLOR_FormatYUV420Planar
    def.format.video.pNativeWindow = NULL;

    def.nBufferSize = (def.format.video.nFrameWidth * def.format.video.nFrameHeight * 3) / 2;

    addPort(def);
}

OMX_ERRORTYPE SoftMJPG::internalGetParameter(
        OMX_INDEXTYPE index, OMX_PTR params) {
    switch (index) {
        case OMX_IndexParamPortDefinition:
        {
            OMX_PARAM_PORTDEFINITIONTYPE *defParams =
                (OMX_PARAM_PORTDEFINITIONTYPE *)params;

            if (defParams->nSize != sizeof(OMX_PARAM_PORTDEFINITIONTYPE)) {
                return OMX_ErrorUndefined;
            }

            OMX_PARAM_PORTDEFINITIONTYPE *def;

            if (defParams->nPortIndex == kInputPortIndex) {
                def = &editPortInfo(kInputPortIndex)->mDef;
            }
            else if (defParams->nPortIndex == kOutputPortIndex) {
                def = &editPortInfo(kOutputPortIndex)->mDef;
            }
            else {
                return OMX_ErrorBadParameter;
            }

            memcpy(defParams, def, sizeof(*def));
            return OMX_ErrorNone;
        }

        case OMX_IndexParamVideoPortFormat:
        {
            OMX_VIDEO_PARAM_PORTFORMATTYPE *formatParams =
                (OMX_VIDEO_PARAM_PORTFORMATTYPE *)params;

            if (formatParams->nIndex != 0) {
                return OMX_ErrorNoMore;
            }

            if (formatParams->nPortIndex == kInputPortIndex) {
                formatParams->eCompressionFormat = OMX_VIDEO_CodingMJPEG;
                formatParams->eColorFormat = OMX_COLOR_FormatUnused;
                formatParams->xFramerate = 0;
            } else if (formatParams->nPortIndex == kOutputPortIndex) {
                formatParams->eCompressionFormat = OMX_VIDEO_CodingUnused;
                formatParams->eColorFormat = OMX_COLOR_FormatYUV420Planar;
                formatParams->xFramerate = 0;
            }
            else {
                return OMX_ErrorBadParameter;
            }

            return OMX_ErrorNone;
        }

        case OMX_IndexParamMJPEGGetAndroidNativeBufferUsage:
        {
            GetAndroidNativeBufferUsageParams* nativeBuffersUsage =
                (GetAndroidNativeBufferUsageParams *)params;

            if (nativeBuffersUsage->nPortIndex > 1) {
                return OMX_ErrorUndefined;
            }

            if (nativeBuffersUsage->nPortIndex == kInputPortIndex || nativeBuffersUsage->nPortIndex == kOutputPortIndex) {
                // Do nothing
            } else {
                return OMX_ErrorBadParameter;
            }

            return OMX_ErrorNone;
        }

        case OMX_IndexParamMJPEGUseAndroidNativeBuffer2:
        {
            OMX_PARAM_PORTDEFINITIONTYPE* portDef =
                (OMX_PARAM_PORTDEFINITIONTYPE *)params;

            if (portDef->nPortIndex == kInputPortIndex) {
                portDef->nBufferSize = 1 * 1024 * 1024;
            }
            else if (portDef->nPortIndex == kOutputPortIndex) {
                portDef->nBufferSize = (mWidth * mHeight * 3) / 2;
            }
            else {
                return OMX_ErrorBadParameter;
            }

            return OMX_ErrorNone;
        }

        default:
            return SimpleSoftOMXComponent::internalGetParameter(index, params);
    }
}

OMX_ERRORTYPE SoftMJPG::internalSetParameter(
        OMX_INDEXTYPE index, const OMX_PTR params) {
    switch (index) {
        case OMX_IndexParamPortDefinition:
        {
            OMX_PARAM_PORTDEFINITIONTYPE *defParams =
                (OMX_PARAM_PORTDEFINITIONTYPE *)params;

            if (defParams->nSize != sizeof(OMX_PARAM_PORTDEFINITIONTYPE)) {
                return OMX_ErrorUndefined;
            }

            OMX_PARAM_PORTDEFINITIONTYPE *def;

            if (defParams->nPortIndex == kInputPortIndex) {
                def = &editPortInfo(kInputPortIndex)->mDef;
            }
            else if (defParams->nPortIndex == kOutputPortIndex) {
                def = &editPortInfo(kOutputPortIndex)->mDef;
            }
            else {
                return OMX_ErrorBadParameter;
            }

            memcpy(def, defParams, sizeof(*defParams));
            return OMX_ErrorNone;
        }

        case OMX_IndexParamStandardComponentRole:
        {
            const OMX_PARAM_COMPONENTROLETYPE *roleParams = (const OMX_PARAM_COMPONENTROLETYPE *)params;
            if (strncmp((const char *)roleParams->cRole, "video_decoder.mjpg", OMX_MAX_STRINGNAME_SIZE - 1)) {
                    return OMX_ErrorUndefined;
            }
            return OMX_ErrorNone;
        }

        case OMX_IndexParamVideoPortFormat:
        {
            OMX_VIDEO_PARAM_PORTFORMATTYPE *formatParams =
                (OMX_VIDEO_PARAM_PORTFORMATTYPE *)params;

            if (formatParams->nPortIndex > 1) {
                return OMX_ErrorUndefined;
            }

            if (formatParams->nIndex != 0) {
                return OMX_ErrorNoMore;
            }

            mFrameRate = formatParams->xFramerate;

            return OMX_ErrorNone;
        }

        case OMX_IndexParamMJPEGEnableAndroidNativeBuffers:
        {
            EnableAndroidNativeBuffersParams* nativeBuffers =
                (EnableAndroidNativeBuffersParams *)params;

            if (nativeBuffers->nPortIndex > 1) {
                return OMX_ErrorUndefined;
            }

            if (nativeBuffers->nPortIndex == kInputPortIndex || nativeBuffers->nPortIndex == kOutputPortIndex) {
                // Do nothing
            } else {
                return OMX_ErrorBadParameter;
            }

            return OMX_ErrorNone;
        }

        default:
            return SimpleSoftOMXComponent::internalSetParameter(index, params);
    }
}

void SoftMJPG::onQueueFilled(OMX_U32 portIndex) {
    if (mSignalledError || mOutputPortSettingsChange != NONE) {
        return;
    }
    while (mThreadWork && !mThreadKill) { // start after thread decoding complete
        usleep(5000);
    }
    onJPEGDecodingProcess();
}

void SoftMJPG::onJPEGDecodingProcess() {
    if (setjmp(c_err.longjmp_buffer)) { // JPEG lib Error process routine
        ALOGD("Exit routine process - cinfo");
        jpeg_destroy_decompress(&cinfo);
        return;
    }
    List<BufferInfo *> &inQueue = getPortQueue(kInputPortIndex);
    List<BufferInfo *> &outQueue = getPortQueue(kOutputPortIndex);

    JSAMPARRAY bufferOddScanLine;   /* Output buffer line*/
    JSAMPARRAY bufferEvenScanLine;  /* Output buffer line*/

    uint8_t *src;       /* address pointer intput buffer */
    uint8_t *dst;       /* address pointer output buffer */
    int row_stride;     /* physical row width in output buffer */
//ALOGI("SoftMJPG::%s()[%d]:: inQueue.size()=[%d], outQueue.size()=[%d]", __FUNCTION__, __LINE__, inQueue.size(), outQueue.size());
    if (!inQueue.empty() && !outQueue.empty()) { // Only EOS Buffer remained case
        BufferInfo *inInfo = *inQueue.begin();
        OMX_BUFFERHEADERTYPE *inHeader = inInfo->mHeader;

        BufferInfo *outInfo = *outQueue.begin();
        OMX_BUFFERHEADERTYPE *outHeader = outInfo->mHeader;

        if (mIsEOSSignaled) { // Already EOS received, this need reset flag case
            //ALOGD("SoftMJPG::%s()[%d]:: Pre-PROCESS EOS reset", __FUNCTION__, __LINE__);
            mIsEOSSignaled = false;
        }

        if (inHeader->nFlags & OMX_BUFFERFLAG_EOS) {
            //ALOGD("SoftMJPG::%s()[%d]:: Pre-PROCESS OMX_BUFFERFLAG_EOS", __FUNCTION__, __LINE__);
            mIsEOSSignaled = true;

            inQueue.erase(inQueue.begin());
            inInfo->mOwnedByUs = false;
            notifyEmptyBufferDone(inHeader);
            //ALOGD("SoftMJPG::%s()[%d]:: Pre-PROCESS notifyEmptyBufferDone [%p]", __FUNCTION__, __LINE__, *inInfo);

            outHeader->nFilledLen = 0;
            outHeader->nFlags |= OMX_BUFFERFLAG_EOS;
            outQueue.erase(outQueue.begin());
            outInfo->mOwnedByUs = false;
            notifyFillBufferDone(outHeader);
            //ALOGD("SoftMJPG::%s()[%d]:: Pre-PROCESS notifyFillBufferDone", __FUNCTION__, __LINE__);

            mThreadWork = false;
            return;
        }
    }

    while (inQueue.size() > 1 && outQueue.size() > 1) { // Main decoding
        memcpy(inBuffInfoBackup, *inQueue.begin(), sizeof(BufferInfo)); // copy bufferinfo for thread decoding
        memcpy(outBuffInfoBackup, *outQueue.begin(), sizeof(BufferInfo));

        BufferInfo *inInfo = (BufferInfo*)inBuffInfoBackup;
        BufferInfo *inInfoPtr = *inQueue.begin(); // for buffer owner definition
        OMX_BUFFERHEADERTYPE *inHeader = inInfo->mHeader;

        uint32_t inBuffFilledLen = inHeader->nFilledLen;
        uint32_t inBuffTimeStamp = inHeader->nTimeStamp;

        BufferInfo *outInfo = (BufferInfo*)outBuffInfoBackup;
        BufferInfo *outInfoPtr = *outQueue.begin(); // for buffer owner definition
        OMX_BUFFERHEADERTYPE *outHeader = outInfo->mHeader;

        src = (uint8_t *)(inHeader->pBuffer + inHeader->nOffset);
        dst = (uint8_t *)(outHeader->pBuffer);

        if (mIsEOSSignaled) { // Already EOS received, this need reset flag case
            //ALOGD("SoftMJPG::%s()[%d]:: PROCESS EOS reset", __FUNCTION__, __LINE__);
            mIsEOSSignaled = false;
        }

        if (inHeader->nFlags & OMX_BUFFERFLAG_EOS) {
            ALOGD("SoftMJPG::%s()[%d]:: PROCESS OMX_BUFFERFLAG_EOS", __FUNCTION__, __LINE__);
            mIsEOSSignaled = true;

            inQueue.erase(inQueue.begin());
            inInfoPtr->mOwnedByUs = false;
            notifyEmptyBufferDone(inHeader);
            //ALOGD("SoftMJPG::%s()[%d]:: PROCESS notifyEmptyBufferDone [%p]", __FUNCTION__, __LINE__, *inInfo);

            outHeader->nFilledLen = 0;
            outHeader->nFlags |= OMX_BUFFERFLAG_EOS;
            outQueue.erase(outQueue.begin());
            outInfoPtr->mOwnedByUs = false;
            notifyFillBufferDone(outHeader);
            //ALOGD("SoftMJPG::%s()[%d]:: PROCESS notifyFillBufferDone", __FUNCTION__, __LINE__);

            mThreadWork = false;
            jpeg_destroy_decompress(&cinfo);
            return;
        }

        if ( !isValidFrame(src, inBuffFilledLen) ) {
            ALOGE("Process - Invalid frame data");
            notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
            mThreadWork = false;
            mIsEOSSignaled = true;
            jpeg_destroy_decompress(&cinfo);
            return;
        }

        jpeg_create_decompress(&cinfo); // Initialize JPEG lib object
        //jpeg_stdio_src(&cinfo, file pointer); - cannot use std io src, because of file pointer, need setting cinfo from buffer source.

        jpeg_buffer_src(&cinfo, src, inBuffFilledLen); // added jpeg function
        jpeg_read_header(&cinfo, TRUE);

        cinfo.out_color_space = JCS_YCbCr; // YUV444 interleaved
        cinfo.dct_method = JDCT_IFAST; // JDCT_DEFAULT is JDCT_ISLOW
        cinfo.dither_mode = JDITHER_ORDERED; // DEFAULT is JDITHER_FS - JDITHER_NONE/JDITHER_ORDERED is another option.
        cinfo.do_fancy_upsampling = FALSE; // DEFAULT is TRUE - main factor
        cinfo.do_block_smoothing = FALSE; // DEFAULT is TRUE - sub factor

        jpeg_start_decompress(&cinfo); // Alloc JPEG lib memory

        if (3 == cinfo.out_color_components && JCS_YCbCr == cinfo.out_color_space) {
            if ((mSkipRate < 0) && ((cinfo.output_width != mWidth) || (cinfo.output_height != mHeight))) {
                ALOGD("%s()[%d]::Update Port Definitions - mWidth=[%u] -> output_width=[%u], mHeight=[%u] -> output_height=[%u]", __FUNCTION__, __LINE__, mWidth, cinfo.output_width, mHeight, cinfo.output_height);
                mWidth = cinfo.output_width;
                mHeight = cinfo.output_height;

                updatePortDefinitions();

                notify(OMX_EventPortSettingsChanged, 1, 0, NULL);
                mOutputPortSettingsChange = AWAITING_DISABLED;

                jpeg_finish_decompress(&cinfo);
                jpeg_destroy_decompress(&cinfo);
                mThreadWork = false;
                return;
            }

            inInfoPtr->mOwnedByUs = false;
            inQueue.erase(inQueue.begin()); // release buffer for thread decoding
            outInfoPtr->mOwnedByUs = false;
            outQueue.erase(outQueue.begin());

            if (!mThreadWork) { // Activate Thread Decoding
                mThreadDecoding = false;
                threadWork();
            }

            outHeader->nOffset = 0;
            outHeader->nFilledLen = ((mWidth * mHeight * 3) >> 1);
            outHeader->nFlags = 0;
            outHeader->nTimeStamp = inBuffTimeStamp;

            row_stride = 3 * mWidth; // # of 1 line buffer
            bufferOddScanLine = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
            bufferEvenScanLine = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

            uint32_t half_of_mWidth = mWidth >> 1;
            uint32_t half_of_mHeight = mHeight >> 1;

            uint8_t* dst_Y_ptr = dst;
            uint8_t* dst_U_ptr = dst_Y_ptr + (mWidth * mHeight);
            uint8_t* dst_V_ptr = dst_U_ptr + half_of_mWidth * half_of_mHeight;

            uint8_t* buff_Odd_ptr = (uint8_t*)bufferOddScanLine[0];
            uint8_t* buff_Even_ptr = (uint8_t*)bufferEvenScanLine[0];
            uint32_t scanline = half_of_mHeight;
            uint32_t offset = 0;

#ifdef LGE_MJPEG_USE_NEON_REGISTER
            bool extraProcessing = false;
            uint32_t extraLength = 0;

            if (mWidth < 16) {
                ALOGE("Decode Fail - Not supported contents size - width=[%u]", mWidth);
                notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
                jpeg_finish_decompress(&cinfo);
                jpeg_destroy_decompress(&cinfo);
                mThreadWork = false;
                return;
            }
            else if (mWidth % 16 != 0) { // Need right Edge processing
                extraProcessing = true;
                extraLength = (mWidth%16) > 1 ? mWidth%16 : 0; // ignore only 1 pixel remaining case
            }
            uint32_t dq_of_mWidth = mWidth;
#endif
            while (scanline-- > 0) {
                if (0 == jpeg_read_scanlines(&cinfo, bufferOddScanLine, 1)) { // put decoded result - Odd line
                    ALOGE("SoftMJPG::Decode Fail - odd_line[%u], scanline=[%u]", cinfo.output_scanline, scanline);
                    notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
                    jpeg_finish_decompress(&cinfo);
                    jpeg_destroy_decompress(&cinfo);
                    mThreadWork = false;
                    return;
                }
                if (0 == jpeg_read_scanlines(&cinfo, bufferEvenScanLine, 1)) { // put decoded result - Even line
                    ALOGE("SoftMJPG::Decode Fail - even_line[%u], scanline=[%u]", cinfo.output_scanline, scanline);
                    notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
                    jpeg_finish_decompress(&cinfo);
                    jpeg_destroy_decompress(&cinfo);
                    mThreadWork = false;
                    return;
                }
#ifdef LGE_MJPEG_USE_NEON_REGISTER
                NeonConvertor(dst_Y_ptr,dst_U_ptr,dst_V_ptr, buff_Odd_ptr,buff_Even_ptr,dq_of_mWidth);

                if (extraProcessing) { // width is not divided by 16
                    offset = dq_of_mWidth - extraLength;
                    for (int i=0; i<extraLength; i++) {
                        dst_Y_ptr[offset + i] = buff_Odd_ptr[3*(offset + i)]; // Y
                    }
                    for (int j=0; j<extraLength; j++) {
                        dst_Y_ptr[dq_of_mWidth + offset + j] = buff_Even_ptr[3*(offset + j)]; // Y
                        if (j%2 == 1) { // process 2 pixels at once - U,V
                            dst_U_ptr[(dq_of_mWidth >> 1) - (extraLength >> 1) + (j >> 1)] =
                                (buff_Odd_ptr[3*(offset - 1 + j) + 1] + buff_Odd_ptr[3*(offset + j) + 1] + buff_Even_ptr[3*(offset - 1 + j) + 1] + buff_Even_ptr[3*(offset + j) + 1]) >> 2; // U
                            dst_V_ptr[(dq_of_mWidth >> 1) - (extraLength >> 1) + (j >> 1)] =
                                (buff_Odd_ptr[3*(offset - 1 + j) + 2] + buff_Odd_ptr[3*(offset + j) + 2] + buff_Even_ptr[3*(offset - 1 + j) + 2] + buff_Even_ptr[3*(offset + j) + 2]) >> 2; // V
                        }
                    }
                }
                dst_Y_ptr += (dq_of_mWidth << 1);
                dst_U_ptr += (dq_of_mWidth >> 1);
                dst_V_ptr += (dq_of_mWidth >> 1);
#else
                for (uint32_t col=0; col < half_of_mWidth; col++) { // YUV444 interleaved -> YUV420 planar Color conversion - 2 by 2 filter
                    offset = ((col << 2) + (col << 1));
                    *dst_Y_ptr++ = buff_Odd_ptr[offset]; // Y
                    *dst_Y_ptr++ = buff_Odd_ptr[offset + 3]; // Y
                }
                for (uint32_t col=0; col < half_of_mWidth; col++) { // YUV444 interleaved -> YUV420 planar Color conversion - 2 by 2 filter
                    offset = ((col << 2) + (col << 1));
                    *dst_Y_ptr++ = buff_Even_ptr[offset]; // Y
                    *dst_Y_ptr++ = buff_Even_ptr[offset + 3]; // Y
                    *dst_U_ptr++ = (buff_Odd_ptr[offset + 1] + buff_Odd_ptr[offset + 4] + buff_Even_ptr[offset + 1] + buff_Even_ptr[offset + 4]) >> 2; // U
                    *dst_V_ptr++ = (buff_Odd_ptr[offset + 2] + buff_Odd_ptr[offset + 5] + buff_Even_ptr[offset + 2] + buff_Even_ptr[offset + 5]) >> 2; // V
                }
#endif
            } // while end
        }
        else {
            ALOGE("NOT Supported format comp[%d], color[%u]", cinfo.out_color_components, cinfo.out_color_space);
            notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
            break;
        }
        //ALOGD("SoftMJPG::%s()[%d]:: PROCESS notifyEmptyBufferDone [%p]", __FUNCTION__, __LINE__, *inInfo);
        notifyEmptyBufferDone(inHeader);
        //ALOGD("SoftMJPG::%s()[%d]:: PROCESS notifyFillBufferDone :: inBuffTimeStamp=[%d]", __FUNCTION__, __LINE__, inBuffTimeStamp);
        notifyFillBufferDone(outHeader);

        jpeg_finish_decompress(&cinfo);
        jpeg_destroy_decompress(&cinfo);
        break;
    }
    mThreadDecoding = true;
    while (mThreadWork && !mThreadKill && !mIsEOSSignaled) {
        usleep(5000);
    }
}

void SoftMJPG::onJPEGDecodingThread() {
    if (setjmp(c_err.longjmp_buffer)) { // JPEG lib Error process routine
        ALOGD("THREAD Exit routine process - dinfo");
        jpeg_destroy_decompress(&dinfo);
        return;
    }

    if (mIsEOSSignaled)
        return;

    List<BufferInfo *> &inQueue = getPortQueue(kInputPortIndex);
    List<BufferInfo *> &outQueue = getPortQueue(kOutputPortIndex);

    JSAMPARRAY bufferOddScanLine;   /* Output buffer line*/
    JSAMPARRAY bufferEvenScanLine;  /* Output buffer line*/

    uint8_t *src;       /* address pointer intput buffer */
    uint8_t *dst;       /* address pointer output buffer */
    int row_stride;     /* physical row width in output buffer */

    while (!inQueue.empty() && !outQueue.empty()) {
        memcpy(inBuffInfoBackupThread, *inQueue.begin(), sizeof(BufferInfo));
        memcpy(outBuffInfoBackupThread, *outQueue.begin(), sizeof(BufferInfo));

        BufferInfo *inInfo = (BufferInfo*)inBuffInfoBackupThread;
        BufferInfo *inInfoPtr = *inQueue.begin();
        OMX_BUFFERHEADERTYPE *inHeader = inInfo->mHeader;

        uint32_t inBuffFilledLen = inHeader->nFilledLen;
        uint32_t inBuffTimeStamp = inHeader->nTimeStamp;

        BufferInfo *outInfo = (BufferInfo*)outBuffInfoBackupThread;
        BufferInfo *outInfoPtr = *outQueue.begin();
        OMX_BUFFERHEADERTYPE *outHeader = outInfo->mHeader;

        src = (uint8_t *)(inHeader->pBuffer + inHeader->nOffset);
        dst = (uint8_t *)(outHeader->pBuffer);

        if (mIsEOSSignaled) { // Already EOS received, this need reset flag case
            //ALOGD("SoftMJPG::%s()[%d]:: THREAD EOS reset", __FUNCTION__, __LINE__);
            mIsEOSSignaled = false;
        }

        if (inHeader->nFlags & OMX_BUFFERFLAG_EOS) {
            //ALOGD("SoftMJPG::%s()[%d]:: THREAD OMX_BUFFERFLAG_EOS", __FUNCTION__, __LINE__);
            mThreadWork = false;

            inQueue.erase(inQueue.begin());
            inInfoPtr->mOwnedByUs = false;
            notifyEmptyBufferDone(inHeader);
            //ALOGD("SoftMJPG::%s()[%d]:: THREAD notifyEmptyBufferDone [%p]", __FUNCTION__, __LINE__, *inInfo);

            outHeader->nFilledLen = 0;
            outHeader->nFlags |= OMX_BUFFERFLAG_EOS;
            outQueue.erase(outQueue.begin());
            outInfoPtr->mOwnedByUs = false;
            notifyFillBufferDone(outHeader);
            //ALOGD("SoftMJPG::%s()[%d]:: THREAD notifyFillBufferDone :: inBuffTimeStamp=[%d]", __FUNCTION__, __LINE__, inBuffTimeStamp);

            mIsEOSSignaled = true;
            jpeg_destroy_decompress(&dinfo);
            return;
        }

        if ( !isValidFrame(src, inBuffFilledLen) ) {
            ALOGE("Thread - Invalid frame data");
            notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
            mThreadWork = false;
            mIsEOSSignaled = true;
            jpeg_destroy_decompress(&dinfo);
            return;
        }

        jpeg_create_decompress(&dinfo); // Initialize JPEG lib object
        //jpeg_stdio_src(&dinfo, file pointer); - cannot use std io src, because of file pointer, need setting dinfo from buffer source.

        jpeg_buffer_src(&dinfo, src, inBuffFilledLen); // added jpeg function

        jpeg_read_header(&dinfo, TRUE);

        dinfo.out_color_space = JCS_YCbCr; // YUV444 interleaved
        dinfo.dct_method = JDCT_IFAST; // JDCT_DEFAULT is JDCT_ISLOW
        dinfo.dither_mode = JDITHER_ORDERED; // DEFAULT is JDITHER_FS - JDITHER_NONE is another option.
        dinfo.do_fancy_upsampling = FALSE; // DEFAULT is TRUE - main factor
        dinfo.do_block_smoothing = FALSE; // DEFAULT is TRUE - sub factor

        jpeg_start_decompress(&dinfo); // Alloc JPEG lib memory

        if (3 == dinfo.out_color_components && JCS_YCbCr == dinfo.out_color_space) {
            inInfoPtr->mOwnedByUs = false;
            inQueue.erase(inQueue.begin());
            outInfoPtr->mOwnedByUs = false;
            outQueue.erase(outQueue.begin());

            outHeader->nOffset = 0;
            outHeader->nFilledLen = ((mWidth * mHeight * 3) >> 1);
            outHeader->nFlags = 0;
            outHeader->nTimeStamp = inBuffTimeStamp;

            row_stride = 3 * mWidth; // # of 1 line buffer
            bufferOddScanLine = (*dinfo.mem->alloc_sarray)((j_common_ptr) &dinfo, JPOOL_IMAGE, row_stride, 1);
            bufferEvenScanLine = (*dinfo.mem->alloc_sarray)((j_common_ptr) &dinfo, JPOOL_IMAGE, row_stride, 1);

            uint32_t half_of_mWidth = mWidth >> 1;
            uint32_t half_of_mHeight = mHeight >> 1;

            uint8_t* dst_Y_ptr = dst;
            uint8_t* dst_U_ptr = dst_Y_ptr + (mWidth * mHeight);
            uint8_t* dst_V_ptr = dst_U_ptr + half_of_mWidth * half_of_mHeight;

            uint8_t* buff_Odd_ptr = (uint8_t*)bufferOddScanLine[0];
            uint8_t* buff_Even_ptr = (uint8_t*)bufferEvenScanLine[0];
            uint32_t scanline = half_of_mHeight;
            uint32_t offset = 0;

#ifdef LGE_MJPEG_USE_NEON_REGISTER
            bool extraProcessing = false;
            uint32_t extraLength = 0;

            if (mWidth < 16) {
                ALOGE("SoftMJPG::Decode Fail - Not supported contents size - width=[%u]", mWidth);
                notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
                jpeg_finish_decompress(&dinfo);
                jpeg_destroy_decompress(&dinfo);
                mThreadWork = false;
                return;
            }
            else if (mWidth % 16 != 0) { // Need right Edge processing
                extraProcessing = true;
                extraLength = (mWidth%16) > 1 ? mWidth%16 : 0; // ignore only 1 pixel remaining case
            }
            uint32_t dq_of_mWidth = mWidth;
#endif
            while (scanline-- > 0) {
                if (0 == jpeg_read_scanlines(&dinfo, bufferOddScanLine, 1)) { // put decoded result - Odd line
                    ALOGE("SoftMJPG::Decode Fail - odd_line[%u], scanline=[%u]", dinfo.output_scanline, scanline);
                    notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
                    jpeg_finish_decompress(&dinfo);
                    jpeg_destroy_decompress(&dinfo);
                    mThreadWork = false;
                    return;
                }
                if (0 == jpeg_read_scanlines(&dinfo, bufferEvenScanLine, 1)) { // put decoded result - Even line
                    ALOGE("SoftMJPG::Decode Fail - even_line[%u], scanline=[%u]", dinfo.output_scanline, scanline);
                    notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
                    jpeg_finish_decompress(&dinfo);
                    jpeg_destroy_decompress(&dinfo);
                    mThreadWork = false;
                    return;
                }
#ifdef LGE_MJPEG_USE_NEON_REGISTER
                NeonConvertor(dst_Y_ptr,dst_U_ptr,dst_V_ptr, buff_Odd_ptr,buff_Even_ptr,dq_of_mWidth);

                if (extraProcessing) { // width is not divided by 16
                    offset = dq_of_mWidth - extraLength;
                    for (int i=0; i<extraLength; i++) {
                        dst_Y_ptr[offset + i] = buff_Odd_ptr[3*(offset + i)]; // Y
                    }
                    for (int j=0; j<extraLength; j++) {
                        dst_Y_ptr[dq_of_mWidth + offset + j] = buff_Even_ptr[3*(offset + j)]; // Y
                        if (j%2 == 1) { // process 2 pixels at once - U,V
                            dst_U_ptr[(dq_of_mWidth >> 1) - (extraLength >> 1) + (j >> 1)] =
                                (buff_Odd_ptr[3*(offset - 1 + j) + 1] + buff_Odd_ptr[3*(offset + j) + 1] + buff_Even_ptr[3*(offset - 1 + j) + 1] + buff_Even_ptr[3*(offset + j) + 1]) >> 2; // U
                            dst_V_ptr[(dq_of_mWidth >> 1) - (extraLength >> 1) + (j >> 1)] =
                                (buff_Odd_ptr[3*(offset - 1 + j) + 2] + buff_Odd_ptr[3*(offset + j) + 2] + buff_Even_ptr[3*(offset - 1 + j) + 2] + buff_Even_ptr[3*(offset + j) + 2]) >> 2; // V
                        }
                    }
                }
                dst_Y_ptr += (dq_of_mWidth << 1);
                dst_U_ptr += (dq_of_mWidth >> 1);
                dst_V_ptr += (dq_of_mWidth >> 1);
#else
                for (uint32_t col=0; col < half_of_mWidth; col++) { // YUV444 interleaved -> YUV420 planar Color conversion - 2 by 2 filter
                    offset = ((col << 2) + (col << 1));
                    *dst_Y_ptr++ = buff_Odd_ptr[offset]; // Y
                    *dst_Y_ptr++ = buff_Odd_ptr[offset + 3]; // Y
                }
                for (uint32_t col=0; col < half_of_mWidth; col++) { // YUV444 interleaved -> YUV420 planar Color conversion - 2 by 2 filter
                    offset = ((col << 2) + (col << 1));
                    *dst_Y_ptr++ = buff_Even_ptr[offset]; // Y
                    *dst_Y_ptr++ = buff_Even_ptr[offset + 3]; // Y
                    *dst_U_ptr++ = (buff_Odd_ptr[offset + 1] + buff_Odd_ptr[offset + 4] + buff_Even_ptr[offset + 1] + buff_Even_ptr[offset + 4]) >> 2; // U
                    *dst_V_ptr++ = (buff_Odd_ptr[offset + 2] + buff_Odd_ptr[offset + 5] + buff_Even_ptr[offset + 2] + buff_Even_ptr[offset + 5]) >> 2; // V
                }
#endif
            } // while end
        }
        else {
            ALOGE("SoftMJPG::NOT Supported format comp[%d], color[%u]", dinfo.out_color_components, dinfo.out_color_space);
            notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
            break;
        }
        while(!mThreadDecoding && !mThreadKill) { // notify thread's fillbufferdone after main decoding complete
            usleep(5000);
        }
        //ALOGD("SoftMJPG::%s()[%d]:: THREAD notifyEmptyBufferDone [%p]", __FUNCTION__, __LINE__, *inInfo);
        notifyEmptyBufferDone(inHeader);
        //ALOGD("SoftMJPG::%s()[%d]:: THREAD notifyFillBufferDone :: inBuffTimeStamp=[%d]", __FUNCTION__, __LINE__, inBuffTimeStamp);
        notifyFillBufferDone(outHeader);

        jpeg_finish_decompress(&dinfo);
        jpeg_destroy_decompress(&dinfo);
        break;
    }
    mThreadWork = false;
}

#ifdef LGE_MJPEG_USE_NEON_REGISTER
void SoftMJPG::NeonConvertor(uint8_t * __restrict dest1,
                             uint8_t * __restrict dest2,
                             uint8_t * __restrict dest3,
                             uint8_t * __restrict src1,
                             uint8_t * __restrict src2,
                             uint32_t n)
{
    uint32_t i;
    uint32_t loop = n >> 4;
    uint8x16x3_t oddYUV444;
    uint8x16x3_t evenYUV444;
    uint8x16_t buffer_8t;
    uint16x8_t buffer_16t;
    uint8x8_t result;
    uint32_t jumpsize1 = n;
    uint32_t jumpsize2 = n-16; // width is must be bigger than 16
    for(i = 0 ; i < loop ; i ++){
        //ALOGI("%s: count = %d", __FUNCTION__, i);
        //Read Data From Source
        oddYUV444 = vld3q_u8(src1);
        evenYUV444 = vld3q_u8(src2);

        //Store Y data 128bits X 2 = 256bits per one time
        vst1q_u8(dest1, oddYUV444.val[0]);
        dest1 +=jumpsize1;
        vst1q_u8(dest1, evenYUV444.val[0]);
        dest1 -=jumpsize2;

        //Conversion U4 -> U1
        buffer_8t = vrhaddq_u8(oddYUV444.val[1],evenYUV444.val[1]);
        buffer_8t = vshrq_n_u8(buffer_8t,1);
        buffer_16t = vpaddlq_u8(buffer_8t);
        result = vmovn_u16(buffer_16t);
        vst1_u8(dest2, result);
        dest2 +=8;

        //Conversion V4 -> V1
        buffer_8t = vrhaddq_u8(oddYUV444.val[2],evenYUV444.val[2]);
        buffer_8t = vshrq_n_u8(buffer_8t,1);
        buffer_16t = vpaddlq_u8(buffer_8t);
        result = vmovn_u16(buffer_16t);
        vst1_u8(dest3, result);
        dest3 +=8;
    }
}
#endif

void SoftMJPG::onPortFlushCompleted(OMX_U32 portIndex) {
}

void SoftMJPG::onPortEnableCompleted(OMX_U32 portIndex, bool enabled) {
    switch (mOutputPortSettingsChange) {
        case NONE:
            break;

        case AWAITING_DISABLED:
        {
            CHECK(!enabled);
            mOutputPortSettingsChange = AWAITING_ENABLED;
            break;
        }

        default:
        {
            CHECK_EQ((int)mOutputPortSettingsChange, (int)AWAITING_ENABLED);
            CHECK(enabled);
            mOutputPortSettingsChange = NONE;
            break;
        }
    }
}

OMX_ERRORTYPE SoftMJPG::getExtensionIndex(const char *name, OMX_INDEXTYPE *index) {
    if(!strncmp(name,"OMX.google.android.index.getAndroidNativeBufferUsage", sizeof("OMX.google.android.index.getAndroidNativeBufferUsage")-1)) {
        *index = (OMX_INDEXTYPE)OMX_IndexParamMJPEGGetAndroidNativeBufferUsage;
    }
    else if(!strncmp(name,"OMX.google.android.index.storeMetaDataInBuffers", sizeof("OMX.google.android.index.storeMetaDataInBuffers")-1)) {
        *index = (OMX_INDEXTYPE)OMX_IndexParamMJPEGStoreMetaDataInBuffers;
    }
    else if(!strncmp(name,"OMX.google.android.index.enableAndroidNativeBuffers", sizeof("OMX.google.android.index.enableAndroidNativeBuffers")-1)) {
        *index = (OMX_INDEXTYPE)OMX_IndexParamMJPEGEnableAndroidNativeBuffers;
    }
    else if(!strncmp(name,"OMX.google.android.index.useAndroidNativeBuffer2", sizeof("OMX.google.android.index.useAndroidNativeBuffer2")-1)) {
        *index = (OMX_INDEXTYPE)OMX_IndexParamMJPEGUseAndroidNativeBuffer2;
    }
    else {
        ALOGE("SoftMJPG::NOT Implemented type - [%s]", name);
        return OMX_ErrorNotImplemented;
    }
    return OMX_ErrorNone;
}

bool SoftMJPG::isValidFrame(uint8_t *src, uint32_t length) {
    uint32_t checkMalform = 0;
    uint32_t loopCount = length;
    if (src == NULL || length < 4) {
        ALOGE("Null address or invalid data[%u]", length);
        return false;
    }

    switch (src[0]) {
        case 0x00:
            if (src[1] == 0xff && src[2] == 0xd8)
                break;
            else if (src[1] == 0x00 && src[2] == 0xff && src[3] == 0xd8)
                break;
            else
                checkMalform = 1;
                break;
        case 0xff:
            if (src[1] == 0xd8)
                break;
            else
                checkMalform = 1;
                break;
        default:
            checkMalform = 1;
            break;
    }

    if (checkMalform) {
        ALOGE("Malformed frame data(start) - [%x][%x][%x][%x]", src[0], src[1], src[2], src[3]);
        return false;
    }
    else {
        checkMalform = 1;
    }

    for(int i=1; i<loopCount; i++) {
        if (src[length - i] == 0xd9 && src[length - i - 1] == 0xff) {
            checkMalform = 0;
            break;
        }
    }

    if (checkMalform) {
        ALOGE("Malformed frame data(end) - No EOI(0xff 0xd9)");
        return false;
    }

    return true;
}

void SoftMJPG::updatePortDefinitions() {
    OMX_PARAM_PORTDEFINITIONTYPE *def = &editPortInfo(0)->mDef;
    def->format.video.nFrameWidth = mWidth;
    def->format.video.nFrameHeight = mHeight;
    def->format.video.nStride = def->format.video.nFrameWidth;
    def->format.video.nSliceHeight = def->format.video.nFrameHeight;

    def = &editPortInfo(1)->mDef;
    def->format.video.nFrameWidth = mWidth;
    def->format.video.nFrameHeight = mHeight;
    def->format.video.nStride = def->format.video.nFrameWidth;
    def->format.video.nSliceHeight = def->format.video.nFrameHeight;
    def->nBufferSize = (def->format.video.nFrameWidth * def->format.video.nFrameHeight * 3) / 2;

#if 0
    if ((mWidth * mHeight * mFrameRate) >= (MAX_SPECOUT_WIDTH * MAX_SPECOUT_HEIGHT * MAX_SPECOUT_FRAMERATE)) {
        mIsSpecOut = true;
        mIsDownScale = (mWidth * mHeight < MAX_SPECOUT_WIDTH * MAX_SPECOUT_HEIGHT) ? false : true;

        if (mFrameRate > MAX_SPECOUT_FRAMERATE) {
            mSkipRate = mFrameRate / (mFrameRate - MAX_SPECOUT_FRAMERATE);
        }
        else if (mFrameRate == MAX_SPECOUT_FRAMERATE) {
            mSkipRate = 0;
        }
        else {// maybe 1080p case -> converse 15fps
            if (mFrameRate > 15)
                mSkipRate = mFrameRate / (mFrameRate - 15);
            else
                mSkipRate = 0;
        }
    }
    else {
        mIsDownScale = false;
        mIsSpecOut = false;
        mSkipRate = 0;
    }
ALOGI("SoftMJPG:: updatePortDefinitions - mIsSpecOut[%d], mIsDownScale[%d], mSkipRate[%d], mWidth[%u], mHeight[%u], mFrameRate[%d]", mIsSpecOut, mIsDownScale, mSkipRate, mWidth, mHeight, mFrameRate);
#else
    mSkipRate = 0;
ALOGV("SoftMJPG:: updatePortDefinitions - mSkipRate[%d], mWidth[%u], mHeight[%u]", mSkipRate, mWidth, mHeight);
#endif
    mThreadWork = false;
}
}  // namespace android

android::SoftOMXComponent *createSoftOMXComponent(
        const char *name, const OMX_CALLBACKTYPE *callbacks,
        OMX_PTR appData, OMX_COMPONENTTYPE **component) {
    return new android::SoftMJPG(name, callbacks, appData, component);
}
