#define LOG_NDEBUG 0
#define LOG_TAG "InputDevice"
#define LOG_TITLE "InputReader"

#include <cutils/log.h>
#include <cutils/properties.h>

#include "fb_input_reader.h"
#include "events_c.h"

#ifdef __cplusplus
extern "C" {
#endif

#define TOUCH_LOCATION 1

static int mScreenWidth = 0;
static int mScreenHeight = 0;
static int mTouchWidth = DEVICE_TOUCH_X_MAX;
static int mTouchHeight = DEVICE_TOUCH_Y_MAX;
static int mBackupX = -1;
static int mBackupY = -1;

static struct RawEvent mEventBuffer[EVENT_BUFFER_SIZE];

static uint32_t mInputDeviceCount = 0;
static struct InputDevice mDevices[MAX_EVENT_COUNT];

/* Touch */
static struct RawPointerData mCurrentCookedTouchData;
static struct RawPointerData mLastCookedTouchData;

static int32_t mCurrentSlot = 0;
static int32_t mTouchTrackingIdMap[MAX_POINTERS] = {0, };
static Bool mIsValidTouchData = false;
static Bool mIsValidKeyData = false;
static Bool mUsingSlotsProtocol = true;
static struct MultiTouchMotionAccumulator mMultiTouch[MAX_POINTERS] = {0, };

static Bool mIsKeyPressed = false;
static int mPressedKey = -1;
static int mHeadsetDeviceID = -1;
/* Keyboard */
struct KeyboardMap mKeyboardMap[] = {
    {KEY_VOLUMEDOWN,        "down"},
    {KEY_VOLUMEUP,          "up"},
    {KEY_POWER,             "power"},
    {KEY_HOMEPAGE,          "home"},
    {KEY_HOME,              "home"},
    {KEY_BACK,              "escape"},
    {KEY_MENU,              "menu"},
    {KEY_MEDIA,             "hook"},
    {KEY_KBDILLUMTOGGLE,    "dual_sim"},
    {KEY_SEARCH,            "dual_sim"},
    {166,                   "dual_sim"},
    {0xF9,                  "dual_sim"},     // 249
    {0xFA,                  "quick_memo"},    // 250
    {165,                   "quick_memo"},
    {0xFF,                  "touch_key_up"}   // 255
};

static int mCurrentKeyCount = 0;
static struct KeyboardAccumulator mKeyboard[MAX_KEY] = {0, };
/* Switch */
struct SwitchMap mSwitchMap[] = {
    {SW_HEADPHONE_INSERT,   "headset"},
    {SW_MICROPHONE_INSERT,  "headset"},
    {SW_ADVANCED_HEADPHONE_INSERT,  "headset"},
    {SW_AUX_ACCESSORY_INSERT,  "headset"}
};
#if TOUCH_LOCATION
extern Bool TouchLocationManager_SendLocation(int finger, int x, int y, int event);
#endif

/* Function */
static Bool Keyboard_TranslateKey(keysym *keysym, int _index)
{
    int index;
    int count;
    int keycode;
    char *name;

    count = (sizeof(mKeyboardMap) / sizeof(struct KeyboardMap));
    for(index = 0; index < count; index++) {
        if(mKeyboardMap[index].keyCode == mKeyboard[_index].keyCode) {
            name = mKeyboardMap[index].name;
            break;
        }
    }

    if(index != count) {
        keycode = GetKeyCode(name);
        if(keycode >= 0) {
            keysym->scancode = mKeyboard[_index].keyCode;
            keysym->sym = keycode;
            keysym->mod = KMOD_NONE;
            keysym->unicode = 0;
            return true;
        }
        else {
            ALOGD("[%s] Unknown key code: %d\n", LOG_TITLE, keycode);
        }
    }
    else {
        ALOGD("[%s] Unsupported key code: %d\n", LOG_TITLE, keycode);
    }

    return false;

}


static void Keyboard_Dispatch()
{
    int index;
    keysym keysym;

    for(index = 0; index < mCurrentKeyCount; index++) {
        if(mKeyboard[index].flag) {
            if(Keyboard_TranslateKey(&keysym, index)) {
                if((mKeyboard[index].value == 0) || (mKeyboard[index].value == 0xFF)) {
                        PrivateKeyboard(RELEASED, &keysym);
                        ALOGD("[%s] Keyboard_Dispatch %d release", LOG_TITLE, mKeyboard[index].keyCode);
                }
                else {
                        PrivateKeyboard(PRESSED, &keysym);
                        ALOGD("[%s] Keyboard_Dispatch %d pressed", LOG_TITLE, mKeyboard[index].keyCode);
                }
            }
        }
    }
}

static Bool Keyboard_CheckValue(int32_t keycode, int32_t value)
{
    int32_t index;
    int32_t count;
    count = (sizeof(mKeyboardMap) / sizeof(struct KeyboardMap));

    for(index = 0; index < count; index++) {
        if(mKeyboardMap[index].keyCode == keycode) {
            if((value == 0) || (value == 1) || (value == 0xFF) || (value == 0x4000000)) {
                return true;
            }
        }
    }

    return false;
}

static void Keyboard_Process(const struct RawEvent* rawEvent)
{
    //ALOGD("[%s] Keyboard_Process, rawEvent->code: %d, rawEvent->value: %d", LOG_TITLE, rawEvent->code, rawEvent->value);
    if (rawEvent->type == EV_SYN && rawEvent->code == SYN_REPORT) {
        Keyboard_Dispatch();
        mCurrentKeyCount = 0;
        memset((void *)mKeyboard, 0x00, (sizeof(struct KeyboardAccumulator) * MAX_KEY));
    }

    if (rawEvent->type == EV_KEY) {
        if(Keyboard_CheckValue(rawEvent->code, rawEvent->value)) {
            mKeyboard[mCurrentKeyCount].keyCode = rawEvent->code;
            if(rawEvent->value == 0xFF) {
                mKeyboard[mCurrentKeyCount].value = 0x00;
                mKeyboard[mCurrentKeyCount].flag = true;
                mCurrentKeyCount++;

                Keyboard_Dispatch();
                mCurrentKeyCount = 0;
                memset((void *)mKeyboard, 0x00, (sizeof(struct KeyboardAccumulator) * MAX_KEY));
            }
            else {
                mKeyboard[mCurrentKeyCount].value = rawEvent->value;
                mKeyboard[mCurrentKeyCount].flag = true;
                mCurrentKeyCount++;
            }

            if(mCurrentKeyCount >= MAX_KEY) {
                Keyboard_Dispatch();
                mCurrentKeyCount = 0;
                memset((void *)mKeyboard, 0x00, (sizeof(struct KeyboardAccumulator) * MAX_KEY));
            }
        }
        else {
            ALOGW("[%s] Not supported key code", LOG_TITLE);
        }
    }
}

static void MultiTouch_SendEvent(int event, int touchstate, int finger, int x, int y)
{
#if PERFORM_GUI_ROTATION
    int rotate_data = mTouchHeight - y;
    y = CONVERT_COORDINATE(x, mScreenHeight, mTouchHeight);
    x = CONVERT_COORDINATE(rotate_data, mScreenWidth, mTouchWidth);
    if(x >= mScreenWidth) {
        x = mScreenWidth - 1;
    }
    if(y >= mScreenHeight) {
        y = mScreenHeight - 1;
    }
#else
    x = CONVERT_COORDINATE(x, mScreenWidth, mTouchWidth);
    y = CONVERT_COORDINATE(y, mScreenHeight, mTouchHeight);
    if(x >= mScreenWidth) {
        x = mScreenWidth - 1;
    }
    if(y >= mScreenHeight) {
        y = mScreenHeight - 1;
    }
#endif

#if TOUCH_LOCATION
    TouchLocationManager_SendLocation(finger, x, y, event);
#endif

    switch(event) {
        case TOUCH_RELEASE :
        case TOUCH_DOWN :
            mBackupX = -1;
            mBackupY = -1;
            PrivateMouseButton(touchstate, finger, x, y);
            break;
        case TOUCH_MOVE :
            if((mBackupX != x) || (mBackupY != y)) {
                mBackupX = x;
                mBackupY = y;
                PrivateMouseMotion(touchstate, finger, x, y);
            }
            break;
        default :
            break;
    }
}

static void MultiTouch_Dispatch()
{
    uint32_t id;
    uint32_t outCount;
    uint32_t currentIdBits = mCurrentCookedTouchData.touchingIdBits;
    uint32_t lastIdBits = mLastCookedTouchData.touchingIdBits;

    if (currentIdBits == lastIdBits) {
        // Dispatch pointer move events using the new pointer locations.
        while (currentIdBits != 0) {
            for(id = 0; id < MAX_POINTERS; id++)
            {
                if(currentIdBits & (1 << id))
                {
                    outCount = mCurrentCookedTouchData.idToIndex[id];
                    MultiTouch_SendEvent(TOUCH_MOVE, 1, id, mCurrentCookedTouchData.pointers[outCount].x, mCurrentCookedTouchData.pointers[outCount].y);
                    ALOGD("[%s][Touch] move[%d] id:%d  x:%d  y:%d\n", LOG_TITLE, id, mTouchTrackingIdMap[id], mCurrentCookedTouchData.pointers[outCount].x, mCurrentCookedTouchData.pointers[outCount].y);
                    currentIdBits &= ~(1 << id);
                }
            }
        }
    } else {
        // There may be pointers going up and pointers going down and pointers moving all at the same time.
        uint32_t upIdBits = (lastIdBits & ~currentIdBits);
        uint32_t downIdBits = (currentIdBits & ~lastIdBits);
        uint32_t moveIdBits = (lastIdBits & currentIdBits);

        // Dispatch pointer up events.
        while (upIdBits != 0) {
            for(id = 0; id < MAX_POINTERS; id++)
            {
                if(upIdBits & (1 << id))
                {
                    outCount = mLastCookedTouchData.idToIndex[id];
                    MultiTouch_SendEvent(TOUCH_RELEASE, RELEASED, id, mLastCookedTouchData.pointers[outCount].x, mLastCookedTouchData.pointers[outCount].y);
                    ALOGD("[%s][Touch] up[%d] id:%d  x:%d  y:%d\n", LOG_TITLE, id, mTouchTrackingIdMap[id], mLastCookedTouchData.pointers[outCount].x, mLastCookedTouchData.pointers[outCount].y);
                    upIdBits &= ~(1 << id);
                }
            }
        }

        // Dispatch pointer move events using the new pointer locations.
        while (moveIdBits != 0) {
            for(id = 0; id < MAX_POINTERS; id++)
            {
                if(moveIdBits & (1 << id))
                {
                    outCount = mCurrentCookedTouchData.idToIndex[id];
                    MultiTouch_SendEvent(TOUCH_MOVE, 1, id, mCurrentCookedTouchData.pointers[outCount].x, mCurrentCookedTouchData.pointers[outCount].y);
                    ALOGD("[%s][Touch] move[%d] id:%d  x:%d  y:%d\n", LOG_TITLE, id, mTouchTrackingIdMap[id], mCurrentCookedTouchData.pointers[outCount].x, mCurrentCookedTouchData.pointers[outCount].y);
                    moveIdBits &= ~(1 << id);
                }
            }
        }

        // Dispatch pointer down events using the new pointer locations.
        while (downIdBits != 0) {
            for(id = 0; id < MAX_POINTERS; id++)
            {
                if(downIdBits & (1 << id))
                {
                    outCount = mCurrentCookedTouchData.idToIndex[id];
                    MultiTouch_SendEvent(TOUCH_DOWN, PRESSED, id, mCurrentCookedTouchData.pointers[outCount].x, mCurrentCookedTouchData.pointers[outCount].y);
                    ALOGD("[%s][Touch] down[%d] id:%d  x:%d  y:%d\n", LOG_TITLE, id, mTouchTrackingIdMap[id], mCurrentCookedTouchData.pointers[outCount].x, mCurrentCookedTouchData.pointers[outCount].y);
                    downIdBits &= ~(1 << id);
                }
            }
        }
    }

    return;
}

static void MultiTouch_Sync()
{
    int32_t inIndex = 0;
    uint32_t outCount = 0;
    int32_t id, offset, trackingId;

    struct Pointer *outPointer;
    struct MultiTouchMotionAccumulator* inSlot;

    mCurrentCookedTouchData.pointerCount = 0;
    mCurrentCookedTouchData.touchingIdBits = 0;

    for(inIndex = 0; inIndex <MAX_POINTERS ; inIndex++) {
        inSlot = &mMultiTouch[inIndex];

        if (!inSlot->mInUse) {
            continue;
        }

        if (outCount >= MAX_POINTERS) {
            ALOGD("[%s][Touch] MultiTouch device limitted more than maximum of %d pointers.\n", LOG_TITLE, MAX_POINTERS);
            break;
        }

        id = -1;
        trackingId = inSlot->mAbsMTTrackingId;

        if (trackingId >= 0) {
            for (offset = 0; offset < MAX_POINTERS; offset++) {
                if (mTouchTrackingIdMap[offset] == trackingId) {
                    id = offset;
                    break;
                }
            }

            if (id < 0) {
                if(!mUsingSlotsProtocol) {
                    id = trackingId;
                    if(id < MAX_POINTERS) {
                        mTouchTrackingIdMap[id] = trackingId;
                    }
                    else {
                        ALOGD("[%s][Touch] Invalid tracking id : %d\n", LOG_TITLE, id);
                        continue;
                    }
                }
                else {
                    id = inIndex;
                    mTouchTrackingIdMap[id] = trackingId;
                }
            }
        }

        outPointer = (struct Pointer *)&mCurrentCookedTouchData.pointers[outCount];
        outPointer->x = inSlot->mAbsMTPositionX;
        outPointer->y = inSlot->mAbsMTPositionY;
        outPointer->pressure = inSlot->mAbsMTPressure;
        outPointer->touchMajor = inSlot->mAbsMTTouchMajor;
        outPointer->touchMinor = inSlot->mAbsMTTouchMinor;
        outPointer->toolMajor = inSlot->mAbsMTWidthMajor;
        outPointer->toolMinor = inSlot->mAbsMTWidthMinor;
        outPointer->orientation = inSlot->mAbsMTOrientation;
        outPointer->distance = inSlot->mAbsMTDistance;
        outPointer->tiltX = 0;
        outPointer->tiltY = 0;
        outPointer->id = id;

        mCurrentCookedTouchData.idToIndex[id] = outCount;
        mCurrentCookedTouchData.touchingIdBits |= (1 << id);

        outCount += 1;
    }

    mCurrentCookedTouchData.pointerCount = outCount;

    if(!mUsingSlotsProtocol) {
        mCurrentSlot = -1;
        for(inIndex = 0; inIndex < MAX_POINTERS; inIndex++) {
            mMultiTouch[inIndex].mInUse = false;
            mMultiTouch[inIndex].mHaveAbsMTTouchMinor = false;
            mMultiTouch[inIndex].mHaveAbsMTWidthMinor = false;
            mMultiTouch[inIndex].mHaveAbsMTToolType = false;
            mMultiTouch[inIndex].mAbsMTPositionX = 0;
            mMultiTouch[inIndex].mAbsMTPositionY = 0;
            mMultiTouch[inIndex].mAbsMTTouchMajor = 0;
            mMultiTouch[inIndex].mAbsMTTouchMinor = 0;
            mMultiTouch[inIndex].mAbsMTWidthMajor = 0;
            mMultiTouch[inIndex].mAbsMTWidthMinor = 0;
            mMultiTouch[inIndex].mAbsMTOrientation = 0;
            mMultiTouch[inIndex].mAbsMTTrackingId = -1;
            mMultiTouch[inIndex].mAbsMTPressure = 0;
            mMultiTouch[inIndex].mAbsMTDistance = 0;
            mMultiTouch[inIndex].mAbsMTToolType = 0;
        }
    }
}

static void MultiTouch_Process(const struct RawEvent* rawEvent)
{
    int inIndex;
    struct MultiTouchMotionAccumulator* slot;

    if ((rawEvent->type == EV_SYN) && (rawEvent->code == SYN_REPORT)) {
        if(!mIsValidTouchData) {
            for(inIndex = 0; inIndex <MAX_POINTERS ; inIndex++) {
                slot = &mMultiTouch[inIndex];
                if(!slot->mInUse) {
                    continue;
                }
                else {
                    if((slot->mAbsMTPositionX != 0) && (slot->mAbsMTPositionY != 0)) {
                        mIsValidTouchData = true;
                        break;
                    }
                }
            }
        }

        if(mIsValidTouchData) {
            MultiTouch_Sync();
            MultiTouch_Dispatch();
            memcpy((void *)&mLastCookedTouchData, (void *)&mCurrentCookedTouchData, sizeof(struct RawPointerData));
        }

        if (mIsValidKeyData) {
            Keyboard_Process(rawEvent);
            mIsValidKeyData = false;
        }
    }

    if (rawEvent->type == EV_ABS) {
        Bool newSlot = false;
        if (rawEvent->code == ABS_MT_SLOT) {
            mCurrentSlot = rawEvent->value;
            newSlot = true;
        }
        else if (mCurrentSlot < 0) {
            mCurrentSlot = 0;
        }

        if ((mCurrentSlot < 0) || (mCurrentSlot >= MAX_POINTERS)) {
            if (newSlot) {
                ALOGD("[%s] MultiTouch device emitted invalid slot index %d but it should be between 0 and %d; ignoring this slot.", LOG_TITLE, mCurrentSlot, (MAX_POINTERS - 1));
            }
        }
        else {
            slot = &mMultiTouch[mCurrentSlot];
            switch (rawEvent->code) {
                case ABS_MT_POSITION_X:
                    slot->mInUse = true;
                    slot->mAbsMTPositionX = rawEvent->value;
                    break;
                case ABS_MT_POSITION_Y:
                    slot->mInUse = true;
                    slot->mAbsMTPositionY = rawEvent->value;
                    break;
                case ABS_MT_TOUCH_MAJOR:
                    slot->mInUse = true;
                    slot->mAbsMTTouchMajor = rawEvent->value;
                    break;
                case ABS_MT_TOUCH_MINOR:
                    slot->mInUse = true;
                    slot->mAbsMTTouchMinor = rawEvent->value;
                    slot->mHaveAbsMTTouchMinor = true;
                    break;
                case ABS_MT_WIDTH_MAJOR:
                    slot->mInUse = true;
                    slot->mAbsMTWidthMajor = rawEvent->value;
                    break;
                case ABS_MT_WIDTH_MINOR:
                    slot->mInUse = true;
                    slot->mAbsMTWidthMinor = rawEvent->value;
                    slot->mHaveAbsMTWidthMinor = true;
                    break;
                case ABS_MT_ORIENTATION:
                    slot->mInUse = true;
                    slot->mAbsMTOrientation = rawEvent->value;
                    break;
                case ABS_MT_TRACKING_ID:
                    if (mUsingSlotsProtocol && rawEvent->value < 0) {
                        // The slot is no longer in use but it retains its previous contents,
                        // which may be reused for subsequent touches.
                        slot->mInUse = false;
                    } else {
                        slot->mInUse = true;
                        slot->mAbsMTTrackingId = rawEvent->value;
                    }
                    break;
                case ABS_MT_PRESSURE:
                    slot->mInUse = true;
                    slot->mAbsMTPressure = rawEvent->value;
                    break;
                case ABS_MT_DISTANCE:
                    slot->mInUse = true;
                    slot->mAbsMTDistance = rawEvent->value;
                    break;
                case ABS_MT_TOOL_TYPE:
                    slot->mInUse = true;
                    slot->mAbsMTToolType = rawEvent->value;
                    slot->mHaveAbsMTToolType = true;
                    break;
            }
        }
    }
    else if (rawEvent->type == EV_SYN && rawEvent->code == SYN_MT_REPORT) {
        // MultiTouch Sync: The driver has returned all data for *one* of the pointers.
        mUsingSlotsProtocol = false;
        mCurrentSlot += 1;
    }
    else if (rawEvent->type == EV_KEY) {
        if (rawEvent->code == BTN_TOUCH)
        {
        }
        else if (rawEvent->code == BTN_TOOL_FINGER)
        {
        }
        else
        {
            mIsValidKeyData = true;
            Keyboard_Process(rawEvent);
        }
    }
}
static Bool Switch_CheckValue(int32_t keycode, int32_t value)
{
    //ALOGD("[%s] Switch_CheckValue", LOG_TITLE);
    int32_t index;
    int32_t count;
    count = (sizeof(mSwitchMap) / sizeof(struct SwitchMap));

    for(index = 0; index < count; index++) {
        if(mSwitchMap[index].switchCode == keycode) {
            if((value == 0) || (value == 1) || (value == 0xFF)) {
                return true;
            }
        }
    }
    return false;
}

static void Switch_Dispatch(int32_t code,int32_t value)
{
    //ALOGD("[%s] Switch_Dispatch", LOG_TITLE);
    Event_t event;
    memset(&event, 0, sizeof(event));

    event.type = USEREVENT;
    if ( ProcessEvents[event.type] == ENABLE ) {
        event.user.type = USEREVENT;
        event.user.code = code;
        event.user.state = value;

        if ( (EventOK == NULL) || EventOK(&event) ) {
            ALOGD("[%s] Switch_Dispatch, code: %d, state: %d", LOG_TITLE, event.user.code, event.user.state);
            PushEvent(&event);
        }
    }
}

static void Switch_Process(const struct RawEvent* rawEvent)
{
    //ALOGD("[%s] Switch_Process, rawEvent->code: %d, rawEvent->value: %d", LOG_TITLE, rawEvent->code, rawEvent->value);
    if (rawEvent->type == EV_SW) {
        if(Switch_CheckValue(rawEvent->code, rawEvent->value))
            Switch_Dispatch(rawEvent->code, rawEvent->value);
    }
/* if hook key event is used headset device, need to implement
    if (rawEvent->type == EV_KEY) {
        //Keyboard_Process(rawEvent);
        // or
        //PrivateKeyboard(RELEASED, &keysym);
        //PrivateKeyboard(PRESSED, &keysym);
    }
*/
}

static void AddInputDevice(int32_t deviceId, uint32_t classes)
{
    if (deviceId >= MAX_EVENT_COUNT) {
        ALOGD("[%s] Out of memory\n", LOG_TITLE);
        return;
    }

    mInputDeviceCount = (deviceId + 1);
    memset((void*)&mDevices[deviceId], 0x00, sizeof(struct InputDevice));
    if (classes & INPUT_DEVICE_CLASS_SWITCH) {
        // headset-like devices.
        mDevices[deviceId].mId = deviceId;
        mDevices[deviceId].mClasses = INPUT_DEVICE_CLASS_SWITCH;
        mDevices[deviceId].process = Switch_Process;
        mHeadsetDeviceID = deviceId;
    }
    else if (classes & INPUT_DEVICE_CLASS_TOUCH_MT) {
        // Touchscreens and touchpad devices.
        mDevices[deviceId].mId = deviceId;
        mDevices[deviceId].mClasses = INPUT_DEVICE_CLASS_TOUCH_MT;
        mDevices[deviceId].process = MultiTouch_Process;
    }
    else if (classes & INPUT_DEVICE_CLASS_KEYBOARD) {
        // Keyboard-like devices.
        mDevices[deviceId].mId = deviceId;
        mDevices[deviceId].mClasses = INPUT_DEVICE_CLASS_KEYBOARD;
        mDevices[deviceId].process = Keyboard_Process;
    }
    else {
        mDevices[deviceId].mId = deviceId;
        mDevices[deviceId].mClasses = 0;
        mDevices[deviceId].process = NULL;
        ALOGD("[%s] Unknown device %d, class 0x%08X\n", LOG_TITLE, deviceId, classes);
    }
}

static void RemoveInputDevice(int32_t deviceId)
{
    int count;

    if(deviceId < mInputDeviceCount) {
        switch (mDevices[deviceId].mClasses) {
            case INPUT_DEVICE_CLASS_KEYBOARD :
                ResetKeyboard();
                break;
            case INPUT_DEVICE_CLASS_TOUCH_MT :
                ResetMouse();
                break;
            default:
                break;
        }

        count = (mInputDeviceCount - deviceId - 1);
        memset((void *)&mDevices[deviceId], 0x00, sizeof(struct InputDevice));
        memcpy((void *)&mDevices[deviceId], (void *)&mDevices[deviceId + 1], (sizeof(struct InputDevice) * count));
        memset((void *)&mDevices[mInputDeviceCount -1], 0x00, sizeof(struct InputDevice));
        mInputDeviceCount--;
    }
    else {
        ALOGD("[%s] Unknown device for remove\n", LOG_TITLE);
    }
}

static void RemoveAllInputDevice()
{
    int index;

    for(index = (mInputDeviceCount -1); index >= 0; index--) {
        FB_CloseDevice(index);
        RemoveInputDevice(index);
    }
}

static void ProcessEventsForInputDevice(int32_t deviceId, struct RawEvent* rawEvents, size_t count)
{
    size_t index;
    struct RawEvent* current_rawEvents = rawEvents;

    for (index = 0; index < count; index++) {
        if (mDevices[deviceId].process != NULL) {
            mDevices[deviceId].process(current_rawEvents);
            current_rawEvents += 1;
        }
    }
}

static void FB_processEvents(const struct RawEvent* rawEvents, size_t count)
{
    struct RawEvent* rawEvent;

    rawEvent = rawEvents;

    while (count > 0) {
        size_t batchSize = 1;
        if (rawEvent->type < FIRST_SYNTHETIC_EVENT) {
            int32_t deviceId = rawEvent->deviceId;
            while (batchSize < count) {
                if ((rawEvent[batchSize].type >= FIRST_SYNTHETIC_EVENT) || (rawEvent[batchSize].deviceId != deviceId)) {
                    break;
                }
                batchSize += 1;
            }

            ProcessEventsForInputDevice(deviceId, rawEvent, batchSize);
        }
        else {
            switch (rawEvent->type) {
                case DEVICE_ADDED:
                    ALOGD("[%s] Add event id: %d, class: %d\n", LOG_TITLE, rawEvent->deviceId, rawEvent->classes);
                    AddInputDevice(rawEvent->deviceId, rawEvent->classes);
                    break;
                case DEVICE_REMOVED:
                    ALOGD("[%s] Remove event id: %d, class: %d\n", LOG_TITLE, rawEvent->deviceId, rawEvent->classes);
                    RemoveInputDevice(rawEvent->deviceId);
                    break;
                case FINISHED_DEVICE_SCAN:
                    break;
                default:
                    ALOGD("[%s] Invalid event type\n", LOG_TITLE);
                    break;
            }
        }
        count -= batchSize;
        rawEvent += batchSize;
    }
}

void InitHeadsetState()
{
    int nHeadphone = FB_GetHeadsetStatus(mHeadsetDeviceID, SW_HEADPHONE_INSERT);
    int nMicrophone = FB_GetHeadsetStatus(mHeadsetDeviceID, SW_MICROPHONE_INSERT);
    int nAdvancedHeadphone = FB_GetHeadsetStatus(mHeadsetDeviceID, SW_ADVANCED_HEADPHONE_INSERT);
    int nAuxAccessory = FB_GetHeadsetStatus(mHeadsetDeviceID, SW_AUX_ACCESSORY_INSERT);

    if(nHeadphone == 1) {
        Switch_Dispatch(SW_HEADPHONE_INSERT, 1);
        ALOGD("[%s] Switch_Dispatch(SW_HEADPHONE_INSERT, 1)", LOG_TITLE);
    }
    if(nAdvancedHeadphone == 1) {
        ALOGD("[%s] Switch_Dispatch(SW_ADVANCED_HEADPHONE_INSERT, 1)", LOG_TITLE);
        Switch_Dispatch(SW_ADVANCED_HEADPHONE_INSERT, 1);
    }

    if(nAuxAccessory == 1) {
        ALOGD("[%s] Switch_Dispatch(SW_AUX_ACCESSORY_INSERT, 1)", LOG_TITLE);
        Switch_Dispatch(SW_AUX_ACCESSORY_INSERT, 1);
    }
    if(nMicrophone == 1) {
        Switch_Dispatch(SW_MICROPHONE_INSERT, 1);
        ALOGD("[%s] Switch_Dispatch(SW_MICROPHONE_INSERT, 1)", LOG_TITLE);
    }
}

int FB_InputReader_Init()
{
    int count, result, index = 0;
    int deviceId = -1;

    memset((void *)mDevices, 0x00, (sizeof(struct InputDevice) * MAX_EVENT_COUNT));
    memset((void *)mEventBuffer, 0x00, (sizeof(struct RawEvent) * EVENT_BUFFER_SIZE));

    count = FB_GetEvent_Init(mEventBuffer);

    if (count > 0) {
        FB_processEvents(mEventBuffer, count);
    }

    for(index=0; index<mInputDeviceCount; index++) {
        if(mDevices[index].mClasses == INPUT_DEVICE_CLASS_TOUCH_MT) {
            deviceId = mDevices[index].mId;
            break;
        }
    }

    if(deviceId > 0) {
        result = getAbsoluteAxisInfo(deviceId, ABS_MT_POSITION_X);
        if(result > 0) {
            mTouchWidth = result;
        }
        else {
            ALOGD("[%s] An error occured, when get touch max width", LOG_TITLE);
        }

        result = getAbsoluteAxisInfo(deviceId, ABS_MT_POSITION_Y);
        if(result> 0) {
            mTouchHeight = result;
        }
        else {
            ALOGD("[%s] An error occured, when get touch max height", LOG_TITLE);
        }
    }
    else {
        ALOGD("[%s] Can,t find touch devices, so remove all input device", LOG_TITLE);
        RemoveAllInputDevice();
        return -1;
    }

    getDisplaySize(&mScreenWidth, &mScreenHeight, 0);
    ALOGD("[%s] using screen width:%d, screen height:%d, touch width:%d, touch height:%d", LOG_TITLE, mScreenWidth, mScreenHeight, mTouchWidth, mTouchHeight);

    return 0;
}

void FB_InputReader_Handle()
{
    int count;

    memset((void *)mEventBuffer, 0x00, (sizeof(struct RawEvent) * EVENT_BUFFER_SIZE));

    count = FB_GetEvent_Handle(mEventBuffer);

    if(count > 0) {
        FB_processEvents(mEventBuffer, count);
    }
}


#ifdef __cplusplus
}
#endif
