/*
***************************************************************************
** SkFontHost_android_hy_loadfonts.cpp
***************************************************************************
**
** 2013.10.08 Android KK
**
** Mobile Communication R&D Center, Hanyang
** Sangmin, Lee (TMSword) ( Mobile Communication R&D Center / Senior Researcher )
**
** This code is a program that changes the font.
**
***************************************************************************
*/
// CAPP_FONTS_HYFONTS Global Font Manager
#ifdef ON_CAPP_FONTS  // This added to disable CAPP_FONTS to  avoid crash for fonts added in system_fonts.xml without serif on TOP.

#ifdef CAPP_FONTS
#include <errno.h>
#include "SkFontHost.h"

#include "SkFontHost_android_hy_types.cpp"
#include "SkFontHost_android_hy_debug.cpp"
#include "SkFontHost_android_hy_api.cpp"

// Font Data
class SkFontData
{
public:
    SkFontData();
    ~SkFontData();

public:
    void init();
    void reset();
    bool checkCertify( bool system );
    bool setCertify( bool certified );
    void setFamily( FamilyRec * family );
    void setFamilyID( FamilyRecID familyID ); // Android Default Font System, Family ID
    FamilyRec * getFamily();
    FamilyRecID getFamilyID();
    void getFamilies( SkTDArray<FontFamily*> &families );
    void loadFont();
    void getFullPathDownloadFont( SkString* fullpath, const char * name );

    void setFontAppName( const char * fontAppName );
    const char * getFontAppName();
    void setFontFullPath( const char * fontFullPath );
    const char * getFontFullPath();
    void setFontDatPath( const char * fontDatPath );
    const char * getFontDatPath();
    void setFontXmlPath( const char * fontXmlPath );
    const char * getFontXmlPath();
    void setWebFaceName( const char * webFaceName );
    const char * getWebFaceName();
    void setFontName( const char * fontName );
    const char * getFontName();
    void readPreData( SkStream * stream );
    void writePreData( SkFILEWStream * stream );
    void deleteFiles();

    // FCI
    void setFCI( void * FCI );
    void * getFCI();

private:
    bool mCertified;
    FamilyRec * mFamily;
    FamilyRecID mFamilyID;

    // Information.
    char mFontFullPath[FULLPATH_LEN];
    char mWebFaceName[WEBFACE_NAME_LEN];

    // DL font only.
    char mFontAppName[DL_FONTAPP_NAME_LEN];
    char mFontName[DL_FONT_NAME_LEN];
    char mFontDatPath[FULLPATH_LEN];
    char mFontXmlPath[FULLPATH_LEN];

    // FCI Pointer
    void * mFCI;
};

SkFontData::SkFontData()
{
    init();
}

SkFontData::~SkFontData()
{

}

void
SkFontData::init()
{
    mCertified = false;
    mFamily = NULL;
    mFamilyID = INVALID_FAMILY_REC_ID;

    memset( mFontFullPath, 0x00, FULLPATH_LEN );
    memset( mWebFaceName, 0x00, WEBFACE_NAME_LEN );
    memset( mFontAppName, 0x00, DL_FONTAPP_NAME_LEN );
    memset( mFontName, 0x00, DL_FONT_NAME_LEN );
    memset( mFontDatPath, 0x00, FULLPATH_LEN );
    memset( mFontXmlPath, 0x00, FULLPATH_LEN );

    mFCI = NULL;
}

void
SkFontData::reset()
{
    init();
}

bool
SkFontData::checkCertify( bool system )
{
    SkString fullPathFontApp;
    SkStream * stFontApp = NULL;
    SkStream * stFont = NULL;
    SkStream * stDat = NULL;
    SkStream * stXml = NULL;
    skhycertified _hy_dat;

    fullPathFontApp.set( SYSTEM_DL_FONTAPP_SRC_DIR );
    fullPathFontApp.append( mFontAppName );

    if( system == true )
    {
        if( ( stFontApp = openReadStream( fullPathFontApp.c_str() ) ) == NULL )
        {
            HyLogif( "%s, not exist", fullPathFontApp.c_str() );
            goto ERROR0;
        }
        SkDELETE( stFontApp );
        stFontApp = NULL;
    }

    if( ( stFont = openReadStream( mFontFullPath ) ) == NULL )
    {
        HyLogif( "%s, not exist", mFontFullPath  );
        goto ERROR0;
    }
    SkDELETE( stFont );
    stFont = NULL;

    if( ( stXml = openReadStream( mFontXmlPath ) ) == NULL )
    {
        HyLogif( "%s, not exist", mFontXmlPath  );
        goto ERROR0;
    }
    SkDELETE( stXml );
    stXml = NULL;

    if( ( stDat = openReadStream( mFontDatPath ) ) == NULL )
    {
        HyLogif( "%s, not exist", mFontDatPath  );
        goto ERROR0;
    }

    if( __certified_check( &_hy_dat, stDat, mFontFullPath ) == false )
    {
        HyLogif( "%s, verification fails", fullPathFontApp.c_str() );
        goto ERROR0;
    }

    memcpy( mFontName, _hy_dat.key7_str, ( _hy_dat.key7 + 1 ) );
    __done_dat( &_hy_dat );
    SkDELETE( stDat );

    return setCertify( true );

ERROR0:
    if( stFontApp )
        SkDELETE( stFontApp );

    if( stFont )
        SkDELETE( stFont );

    if( stDat )
        SkDELETE( stDat );

    if( stXml )
        SkDELETE( stXml );

    return setCertify( false );
}

bool
SkFontData::setCertify( bool certified )
{
    return ( mCertified = certified );
}

void
SkFontData::setFamily( FamilyRec * family )
{
    mFamily = family;
}

void
SkFontData::setFamilyID( FamilyRecID familyID )
{
    mFamilyID = familyID;
}

FamilyRec *
SkFontData::getFamily()
{
    return mFamily;
}

FamilyRecID
SkFontData::getFamilyID()
{
    return mFamilyID;
}

void
SkFontData::getFamilies( SkTDArray<FontFamily*> &families )
{
    SkFontConfigParser::GetTestFontFamilies( families, mFontXmlPath, NULL );
}

void
SkFontData::getFullPathDownloadFont( SkString* fullpath, const char * name )
{
    fullpath->set( SYSTEM_DL_FONTAPP_DST_DIR );
    fullpath->append( mFontAppName );
    fullpath->append( "/" );
    fullpath->append( name );
}

void
SkFontData::setFontAppName( const char * fontAppName )
{
    if( !fontAppName )
        return;
    strncpy( mFontAppName, fontAppName, strlen( fontAppName ) );
}

const char *
SkFontData::getFontAppName()
{
    return ( const char * )mFontAppName;
}

void
SkFontData::setFontFullPath( const char * fontFullPath )
{
    if( !fontFullPath )
        return;
    strncpy( mFontFullPath, fontFullPath, strlen( fontFullPath ) );
}

const char *
SkFontData::getFontFullPath()
{
    return ( const char * )mFontFullPath;
}

void
SkFontData::setFontDatPath( const char * fontDatPath )
{
    if( !fontDatPath )
        return;
    strncpy( mFontDatPath, fontDatPath, strlen( fontDatPath ) );
}

const char *
SkFontData::getFontDatPath()
{
    return ( const char * )mFontDatPath;
}

void
SkFontData::setFontXmlPath( const char * fontXmlPath )
{
    if( !fontXmlPath )
        return;
    strncpy( mFontXmlPath, fontXmlPath, strlen( fontXmlPath ) );
}

const char *
SkFontData::getFontXmlPath()
{
    return ( const char * )mFontXmlPath;
}

void
SkFontData::setWebFaceName( const char * webFaceName )
{
    if( !webFaceName )
        return;
    strncpy( mWebFaceName, webFaceName, strlen( webFaceName ) );
}

const char *
SkFontData::getWebFaceName()
{
    return ( const char * )mWebFaceName;
}

void
SkFontData::setFontName( const char * fontName )
{
    if( !fontName )
        return;
    memcpy( mFontName, fontName, DL_FONT_NAME_LEN );
}

const char *
SkFontData::getFontName()
{
    return ( const char * )mFontName;
}

void
SkFontData::readPreData( SkStream * stream )
{
    stream->read( mFontAppName, DL_FONTAPP_NAME_LEN );
    stream->read( mFontFullPath, FULLPATH_LEN );
    stream->read( mFontDatPath, FULLPATH_LEN );
    stream->read( mFontXmlPath, FULLPATH_LEN );
    stream->read( mFontName, DL_FONT_NAME_LEN );
    stream->read( mWebFaceName, WEBFACE_NAME_LEN );
}

void
SkFontData::writePreData( SkFILEWStream * stream )
{
    stream->write( mFontAppName, DL_FONTAPP_NAME_LEN );
    stream->write( mFontFullPath, FULLPATH_LEN );
    stream->write( mFontDatPath, FULLPATH_LEN );
    stream->write( mFontXmlPath, FULLPATH_LEN );
    stream->write( mFontName, DL_FONT_NAME_LEN );
    stream->write( mWebFaceName, WEBFACE_NAME_LEN );
}

void
SkFontData::deleteFiles()
{
    SkString delCmdFiles;
    delCmdFiles.set( "rm " );
    delCmdFiles.append( SYSTEM_DL_FONTAPP_DST_DIR );
    delCmdFiles.append( mFontAppName );
    delCmdFiles.append( "/*" );
    system( delCmdFiles.c_str() );

    SkString delCmdDir;
    delCmdDir.set( "rmdir " );
    delCmdDir.append( SYSTEM_DL_FONTAPP_DST_DIR );
    delCmdDir.append( mFontAppName );
    system( delCmdDir.c_str() );
 }

class SkFontDataList
{
public:
    SkFontDataList();
    ~SkFontDataList();

public:
    void init();
    bool isMax();
    bool setMaxFonts( uint32_t max );
    void resetFonts();
    void release();
    uint32_t numFonts();
    bool addFonts( SkFontData * font );
    FamilyRec * getFamily( uint32_t idxFont );
    FamilyRecID getFamilyID( uint32_t idxFont );
    SkFontData ** getFonts();
    SkFontData * getFont( uint32_t fontIndex );

private:
    uint32_t mMaxFonts;
    uint32_t mNumFonts;
    SkFontData ** mFonts;
};

SkFontDataList::SkFontDataList()
{
    init();
}

SkFontDataList::~SkFontDataList()
{
    release();
}

void
SkFontDataList::init()
{
    mMaxFonts = 0;
    mNumFonts = 0;
    mFonts = NULL;
}

bool
SkFontDataList::isMax()
{
    return ( ( mMaxFonts > 0 ) ? true : false );
}

bool
SkFontDataList::setMaxFonts( uint32_t maxFonts )
{
    if( maxFonts > 0 )
    {
        if( ( mFonts = ( SkFontData ** )calloc( maxFonts, sizeof( SkFontData * ) ) ) == NULL )
        {
            HyLogef( "Fonts array, alloc()" );
            goto ERROR0;
        }
        mMaxFonts = maxFonts;
        return true;
    }
ERROR0:
    return false;
}

// Clear list.
void
SkFontDataList::resetFonts()
{
    if( isMax() )
    {
        for( uint32_t n = 0 ; n < mNumFonts ; n++ )
        {
            if( mFonts[n] )
            {
                SkDELETE( mFonts[n] );
                mFonts[n] = NULL;
            }
        }
        mNumFonts = 0;
    }
}

// Delete list.
void
SkFontDataList::release()
{
    if( isMax() )
    {
        resetFonts();
        free( mFonts );
        mFonts = NULL;
    }
}

uint32_t
SkFontDataList::numFonts()
{
    return mNumFonts;
}

bool
SkFontDataList::addFonts( SkFontData * font )
{
    if( ( mNumFonts + 1 ) < mMaxFonts )
    {
        mFonts[mNumFonts++] = font;
        return true;
    }
    return false;
}

FamilyRec *
SkFontDataList::getFamily( uint32_t idxFont )
{
    if( idxFont < mNumFonts )
    {
        SkFontData * font = mFonts[idxFont];

        if( font )
            return font->getFamily();
    }
    return NULL;
}

FamilyRecID
SkFontDataList::getFamilyID( uint32_t idxFont )
{
    if( idxFont < mNumFonts )
    {
        SkFontData * font = mFonts[idxFont];

        if( font )
            return font->getFamilyID();
    }
    return INVALID_FAMILY_REC_ID;
}

SkFontData **
SkFontDataList::getFonts()
{
    return mFonts;
}

SkFontData *
SkFontDataList::getFont( uint32_t fontIndex )
{
    return mFonts[fontIndex];
}

// The default font list.
class SkFontEmbeddedList : public SkFontDataList
{
public:
    SkFontEmbeddedList();
    ~SkFontEmbeddedList();
};

SkFontEmbeddedList::SkFontEmbeddedList()
{
    setMaxFonts( SYSTEM_EMBEDDED_FAMILY_MAX );
}

SkFontEmbeddedList::~SkFontEmbeddedList()
{

}

// Verified list of fonts.
class SkFontDownloadList : public SkFontDataList
{
public:
    SkFontDownloadList();
    ~SkFontDownloadList();

public:
    bool isExist( const char * fontAppName );
};

SkFontDownloadList::SkFontDownloadList()
{
    setMaxFonts( SYSTEM_DOWNLOAD_FAMILY_MAX );
}

SkFontDownloadList::~SkFontDownloadList()
{

}

bool
SkFontDownloadList::isExist( const char * fontAppName )
{
    if( numFonts() == 0 )
        return false;
    else
    {
        SkFontData ** fonts = getFonts();

        for( uint32_t n = 0 ; n < numFonts() ; n++ )
            if( !strcmp( fontAppName, fonts[n]->getFontAppName() ) )
                return true;

        return false;
    }
}

// Search the list of fonts.
class SkFontSearchList : public SkFontDataList
{
public:
    SkFontSearchList();
    ~SkFontSearchList();

public:
    void setError( uint32_t error );
    uint32_t getError();
    SearchState setState( SearchState state );
    SearchState getState();

    void ready();
    SearchState search( SkFontDownloadList * dlFonts );

public:
    uint32_t mError;
    SearchState mState;
};

SkFontSearchList::SkFontSearchList()
{
    setMaxFonts( SYSTEM_SEARCH_FAMILY_MAX );
    setError( ERR_NO );
    setState( STATE_DL_FONTAPP_KEEP );
}

SkFontSearchList::~SkFontSearchList()
{

}

void
SkFontSearchList::setError( uint32_t error )
{
    mError = error;
}

uint32_t
SkFontSearchList::getError()
{
    return mError;
}

SearchState
SkFontSearchList::setState( SearchState state )
{
    return ( mState = state );
}

SearchState
SkFontSearchList::getState()
{
    return mState;
}

void
SkFontSearchList::ready()
{
    // Clear list.
    resetFonts();
    setError( ERR_NO );
    setState( STATE_DL_FONTAPP_KEEP );
}

SearchState
SkFontSearchList::search( SkFontDownloadList * dlFonts )
{
    DIR * dp = NULL;
    struct dirent * d = NULL;

    ready();

    if( ( dp = opendir( SYSTEM_DL_FONTAPP_SRC_DIR ) ) == NULL )
    {
        HyLogef( "%s, opendir()", SYSTEM_DL_FONTAPP_SRC_DIR );
        goto ERROR0;
    }

    while( ( d = readdir( dp ) ) != NULL )
    {
        if( !strncmp( d->d_name, ".", 1 ) )
            continue;

        if( !strncmp( d->d_name, DL_FONTAPP_SEARCH_STR, DL_FONTAPP_SEARCH_LEN ) )
        {
            if( dlFonts->isExist( d->d_name ) == true )
                continue;
            else
            {
                SkFontData * font = SkNEW( SkFontData );

                if( font )
                {
                    SkString fontFullPath;
                    SkString fontDatPath;
                    SkString fontXmlPath;

                    font->setFontAppName( d->d_name );

                    fontFullPath.set( SYSTEM_DL_FONTAPP_DST_DIR );
                    fontFullPath.append( font->getFontAppName() );
                    fontFullPath.append( DLFILE_TTF );

                    fontDatPath.set( SYSTEM_DL_FONTAPP_DST_DIR );
                    fontDatPath.append( font->getFontAppName() );
                    fontDatPath.append( DLFILE_DAT );

                    fontXmlPath.set( SYSTEM_DL_FONTAPP_DST_DIR );
                    fontXmlPath.append( font->getFontAppName() );
                    fontXmlPath.append( DLFILE_XML );

                    font->setFontFullPath( fontFullPath.c_str() );
                    font->setFontDatPath( fontDatPath.c_str() );
                    font->setFontXmlPath( fontXmlPath.c_str() );
                    addFonts( font );
                }
                else
                {
                    HyLogef( "Error : SkNEW( SkFontData )." );
                    continue;
                }
            }
        }
    }

    closedir( dp );

    if( numFonts() > 0 )
        return setState( STATE_DL_FONTAPP_FIND );
    else
        return setState( STATE_DL_FONTAPP_KEEP );

ERROR0:
    setError( ERR_YES );
    return setState( STATE_DL_FONTAPP_KEEP );
}

/* --------------------------------------------------------------------------------------
    SkFontHost_fontconfig.cpp, RefFCI()
-------------------------------------------------------------------------------------- */
extern SkFontConfigInterface * getRefFCI();

/* --------------------------------------------------------------------------------------
    SkFontConfigInterface_android.cpp
    class SkFontConfigInterfaceAndroid
-------------------------------------------------------------------------------------- */
class SkFontConfigInterfaceAndroid : public SkFontConfigInterface {
friend class SkFontManager;
friend class SkFontData;

public:
    SkFontConfigInterfaceAndroid( SkTDArray<FontFamily*>& fontFamilies, bool useFontManager ); // Added.
    SkFontConfigInterfaceAndroid(SkTDArray<FontFamily*>& fontFamilies);
    virtual ~SkFontConfigInterfaceAndroid();

    virtual bool matchFamilyName(const char familyName[],
                                 SkTypeface::Style requested,
                                 FontIdentity* outFontIdentifier,
                                 SkString* outFamilyName,
                                 SkTypeface::Style* outStyle) SK_OVERRIDE;
    virtual SkStream* openStream(const FontIdentity&) SK_OVERRIDE;

    // new APIs
    virtual SkDataTable* getFamilyNames() SK_OVERRIDE;
    virtual bool matchFamilySet(const char inFamilyName[],
                                SkString* outFamilyName,
                                SkTArray<FontIdentity>*) SK_OVERRIDE;

    /**
     *  Get the family name of the font in the default fallback font list that
     *  contains the specified chararacter. if no font is found, returns false.
     */
    bool getFallbackFamilyNameForChar(SkUnichar uni, const char* lang, SkString* name);
    /**
     *
     */
    SkTypeface* getTypefaceForChar(SkUnichar uni, SkTypeface::Style style,
                                   SkPaintOptionsAndroid::FontVariant fontVariant);
    SkTypeface* nextLogicalTypeface(SkFontID currFontID, SkFontID origFontID,
                                    const SkPaintOptionsAndroid& options);
    SkTypeface* getTypefaceForGlyphID(uint16_t glyphID, const SkTypeface* origTypeface,
                                      const SkPaintOptionsAndroid& options,
                                      int* lowerBounds, int* upperBounds);

private:
    void addFallbackFamily(FamilyRecID fontRecID);
    SkTypeface* getTypefaceForFontRec(FontRecID fontRecID);
    FallbackFontList* getCurrentLocaleFallbackFontList();
    FallbackFontList* findFallbackFontList(const SkLanguage& lang, bool isOriginal = true);

    SkTArray<FontRec> fFonts;
    SkTArray<FamilyRec> fFontFamilies;
    SkTDict<FamilyRecID> fFamilyNameDict;
    FamilyRecID fDefaultFamilyRecID;

    // (SkLanguage)<->(fallback chain index) translation
    SkTDict<FallbackFontList*> fFallbackFontDict;
    SkTDict<FallbackFontList*> fFallbackFontAliasDict;
    FallbackFontList fDefaultFallbackList;

    // fallback info for current locale
    SkString fCachedLocale;
    FallbackFontList* fLocaleFallbackFontList;
};

static SkFontConfigInterfaceAndroid * getSingletonInterface();

/* --------------------------------------------------------------------------------------
    class FontManager
-------------------------------------------------------------------------------------- */
class SkFontManager
{
friend class SkFontData;

public:
    SkFontManager();
    ~SkFontManager();

public:
    SearchState searchFontApps();
    void initSystemFontsLocked();
    void initDownloadFontsLocked();
    uint32_t getCurrentFontIndex();
    void saveFontIndex( uint32_t fontIndex );
    uint32_t getFamilyCount();
    uint32_t getEmbeddedFamilyCount();
    LoadState loadPreData( uint32_t * newFontIndex, bool boot );
    bool openPreDataRead( SkStream ** preData );
    bool savePreData();

    uint32_t getUpdateVersion();
    void setUpdateVersion( uint32_t updateVersion );
    void updateDefaultNormal();
    SkTypeface * getDefaultTypeface( SkTypeface::Style style );
    FamilyRec * getDefaultFamily();
    const char * getDefaultFamilyName();
    FamilyRec * getFamily( uint32_t fontIndex );
    FamilyRecID getFamilyID( uint32_t fontIndex );
    SkFontData * getFont( uint32_t fontIndex );
    SkFontData * getFont();

    const char * getDownloadFontName( uint32_t fontIndex ); // font.dat Font Name
    const char * getDownloadFontAppName( uint32_t fontIndex ); // Multi Locale Font Name Support
    const char * getFontWebFaceName( uint32_t fontIndex );
    const char * getFontFullPath( uint32_t fontIndex );

    uint32_t getNumNewFonts();
    const char * getNewFontAppName( uint32_t fontIndex );
    SearchState getCheckNewFont( uint32_t fontIndex );
    bool addNewFont( uint32_t fontIndex );
    bool copyFont( SkFontData * dst, SkFontData * src );
    void setPermission( SkFontData * src );
    void saveNewFont();

    LoadState updateFontManager();
    LoadState checkLoadedFonts();
    LoadState checkPreData();

    SkFontEmbeddedList * getFontEmbeddedList();

    void setFCI( SkFontConfigInterfaceAndroid * FCI );
    SkFontConfigInterfaceAndroid * getFCI();
    void setBootLoaded( bool load );
    bool isBootLoaded();

private:
    bool mBootLoaded;
    SkFontEmbeddedList mEmbeddedFonts;
    SkFontDownloadList mDownloadFonts;
    SkFontSearchList mSearchFonts;

    // PreData.
    preDataHeader mHeader;
    uint32_t mFontIndex;

//                                                                        
#if defined( CAPP_FONTS ) && defined( CAPP_FONTS_HINTING ) && !defined( CAPP_FONTS_JP )
public:
    bool isRobotoTypefaceIndex()
    {
        return ( mRobotoTypefaceIndex == mFontIndex ) ? true : false;
    }

private:
    void findRobotoTypefaceIndex()
    {
        setRobotoTypefaceIndex( SYSTEM_DEFAULT_FAMILY_INDEX );
        uint32_t nFonts = mEmbeddedFonts.numFonts();
        for( uint32_t n = 0 ; n < nFonts ; n++ )
        {
            if( compareRobotoTypeface( getFontWebFaceName( n ) ) == true )
            {
                setRobotoTypefaceIndex( n );
                break;
            }
        }
    }

    bool compareRobotoTypeface( const char * webfaceName )
    {
        return ( !strncmp( "roboto", webfaceName, strlen( webfaceName ) ) ) ? true : false;
    }

    void setRobotoTypefaceIndex( uint32_t fontIndex )
    {
        mRobotoTypefaceIndex = fontIndex;
    }

private:
    uint32_t mRobotoTypefaceIndex;
#endif
// CAPP_FONTS_HINTING_HYFONTS_END

    FamilyRec * mGDefaultFamily;
    SkTypeface * mGDefaultTypefaces[4];
    SkFontConfigInterfaceAndroid * mFCI;
};

static SkFontManager * gFontManager = NULL;

SkFontManager::SkFontManager() :
    mBootLoaded( false ),
    mFontIndex( SYSTEM_DEFAULT_FAMILY_INDEX ),
    mGDefaultFamily( NULL ),
    mFCI( NULL )
{
    memset( &mHeader, 0x00, sizeof( preDataHeader ) );
    memset( mGDefaultTypefaces, 0x00, sizeof( SkTypeface * ) * 4 );
}

SkFontManager::~SkFontManager()
{

}

SearchState
SkFontManager::searchFontApps()
{
    return mSearchFonts.search( &mDownloadFonts );
}

void
SkFontManager::initSystemFontsLocked()
{
    initDownloadFontsLocked();
//                                                                        
#if defined( CAPP_FONTS ) && defined( CAPP_FONTS_HINTING ) && !defined( CAPP_FONTS_JP )
    findRobotoTypefaceIndex();
#endif
// CAPP_FONTS_HINTING_HYFONTS_END
    updateDefaultNormal();
    setBootLoaded( true );

    HyLogi( "We have %d system fonts", getFCI()->fFonts.count() );
    HyLogi( "We have %d system familys", getFCI()->fFontFamilies.count() );
    HyLogi( "Number of families : %d ( BootTypeface(%d) : %s )", getFamilyCount(), mFontIndex, getFontWebFaceName( mFontIndex ) );
}

void
SkFontManager::initDownloadFontsLocked()
{
    uint32_t newFontIndex = 0;
    if( loadPreData( &newFontIndex, true ) >= load_change )
    {
        saveFontIndex( newFontIndex );
        savePreData();
        HyLogif( "Download font information has changed." );
    }
}

uint32_t
SkFontManager::getCurrentFontIndex()
{
    char wfaceName[WEBFACE_NAME_LEN];
    memset( wfaceName, 0x00, WEBFACE_NAME_LEN );
    uint32_t fontIndex = readFontConfigV2( wfaceName );
    if( fontIndex >= getFamilyCount() )
    {
        if( strnlen( wfaceName, WEBFACE_NAME_LEN ) > 0 )
        {
            uint32_t numAllFonts = getFamilyCount();
            for( uint32_t n = 0 ; n < numAllFonts ; n++ )
            {
                if( strncmp( wfaceName, getFontWebFaceName( n ), WEBFACE_NAME_LEN ) == 0 )
                {
                    fontIndex = n;
                    goto END0;
                }
            }
        }
        fontIndex = SYSTEM_DEFAULT_FAMILY_INDEX;
    }
    else
    {
        if( strnlen( wfaceName, WEBFACE_NAME_LEN ) > 0 )
        {
            if( strncmp( wfaceName, getFontWebFaceName( fontIndex ), WEBFACE_NAME_LEN ) != 0 )
            {
                uint32_t numAllFonts = getFamilyCount();
                for( uint32_t n = 0 ; n < numAllFonts ; n++ )
                {
                    if( strncmp( wfaceName, getFontWebFaceName( n ), WEBFACE_NAME_LEN ) == 0 )
                    {
                        fontIndex = n;
                        goto END0;
                    }
                }
                fontIndex = SYSTEM_DEFAULT_FAMILY_INDEX;
                goto END0;
            }
        }
    }
END0:
    return ( mFontIndex = fontIndex );
}

void
SkFontManager::saveFontIndex( uint32_t fontIndex )
{
    if( ( fontIndex == mFontIndex ) || ( fontIndex >= getFamilyCount() ) )
        return;
    saveFontConfigV2( fontIndex, getFontWebFaceName( fontIndex ) );
}

uint32_t
SkFontManager::getFamilyCount()
{
    return ( mEmbeddedFonts.numFonts() + mDownloadFonts.numFonts() );
}

uint32_t
SkFontManager::getEmbeddedFamilyCount()
{
    return mEmbeddedFonts.numFonts();
}

LoadState
SkFontManager::loadPreData( uint32_t * newFontIndex, bool boot )
{
    LoadState state = load_true;
    preDataHeader header = { 0, 0, 0 };
    uint32_t curFontIndex = ( boot == true ? readFontConfig() : getCurrentFontIndex() );
    uint32_t curNewFontIndex = curFontIndex;
    uint32_t numEmbeddedFonts = mEmbeddedFonts.numFonts();
    bool statePass = ( curFontIndex < numEmbeddedFonts ? true : false );
    bool system = ( newFontIndex != NULL ? true : false );
    SkStream * stream = NULL;

    if( openPreDataRead( &stream ) == false )
    {
        if( no_such_file() || permission_denied() )
            goto SKIP_ERROR0;

        HyLogef( "openPreDataRead(%s (%d))", strerror( errno ), errno );
        goto ERROR0;
    }

    stream->read( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

    if( header.tag != __key() )
    {
        HyLogef( "Wrong tag" );
        goto ERROR0;
    }

LOAD_START:
    if( getUpdateVersion() == 0 ) // Boot Load & reload.
    {
        setUpdateVersion( header.updateVersion );

        if( header.count == 0 )
            goto SUCCEED0;

        for( uint32_t n = 0 ; n < header.count ; n++ )
        {
            SkFontData * font = SkNEW( SkFontData );

            if( font )
            {
                font->readPreData( stream );

                if( font->checkCertify( system ) == true )
                {
                    font->setFCI( getFCI() );
                    font->loadFont();
                    font->setFCI( NULL ); // clear
                    mDownloadFonts.addFonts( font );
                }
                else
                {
                    if( ( system == true ) && ( statePass == false ) )
                    {
                        if( state != load_restart )
                        {
                            if( ( numEmbeddedFonts + n ) == curFontIndex )
                            {
                                state = load_restart;
                            }
                            else
                            {
                                state = load_change;
                                if( ( numEmbeddedFonts + n ) < curFontIndex )
                                    curNewFontIndex--;
                            }
                        }
                    }
                    else
                    {
                        state = load_change;
                    }

                    font->deleteFiles();
                    SkDELETE( font );
                    font = NULL;
                }
            }
            else
            {
                HyLogef( "SkNEW( SkFontData )" );
                continue;
            }
        }
    }
    else // Fonts change
    {
        if( getUpdateVersion() == header.updateVersion )
        {
            goto SUCCEED0;
        }
        else
        {
            mDownloadFonts.resetFonts();
            setUpdateVersion( 0 );
            goto LOAD_START;
        }
    }

SUCCEED0:
    if( ( system == true ) && ( statePass == false ) )
    {
        switch( state )
        {
        case load_change: *newFontIndex = curNewFontIndex; break;
        case load_restart: *newFontIndex = SYSTEM_DEFAULT_FAMILY_INDEX; break;
        default: *newFontIndex = curFontIndex; break;
        }
    }
    else if( ( system == true ) && ( statePass == true ) )
    {
        *newFontIndex = curFontIndex;
    }

    if( stream )
        SkDELETE( stream );
    return state;

SKIP_ERROR0:
    return load_false;

ERROR0:
    if( stream )
        SkDELETE( stream );
    return load_false;
}

bool
SkFontManager::openPreDataRead( SkStream ** preData )
{
    SkStream * stream = openReadStream( SYSTEM_DL_PREDATA_FILE );

    if( !stream )
    {
        if( permission_denied() )
            goto SKIP_ERROR0;

        if( !no_such_file() )
            goto ERROR0;

        goto SKIP_ERROR0;
    }

    *preData = stream;
    return true;

SKIP_ERROR0:
    return false;

ERROR0:
    if( stream )
        SkDELETE( stream );
    return false;
}

LoadState
SkFontManager::checkPreData()
{
    LoadState state = load_true;
    SkStream * inStream = openReadStream( SYSTEM_DL_PREDATA_FILE );
    preDataHeader header = { 0, 0, 0 };

    if( !inStream )
    {
        state = load_false;
        goto END0;
    }

    inStream->read( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

    if( header.tag != __key() )
    {
        state = load_false;
        goto END0;
    }

    if( getUpdateVersion() == header.updateVersion )
    {
        state = load_true;
        goto END0;
    }
    else
    {
        state = load_change;
        goto END0;
    }

END0:
    if( inStream )
        SkDELETE( inStream );
    return state;
}

bool
SkFontManager::savePreData()
{
    SkStream * inStream = openReadStream( SYSTEM_DL_PREDATA_FILE );
    SkFILEWStream * outStream = NULL;
    preDataHeader header = { 1, __key(), 0 };
    SkString permission;

    if( !inStream )
    {
        setUpdateVersion( 2 );
        header.updateVersion++;
        header.count = mDownloadFonts.numFonts();
        goto SAVE0;
    }
    else
    {
        inStream->read( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

        if( header.tag != __key() )
        {
            header.updateVersion = 1;
            header.tag = __key();
            header.count = mDownloadFonts.numFonts();
            goto SAVE1;
        }

        if( getUpdateVersion() == 1 && header.updateVersion == 1 )
            setUpdateVersion( 2 );

        header.count = mDownloadFonts.numFonts();
        header.updateVersion++;
        if( header.updateVersion >= 0xFFFFFFFE )
            header.updateVersion = 2;
    }

SAVE1:
    SkDELETE( inStream );
    inStream = NULL;

SAVE0:
    outStream = openWriteStream( SYSTEM_DL_PREDATA_FILE );

    if( !outStream )
    {
        if( permission_denied() )
            goto SKIP_ERROR0;
        HyLogef( "outStream is null" );
        goto ERROR0;
    }

    outStream->write( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

    for( uint32_t n = 0 ; n < header.count ; n++ )
    {
        SkFontData * font = mDownloadFonts.getFont( n );
        font->writePreData( outStream );
    }

    SkDELETE( outStream );
    outStream = NULL;

    //setUpdateVersion( header.updateVersion );

    permission.set( SYSTEM_FONT_PERMISSION_SET );
    permission.append( SYSTEM_DL_PREDATA_FILE );
    system( permission.c_str() );

    HyLogi( "Font Configuration Data (%d), saved.", header.updateVersion );
    return true;

SKIP_ERROR0:
    return false;

ERROR0:
    if( outStream )
        SkDELETE( outStream );
    return false;
}

uint32_t
SkFontManager::getUpdateVersion()
{
    return mHeader.updateVersion;
}

void
SkFontManager::setUpdateVersion( uint32_t updateVersion )
{
    mHeader.updateVersion = updateVersion;
}

void
SkFontManager::updateDefaultNormal()
{
    mGDefaultFamily = getDefaultFamily();

    if( isBootLoaded() )
    {
        mGDefaultTypefaces[SkTypeface::kNormal] = SkFontHost::CreateDefaultTypeface( SkTypeface::kNormal );
        mGDefaultTypefaces[SkTypeface::kBold] = SkFontHost::CreateDefaultTypeface( SkTypeface::kBold );
        mGDefaultTypefaces[SkTypeface::kItalic] = SkFontHost::CreateDefaultTypeface( SkTypeface::kItalic );
        mGDefaultTypefaces[SkTypeface::kBoldItalic] = SkFontHost::CreateDefaultTypeface( SkTypeface::kBoldItalic );
    }
}

SkTypeface *
SkFontManager::getDefaultTypeface( SkTypeface::Style style )
{
    return mGDefaultTypefaces[style];
}

FamilyRec *
SkFontManager::getDefaultFamily()
{
    uint32_t fontIndex = getCurrentFontIndex();
    getFCI()->fDefaultFamilyRecID = getFamilyID( fontIndex ); // Default Family Set.
    return getFamily( fontIndex );
}

const char *
SkFontManager::getDefaultFamilyName()
{
    return getFontWebFaceName( mFontIndex );
}

FamilyRec *
SkFontManager::getFamily( uint32_t fontIndex )
{
    if( fontIndex >= mEmbeddedFonts.numFonts() )
        return mDownloadFonts.getFamily( fontIndex - mEmbeddedFonts.numFonts() );
    return mEmbeddedFonts.getFamily( fontIndex );
}

FamilyRecID
SkFontManager::getFamilyID( uint32_t fontIndex )
{
    if( fontIndex >= mEmbeddedFonts.numFonts() )
        return mDownloadFonts.getFamilyID( fontIndex - mEmbeddedFonts.numFonts() );
    return mEmbeddedFonts.getFamilyID( fontIndex );
}

SkFontData *
SkFontManager::getFont( uint32_t fontIndex )
{
    if( fontIndex >= mEmbeddedFonts.numFonts() )
        return mDownloadFonts.getFont( fontIndex - mEmbeddedFonts.numFonts() );
    return mEmbeddedFonts.getFont( fontIndex );
}

SkFontData *
SkFontManager::getFont()
{
    return getFont( mFontIndex );
}

const char *
SkFontManager::getDownloadFontName( uint32_t fontIndex )
{
    return mDownloadFonts.getFont( fontIndex - mEmbeddedFonts.numFonts() )->getFontName();
}

const char *
SkFontManager::getDownloadFontAppName( uint32_t fontIndex )
{
    return mDownloadFonts.getFont( fontIndex - mEmbeddedFonts.numFonts() )->getFontAppName();
}

const char *
SkFontManager::getFontWebFaceName( uint32_t fontIndex )
{
    return getFont( fontIndex )->getWebFaceName();
}

const char *
SkFontManager::getFontFullPath( uint32_t fontIndex )
{
    return getFont( fontIndex )->getFontFullPath();
}

uint32_t
SkFontManager::getNumNewFonts()
{
    return mSearchFonts.numFonts();
}

const char *
SkFontManager::getNewFontAppName( uint32_t fontIndex )
{
    if( fontIndex >= mSearchFonts.numFonts() )
        return NULL;
    return mSearchFonts.getFont( fontIndex )->getFontAppName();
}

SearchState
SkFontManager::getCheckNewFont( uint32_t fontIndex )
{
    if( fontIndex >= mSearchFonts.numFonts() )
        return STATE_DL_FONTAPP_KEEP;

    if( mSearchFonts.getFont( fontIndex )->checkCertify( true ) == true )
        return STATE_DL_FONTAPP_UPDATE;
    else
        mSearchFonts.getFont( fontIndex )->deleteFiles();

    return STATE_DL_FONTAPP_KEEP;
}

bool
SkFontManager::addNewFont( uint32_t fontIndex )
{
    if( fontIndex >= mSearchFonts.numFonts() )
        goto ERROR0;
    else
    {
        SkFontData * newFont = SkNEW( SkFontData );

        if( newFont )
        {
            if( copyFont( newFont, mSearchFonts.getFont( fontIndex ) ) == false )
                goto ERROR1;

            if( mDownloadFonts.addFonts( newFont ) == false )
                goto ERROR1;

            setPermission( newFont );
            newFont->loadFont();
            return true;
        }

    ERROR1:
        if( newFont )
            SkDELETE( newFont );
        goto ERROR0;
    }

ERROR0:
    return false;
}

bool
SkFontManager::copyFont( SkFontData * dst, SkFontData * src )
{
    dst->setFontFullPath( src->getFontFullPath() );
    dst->setWebFaceName( src->getWebFaceName() );
    dst->setFontAppName( src->getFontAppName() );
    dst->setFontName( src->getFontName() );
    dst->setFontDatPath( src->getFontDatPath() );
    dst->setFontXmlPath( src->getFontXmlPath() );
    return dst->checkCertify( true );
}

void
SkFontManager::setPermission( SkFontData * src )
{
    SkString path;
    SkString file;

    path.set( SYSTEM_FONT_PERMISSION_SET );
    path.append( SYSTEM_DL_FONTAPP_DST_DIR );
    path.append( src->getFontAppName() );

    file.set( SYSTEM_FONT_PERMISSION_SET );
    file.append( SYSTEM_DL_FONTAPP_DST_DIR );
    file.append( src->getFontAppName() );
    file.append( "/*" );

    system( path.c_str() );
    system( file.c_str() );
}

void
SkFontManager::saveNewFont()
{
    savePreData();
}

LoadState
SkFontManager::updateFontManager()
{
    LoadState state = load_true;
    uint32_t newFontIndex = 0;

    if( ( ( state = checkLoadedFonts() ) == load_change  ) || ( ( state = checkPreData() ) == load_change ) )
    {
        mHeader.updateVersion--;
        state = loadPreData( &newFontIndex, false );
    }

    switch( state )
    {
    case load_change:
    case load_restart:
        savePreData();
        saveFontIndex( newFontIndex );
        return state;
    default:
        return load_true;
    }
}

LoadState
SkFontManager::checkLoadedFonts()
{
    uint32_t numFonts = mDownloadFonts.numFonts();

    for( uint32_t n = 0 ; n < numFonts ; n++ )
    {
        SkFontData * font = mDownloadFonts.getFont( n );
        if( font->checkCertify( true ) == false )
            return load_change;
    }
    return load_true;
}

SkFontEmbeddedList *
SkFontManager::getFontEmbeddedList()
{
    return &mEmbeddedFonts;
}

void
SkFontManager::setFCI( SkFontConfigInterfaceAndroid * FCI )
{
    mFCI = FCI;
}

SkFontConfigInterfaceAndroid *
SkFontManager::getFCI()
{
    return ( mFCI ? mFCI : getSingletonInterface() );
}

void
SkFontManager::setBootLoaded( bool load )
{
    mBootLoaded = load;
}

bool
SkFontManager::isBootLoaded()
{
    return mBootLoaded;
}

/* --------------------------------------------------------------------------------------
    FontManager API
-------------------------------------------------------------------------------------- */
static bool
initFontManager()
{
    if( gFontManager )
        return true;

    gFontManager = SkNEW( SkFontManager );

    if( gFontManager )
    {
        HyLogi( "Font Manager creation successful." );
        return true;
    }

    HyLogef( "Font Manager creation failed." );
    return false;
}

static SkFontManager *
getInstance()
{
    return gFontManager;
}

/* --------------------------------------------------------------------------------------
    SkFontConfigInterface_android.cpp
    class SkFontConfigInterfaceAndroid::SkFontConfigInterfaceAndroid()
-------------------------------------------------------------------------------------- */
SkFontConfigInterfaceAndroid::SkFontConfigInterfaceAndroid( SkTDArray<FontFamily*>& fontFamilies, bool useFontManager ) :
    fFonts( fontFamilies.count() ),
    fFontFamilies( fontFamilies.count() / FamilyRec::FONT_STYLE_COUNT ),
    fFamilyNameDict( 1024 ),
    fDefaultFamilyRecID( INVALID_FAMILY_REC_ID ),
    fFallbackFontDict( 128 ),
    fFallbackFontAliasDict( 128 ),
    fLocaleFallbackFontList( NULL )
{
    SkFontEmbeddedList * embeddedFonts = getInstance()->getFontEmbeddedList();

    for ( int i = 0 ; i < fontFamilies.count() ; ++i )
    {
        FontFamily * family = fontFamilies[i];
        FamilyRec * familyRec = NULL;
        FamilyRecID familyRecID = INVALID_FAMILY_REC_ID;

        for( int j = 0 ; j < family->fFontFiles.count() ; ++j )
        {
            SkString filename;
            get_path_for_sys_fonts( &filename, family->fFontFiles[j]->fFileName );

            if( has_font( fFonts, filename ) )
            {
                SkDebugf( "---- system font and fallback font files specify a duplicate font %s, skipping the second occurrence", filename.c_str() );
                continue;
            }

            FontRec& fontRec = fFonts.push_back();
            fontRec.fFileName = filename;
            fontRec.fStyle = SkTypeface::kNormal;
            fontRec.fIsValid = false;
            fontRec.fFamilyRecID = familyRecID;

            const FontRecID fontRecID = fFonts.count() - 1;
            SkAutoTUnref<SkStream> stream( SkStream::NewFromFile( filename.c_str() ) );

            if( stream.get() != NULL )
            {
                bool isFixedWidth;
                SkString name;

                fontRec.fIsValid = find_name_and_attributes( stream.get(), &name, &fontRec.fStyle, &isFixedWidth );
            }
            else
            {
                if( !family->fIsFallbackFont )
                    SkDebugf( "---- failed to open <%s> as a font\n", filename.c_str() );
            }

            if( fontRec.fIsValid )
            {
                DEBUG_FONT( ( "---- SystemFonts[%d][%d] fallback=%d file=%s", i, fFonts.count() - 1, family->fIsFallbackFont, filename.c_str() ) );
            }
            else
            {
                DEBUG_FONT( ( "---- SystemFonts[%d][%d] fallback=%d file=%s (INVALID)", i, fFonts.count() - 1, family->fIsFallbackFont, filename.c_str() ) );
                continue;
            }

            if( familyRec == NULL )
            {
                familyRec = &fFontFamilies.push_back();
                familyRecID = fFontFamilies.count() - 1;
                fontRec.fFamilyRecID = familyRecID;

                familyRec->fIsFallbackFont = family->fIsFallbackFont;
                familyRec->fPaintOptions = family->fFontFiles[j]->fPaintOptions;

                //if( familyRec->fIsFallbackFont )
                //{
                //    addFallbackFamily( familyRecID );
                //}
            }
            else if( familyRec->fPaintOptions != family->fFontFiles[j]->fPaintOptions )
            {
                SkDebugf( "Every font file within a family must have identical language and variant attributes" );
                sk_throw();
            }

            if( INVALID_FONT_REC_ID != familyRec->fFontRecID[fontRec.fStyle] )
            {
                DEBUG_FONT( ( "Overwriting familyRec for style[%d] old,new:(%d,%d)", fontRec.fStyle, familyRec->fFontRecID[fontRec.fStyle], fontRecID ) );
            }

            familyRec->fFontRecID[fontRec.fStyle] = fontRecID;

            //if( familyRec && familyRec->fIsFallbackFont )
            //{
            //    insert_into_name_dict( fFamilyNameDict, fontRec.fFileName.c_str(), familyRecID );
            //}
        }


        if (familyRec) {
            if (familyRec->fIsFallbackFont) {
                // add the font to the appropriate fallback chains and also insert a
                // unique name into the familyNameDict for internal usage
                addFallbackFamily(familyRecID);
            } else {

                const SkTDArray<const char*>& names = family->fNames;

                if( names.isEmpty() )
                {
                    SkDEBUGFAIL( "ERROR: non-fallback font with no name" );
                    continue;
                }

                // set embedded default sans-serif family information.
                if( isDefaultSansSerifFamily( names ) == true )
                {
                    SkFontData * font = SkNEW( SkFontData );
                    FontRecID fontRecID = fFontFamilies[familyRecID].fFontRecID[SkTypeface::kNormal];
                    FontRec * fontRec = &fFonts[fontRecID];

                    if( font )
                    {
                        font->setFamily( familyRec );
                        font->setFamilyID( familyRecID );
                        font->setFontFullPath( fontRec->fFileName.c_str() );
                        font->setWebFaceName( names[0] );
                        embeddedFonts->addFonts( font );
                        HyLogif( "ADD Default SansSerif : familyRecID(%d), filename(%s)", familyRecID, fontRec->fFileName.c_str() );
                    }
                    else
                    {
                        HyLogef( "SkNEW( SkFontData ) (%s)", fontRec->fFileName.c_str() );
                        continue;
                    }

                    for( int i = 0 ; i < names.count() ; i++ )
                    {
                        insert_into_name_dict( fFamilyNameDict, names[i], familyRecID );

                        if( familyRecID > 0 )
                            break;
                    }
                }
                else
                {
                    for( int i = 0 ; i < names.count() ; i++ )
                    {
                        insert_into_name_dict( fFamilyNameDict, names[i], familyRecID );
                    }
                }
            }
        }
    }

    DEBUG_FONT( ( "---- We have %d system fonts", fFonts.count() ) );

    if( fFontFamilies.count() > 0 )
    {
        fDefaultFamilyRecID = 0;
    }

    FallbackFontList * fallbackList;
    SkTDict<FallbackFontList*>::Iter iter(fFallbackFontDict);
    const char* fallbackLang = iter.next(&fallbackList);

    while( fallbackLang != NULL )
    {
        for( int i = 0 ; i < fDefaultFallbackList.count() ; i++ )
        {
            FamilyRecID familyRecID = fDefaultFallbackList[i];
            const SkString& fontLang = fFontFamilies[familyRecID].fPaintOptions.getLanguage().getTag();

            if( strcmp( fallbackLang, fontLang.c_str() ) != 0 )
            {
                fallbackList->push( familyRecID );
            }
        }

        fallbackLang = iter.next( &fallbackList );
    }
}

/* --------------------------------------------------------------------------------------
    SkFontConfigInterface_android.cpp, getSingletonInterface()
-------------------------------------------------------------------------------------- */
static SkFontConfigInterfaceAndroid *
getSingletonInterface()
{
    SK_DECLARE_STATIC_MUTEX( gMutex );
    static SkFontConfigInterfaceAndroid * gFontConfigInterface;
    SkAutoMutexAcquire ac( gMutex );

    if( gFontConfigInterface )
        return gFontConfigInterface;
    else if( initFontManager() )
    {
        SkTDArray<FontFamily*> fontFamilies;
        SkFontConfigParser::GetFontFamilies( fontFamilies );
        gFontConfigInterface = new SkFontConfigInterfaceAndroid( fontFamilies, true );

        getInstance()->setFCI( gFontConfigInterface );
        getInstance()->initSystemFontsLocked();
        getInstance()->setFCI( NULL ); // clear

        // cleanup the data we received from the parser
        fontFamilies.deleteAll();
        return gFontConfigInterface;
    }
    return NULL;
}

SkFontConfigInterface* SkFontConfigInterface::GetSingletonDirectInterface() {
    return getSingletonInterface();
}

/* --------------------------------------------------------------------------------------
    class SkFontData::loadFont()
-------------------------------------------------------------------------------------- */
static FontRecID
findFontRec( const SkTArray<FontRec>& array, const SkString& filename )
{
    for( int i = 0 ; i < array.count() ; i++ )
        if( array[i].fFileName == filename )
            return i;
    return INVALID_FONT_REC_ID;
}

void
SkFontData::setFCI( void * FCI )
{
    mFCI = FCI;
}

void *
SkFontData::getFCI()
{
    return ( mFCI ? mFCI : getSingletonInterface() );
}

void
SkFontData::loadFont()
{
    SkTDArray<FontFamily*> fontFamilies;
    getFamilies( fontFamilies );
    SkFontConfigInterfaceAndroid * FCI = ( SkFontConfigInterfaceAndroid * )getFCI();

    if( fontFamilies.count() == 0 || fontFamilies.count() > 1 )
    {
        HyLogef( "invalid fontFamilies.count(%d)", fontFamilies.count() );
        goto ERROR0;
    }
    else
    {
        FontFamily * family = fontFamilies[0];
        FamilyRec * familyRec = NULL;
        FamilyRecID familyRecID = INVALID_FAMILY_REC_ID;
        SkTDArray<const char*> names = family->fNames;
        SkString filename;
        FontRec * fontRec = NULL;
        FontRecID fontRecID = INVALID_FONT_REC_ID;

        if( family->fIsFallbackFont )
        {
            HyLogef( "%s is fallback font", mFontAppName );
            goto ERROR0;
        }

        if( !isDefaultSansSerifFamily( names ) )
        {
            HyLogef( "%s is not Default SansSerif family", mFontAppName );
            goto ERROR0;
        }

        if( family->fFontFiles.count() == 0 || family->fFontFiles.count() > 1 )
        {
            HyLogef( "%s, invalid font count (%d)", mFontAppName, family->fFontFiles.count() );
            goto ERROR0;
        }

        getFullPathDownloadFont( &filename, family->fFontFiles[0]->fFileName );

        // find font record.
        if( ( fontRecID = findFontRec( FCI->fFonts, filename ) ) != INVALID_FONT_REC_ID )
        {
            fontRec = &FCI->fFonts[fontRecID];

            if( fontRec->fFamilyRecID != INVALID_FAMILY_REC_ID )
            {
                familyRecID = fontRec->fFamilyRecID;
                familyRec = &FCI->fFontFamilies[familyRecID];

                setFamily( familyRec);
                setFamilyID( familyRecID );
                setFontFullPath( fontRec->fFileName.c_str() );
                setWebFaceName( names[0] );

                HyLogif( "%s, aleady installed", names[0] );
                goto LOAD_OK;
            }

            fontRec = NULL;
        }

        // Register new family.
        fontRec = &FCI->fFonts.push_back();
        fontRec->fFileName = filename;
        fontRec->fStyle = SkTypeface::kNormal;
        fontRec->fIsValid = false;
        fontRec->fFamilyRecID = familyRecID;

        fontRecID = FCI->fFonts.count() - 1;

        SkAutoTUnref<SkStream> stream( SkStream::NewFromFile( filename.c_str() ) );

        if( stream.get() != NULL )
        {
            bool isFixedWidth;
            SkString name;
            fontRec->fIsValid = find_name_and_attributes( stream.get(), &name, &fontRec->fStyle, &isFixedWidth );

            if( !fontRec->fIsValid )
            {
                HyLogef( "%s, find_name_and_attributes : false", mFontAppName );
                goto ERROR0;
            }
        }
        else
        {
            HyLogef( "failed to open <%s> as a font, %s (%d) \n", filename.c_str(), strerror( errno ), errno );
            goto ERROR0;
        }

        familyRec = &FCI->fFontFamilies.push_back();
        familyRecID = FCI->fFontFamilies.count() - 1;
        fontRec->fFamilyRecID = familyRecID;
        familyRec->fIsFallbackFont = family->fIsFallbackFont;
        familyRec->fPaintOptions = family->fFontFiles[0]->fPaintOptions;
        familyRec->fFontRecID[fontRec->fStyle] = fontRecID;

        // add the names that map to this family to the dictionary for easy lookup
        insert_into_name_dict( FCI->fFamilyNameDict, names[0], familyRecID );

        setFamily( familyRec );
        setFamilyID( familyRecID );
        setFontFullPath( fontRec->fFileName.c_str() );
        setWebFaceName( names[0] );
    }

LOAD_OK:
    HyLogi( "%s load ok", mFontAppName );
    return;

ERROR0:
    HyLogef( "%s load failed", mFontAppName );
    return;
}

/* --------------------------------------------------------------------------------------
    SkFontHost_fontconfig.cpp
-------------------------------------------------------------------------------------- */
struct FindRec
{
    FindRec( const char * name, SkTypeface::Style style ) :
        fFamilyName( name ),  // don't need to make a deep copy
        fStyle( style ) {}

    const char* fFamilyName;
    SkTypeface::Style fStyle;
};

static bool
_find_proc( SkTypeface * face, SkTypeface::Style style, void * ctx )
{
    FontConfigTypeface * fci = ( FontConfigTypeface * )face;
    const FindRec * rec = ( const FindRec * )ctx;
    return rec->fStyle == style && fci->isFamilyName( rec->fFamilyName );
}

SkTypeface *
FontConfigTypeface::LegacyCreateTypeface(    const SkTypeface * familyFace,
                                                                                const char familyName[],
                                                                                SkTypeface::Style style)
{
    HyLogif( "LegacyCreateTypeface( 0x%x, %s, %d ) START", familyFace, familyName, style );

    SkAutoTUnref<SkFontConfigInterface> fci( getRefFCI() );

    if( NULL == fci.get() )
        return NULL;

    if( familyFace )
    {
        FontConfigTypeface * fct = ( FontConfigTypeface * )familyFace;
        familyName = fct->getFamilyName();
    }

    bool loadDefaultFamilyName = false;
    SkString defaultFamilyName;

#ifdef CAPP_FONT_NO_COMPARING_FAMILY_NAME
    if( !familyName )
#else
    if( !familyName || compareSystemFamilyName( familyName ) )
#endif
    {
        defaultFamilyName.set( getInstance()->getDefaultFamilyName() );
        loadDefaultFamilyName = true;
    }

    const char * findFamilyName = ( const char * )( loadDefaultFamilyName ? defaultFamilyName.c_str() : familyName );
    FindRec rec( findFamilyName, style ); // FindRec rec( familyName, style );
    SkTypeface * face = SkTypefaceCache::FindByProcAndRef( _find_proc, &rec );

    if( face )
    {
        // HyLogif( "TypefaceCache : find face : uniqueID(%d), style(%d), familyName(%s)", face->uniqueID(), face->style(), findFamilyName );
        // SkDebugf( "found cached face <%s> <%s> %p [%d]\n", familyName, ( ( FontConfigTypeface * )face )->getFamilyName(), face, face->getRefCnt() );
        return face;
    }

    SkFontConfigInterface::FontIdentity indentity;
    SkString outFamilyName;
    SkTypeface::Style outStyle;

    if( !fci->matchFamilyName( findFamilyName, style, &indentity, &outFamilyName, &outStyle ) )
    {
        // HyLogef( "LegacyCreateTypeface(%s) failed", findFamilyName );
        return NULL;
    }

    // check if we, in fact, already have this. perhaps fontconfig aliased the
    // requested name to some other name we actually have...
    rec.fFamilyName = outFamilyName.c_str();
    rec.fStyle = outStyle;
    face = SkTypefaceCache::FindByProcAndRef(_find_proc, &rec);
    if (face) {
        return face;
    }

    // HyLogif( "Create New Typeface : %s", outFamilyName.c_str() );

    face = SkNEW_ARGS( FontConfigTypeface, ( outStyle, indentity, outFamilyName ) );
    SkTypefaceCache::Add( face, style );

    // SkDebugf( "add face <%s> <%s> %p [%d]\n", findFamilyName, outFamilyName.c_str(), face, face->getRefCnt() );
    return face;
}

/* --------------------------------------------------------------------------------------
    include SkFontHost_android_hy_fonthost.cpp
-------------------------------------------------------------------------------------- */
#include "SkFontHost_android_hy_fonthost.cpp"
#endif
// CAPP_FONTS_HYFONTS_END
// EOF

#endif /* ON_CAPP_FONTS */